/*
 * sys_chdir.c - kernel function implementing the "chdir" syscall.
 *
 * Author    Alain Greiner (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <chdev.h>
#include <rwlock.h>
#include <vfs.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>

#include <syscalls.h>

/////////////////////////////////
int sys_chdir ( char * pathname )
{
    error_t   error;
    vseg_t  * vseg; 
    xptr_t    root_inode_xp;

    char      kbuf[CONFIG_VFS_MAX_PATH_LENGTH];

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

#if (DEBUG_SYS_CHDIR || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_start = hal_get_cycles();
#endif

    // check pathname length
    if( hal_strlen_from_uspace( pathname ) >= CONFIG_VFS_MAX_PATH_LENGTH )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : pathname too long / thread[%x,%x]\n",
__FUNCTION__, process->pid, this->trdid );
#endif
        this->errno = EINVAL;
        return -1;
    }

    // check pathname in user space
    if( vmm_get_vseg( process, (intptr_t)pathname , &vseg ) )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : user buffer unmapped %x for thread[%x,%x]\n",
__FUNCTION__ , (intptr_t)pathname , process->pid, this->trdid );
#endif
		this->errno = EINVAL;
        return -1;
	}

    // copy pathname in kernel space
    hal_strcpy_from_uspace( kbuf , pathname , CONFIG_VFS_MAX_PATH_LENGTH );

#if DEBUG_SYS_CHDIR
if( DEBUG_SYS_CHDIR < tm_start )
printk("\n[%s] thread[%x,%x] enter for <%s> / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, kbuf, (uint32_t)tm_start );
#endif

    // compute root inode for path 
    if( kbuf[0] == '/' )                        // absolute path
    {
        // use extended pointer on VFS root inode
        root_inode_xp = process->vfs_root_xp;
    }
    else                                        // relative path
    {
        // get cluster and local pointer on reference process
        xptr_t      ref_xp  = process->ref_xp;
        process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );
        cxy_t       ref_cxy = GET_CXY( ref_xp );

        // use extended pointer on CWD inode
        root_inode_xp = hal_remote_l64( XPTR( ref_cxy , &ref_ptr->cwd_xp ) );
    }

    // call the relevant VFS function
    error = vfs_chdir( root_inode_xp , kbuf );

    if( error )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s / thread[%x,%x] : cannot change CWD\n",
__FUNCTION__ , process->pid , this->trdid );
#endif
        this->errno = error;
        return -1;
    }

    hal_fence();

#if (DEBUG_SYS_CHDIR || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_CHDIR
if( DEBUG_SYS_CHDIR < tm_end )
printk("\n[%s] thread[%x,%x] exit  / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, (uint32_t)tm_end );
#endif
 
#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_CHDIR] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_CHDIR] , 1 );
#endif

    return 0;
}
