/*
 * hal_vmm.c - Virtual Memory Manager Initialisation for TSAR
 *
 * Authors  Alain Greiner (2016,2017,2018,2019)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_vmm.h>
#include <hal_gpt.h>
#include <process.h>
#include <vseg.h>
#include <xlist.h>
#include <vmm.h>
#include <remote_rwlock.h>

//////////////////////////////////////////////////////////////////////////////////////////
// This file contains the TSAR specific code used to initialize the kernel process VMM,
// or to update an user process VMM with informations related to the kernel vsegs.
// As the TSAR architure does not use the DATA MMU, but use only the DATA extension
// address register to access local and remote kernel data, the kernel VSL contains only
// one "kcode" segment, and the kernel GPT contains only one big page in PT1[0] slot.
//////////////////////////////////////////////////////////////////////////////////////////

// extern global variables
extern process_t process_zero;

//////////////////////////////////////////////////////////////////////////////////////////
// This function is called by the process_zero_init() function during kernel_init.
// It initializes the VMM of the kernel proces_zero (containing all kernel threads)
// in the local cluster.
//////////////////////////////////////////////////////////////////////////////////////////
error_t  hal_vmm_kernel_init( boot_info_t * info )
{
    error_t   error;

    // get pointer on kernel GPT
    gpt_t * gpt = &process_zero.vmm.gpt;

    // get cluster identifier
    cxy_t cxy = local_cxy;

    // allocate memory for kernel GPT
    error = hal_gpt_create( gpt );

    if( error )
    {
        printk("\n[PANIC] in %s : cannot allocate kernel GPT in cluster %x\n",
        __FUNCTION__ , cxy );
        hal_core_sleep();
    }

    // compute attr and ppn for one PTE1
    uint32_t attr  = 0x8A800000;           // bits : V,C,X,G
    uint32_t ppn   = (cxy << 20) >> 9;     // physical page index is 0

    // set PTE1  in slot[0]
    error = hal_gpt_set_pte( XPTR( cxy , gpt ) , 0 , attr , ppn );

    if( error )
    {
        printk("\n[PANIC] in %s : cannot initialize kernel GPT in cluster %x\n",
        __FUNCTION__ , cxy );
        hal_core_sleep();
    }

    // create kcode vseg and register it in kernel VSL
    vseg_t * vseg = vmm_create_vseg( &process_zero,
                                     VSEG_TYPE_CODE,
                                     info->kcode_base,
                                     info->kcode_size,
                                     0, 0,                  // file ofset and file size (unused)
                                     XPTR_NULL,             // no mapper
                                     local_cxy );
    if( vseg == NULL )
    {
        printk("\n[PANIC] in %s : cannot register vseg to VSL in cluster %x\n",
        __FUNCTION__ , cxy );
        hal_core_sleep();
    }

}  // end hal_vmm_init()

//////////////////////////////////////////////////////////////////////////////////////////
// This function is called by the vmm_init() function to update the VMM of an user
// process identified by the <process> argument.
// It registers in the user VSL the "kcode" vseg, registered in the local kernel VSL,
// and register in the user GPT the big page[0] mapped in the local kernel GPT. 
//////////////////////////////////////////////////////////////////////////////////////////
error_t hal_vmm_kernel_update( process_t * process )
{
    error_t error;
    uint32_t attr;
    uint32_t ppn;

// TODO check ppn value in kernel GPT (must be 0)

    // get cluster identifier
    cxy_t cxy = local_cxy;

    // get extended pointer on user GPT
    xptr_t gpt_xp = XPTR( cxy , &process->vmm.gpt );

    // get ppn and attributes from slot[0] in kernel GPT
    hal_gpt_get_pte( gpt_xp , 0 , &attr , &ppn );

// check ppn and attributes 
assert( (attr == 0x8A800000) && (ppn == ((cxy << 20) >> 9)),  __FUNCTION__,
"bad ppn = %x or attr = %x in slot[0] of kernel GPT\n", ppn , attr );
 
    // update user GPT : set PTE1 in slot[0]
    error = hal_gpt_set_pte( gpt_xp , 0 , attr , ppn );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot update GPT in cluster %x\n",
        __FUNCTION__ , cxy );
        return -1;
    }

    // get pointer on the unique vseg registered in kernel VSL
    xptr_t root_xp = XPTR( cxy , &process_zero.vmm.vsegs_root );
    vseg_t * vseg = XLIST_FIRST( root_xp , vseg_t , xlist );

// check vsegs_nr 
assert( (process_zero.vmm.vsegs_nr == 1 ) , __FUNCTION__,
"bad vsegs number in kernel VSL\n" );

    // update user VSL : register one new vseg for kcode
    vseg_t * new = vmm_create_vseg( process,
                                    vseg->type,
                                    vseg->min,
                                    vseg->max - vseg->min,
                                    0, 0,                  // file ofset and file size (unused)
                                    XPTR_NULL,             // no mapper
                                    local_cxy );
    if( new == NULL )
    {
        printk("\n[ERROR] in %s : cannot update VSL in cluster %x\n",
        __FUNCTION__ , cxy );
        return -1;
    }
}


