/*
 * page.c - physical page related operations implementation
 *
 * Authors  Ghassan Almaless (2008,2009,2010,2011,2012)
 *          Alain Greiner    (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_special.h>
#include <hal_atomic.h>
#include <list.h>
#include <queuelock.h>
#include <memcpy.h>
#include <printk.h>
#include <vfs.h>
#include <process.h>
#include <page.h>


////////////////////////////////////////
inline void page_init( page_t   * page )
{
	page->flags    = 0;
	page->order    = 0;
	page->mapper   = NULL;
	page->index    = 0;
	page->refcount = 0;
	page->forks    = 0;

	remote_busylock_init( XPTR( local_cxy , &page->lock ), LOCK_PAGE_STATE );

	list_entry_init( &page->list );
}

////////////////////////////////////////////
inline void page_set_flag( page_t   * page,
                           uint32_t   value )
{
	hal_atomic_or( (uint32_t *)&page->flags , (uint32_t)value );
}

//////////////////////////////////////////////
inline void page_clear_flag( page_t   * page,
                             uint32_t   value )
{
	hal_atomic_and( (uint32_t *)&page->flags , ~((uint32_t)value) );
}

//////////////////////////////////////////////
inline bool_t page_is_flag( page_t   * page,
                            uint32_t   value )
{
    return ( (page->flags & value) ? 1 : 0 );
}

////////////////////////////////////////////
inline void page_refcount_up( page_t *page )
{
    hal_atomic_add( &page->refcount , +1 );
}

//////////////////////////////////////////////
inline void page_refcount_down( page_t *page )
{
    hal_atomic_add( &page->refcount , -1 );
}

///////////////////////////////
void page_zero( page_t * page )
{
	uint32_t   size = (1 << page->order) * CONFIG_PPM_PAGE_SIZE;

	xptr_t base_xp = ppm_page2base( XPTR( local_cxy , page ) );

	memset( GET_PTR( base_xp ) , 0 , size );
}


////////////////////////////////////////////////////
inline void page_remote_set_flag( xptr_t    page_xp,
                                  uint32_t  value )
{
    cxy_t    page_cxy = GET_CXY( page_xp );
    page_t * page_ptr = GET_PTR( page_xp );

	hal_remote_atomic_or( XPTR( page_cxy , &page_ptr->flags ) , value );
}

//////////////////////////////////////////////////////
inline void page_remote_clear_flag( xptr_t    page_xp,
                                    uint32_t  value )
{
    cxy_t    page_cxy = GET_CXY( page_xp );
    page_t * page_ptr = GET_PTR( page_xp );

	hal_remote_atomic_and( XPTR( page_cxy , &page_ptr->flags ) , value );
}

/////////////////////////////////////////////////////
inline bool_t page_remote_is_flag( xptr_t    page_xp,
                                   uint32_t  value )
{
    cxy_t    page_cxy = GET_CXY( page_xp );
    page_t * page_ptr = GET_PTR( page_xp );

    uint32_t flags = hal_remote_l32( XPTR( page_cxy , &page_ptr->flags ) );
    
    return (flags & value) ? 1 : 0;
}

/////////////////////////////////////////////////////
inline void page_remote_refcount_up( xptr_t page_xp )
{
    cxy_t    page_cxy = GET_CXY( page_xp );
    page_t * page_ptr = GET_PTR( page_xp );

    hal_remote_atomic_add( XPTR( page_cxy , &page_ptr->refcount ) , 1 );
}

///////////////////////////////////////////////////////
inline void page_remote_refcount_down( xptr_t page_xp )
{
    cxy_t    page_cxy = GET_CXY( page_xp );
    page_t * page_ptr = GET_PTR( page_xp );

    hal_remote_atomic_add( XPTR( page_cxy , &page_ptr->refcount ) , -1 );
}

///////////////////////////////////////////
void page_remote_display( xptr_t  page_xp )
{
    page_t  page;   // local copy of page decriptor

    hal_remote_memcpy( XPTR( local_cxy , &page ) , page_xp , sizeof( page_t ) );
                      
	printk("*** Page %d in cluster %x : ppn %x / flags %x / order %d / refcount %d\n",
	        page.index,
            GET_CXY( page_xp ),
	        ppm_page2ppn( page_xp ),
	        page.flags,
	        page.order,
	        page.refcount );
}



