/*
 * sys_get_core.c - get calling core cluster and local index.
 * 
 * Author    Alain Greiner (2016,2017,2018,2019)
 *  
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_special.h>
#include <errno.h>
#include <core.h>
#include <thread.h>
#include <process.h>
#include <vmm.h>
#include <printk.h>

#include <syscalls.h>

///////////////////////////////////
int sys_place_fork( uint32_t  cxy )
{
    thread_t  * this    = CURRENT_THREAD;

#if DEBUG_SYS_PLACE_FORK || DEBUG_SYSCALLS_ERROR || CONFIG_INSTRUMENTATION_SYSCALLS
uint64_t    tm_start = hal_get_cycles();
process_t * process  = this->process;
#endif

#if DEBUG_SYS_PLACE_FORK
if( DEBUG_SYS_PLACE_FORK < tm_start )
printk("\n[%s] thread[%x,%x] enter / cxy %x / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, cxy, (uint32_t)tm_start );
#endif

    // check cxy argument
    if( cluster_is_active( cxy ) == false )
    {
        
#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start );
printk("\n[ERROR] in %s : thread[%x,‰x] / illegal cxy argument %x\n",
__FUNCTION__ , process->pid , this->trdid , cxy );
#endif
        this->errno = EFAULT;
		return -1;
	}

    // set relevant arguments in calling thread descriptor
    this->fork_user = true;
    this->fork_cxy  = cxy;

#if (DEBUG_SYS_PLACE_FORK || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_PLACE_FORK
if( DEBUG_SYS_PLACE_FORK < tm_end )
printk("\n[%s] thread[%x,%x] exit / cycle %d\n",
__FUNCTION__ , process->pid, this->trdid, (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_PLACE_FORK] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_PLACE_FORK] , 1 );
#endif

	return 0; 

}  // end sys_place_fork()
