///////////////////////////////////////////////////////////////////////////////////////
//  file   : udp_chat.c
//  author : Alain Greiner
//  date   : march 2020
///////////////////////////////////////////////////////////////////////////////////////
//  This file describes an UDP based chat application.
//  It can be used to launch the UDP client, or the UDP server application.
//  The client send the first message. The server wait this first message. 
//  The 4 command line arguments are: is_server, local_ip_addr, remote_ip_addr, port
///////////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <almosmkh.h>
#include <unistd.h>
#include <sys/socket.h>

#define   BUF_SIZE       256

////////////////////////////
void client_chat( int fdid )
{
    char   buffer[BUF_SIZE];   // string buffer (for both send and receive)
    int    size;               // string length (including NUL)
    int    nbytes;             // number of characters actually sent or received

    while( 1 )
    {
        //////// display client prompt to wait local client message ////////
        printf("\n[local client] ");

        // build local client message
        size = get_string( buffer , BUF_SIZE );

        // exit chat function when client message is the "exit" string
        if( strncmp( "exit" , buffer , 4 ) == 0 )
        {
            printf("\n[client_chat] local message is <exit> => return to main\n" );
            return;
        }
 
       // send local client message
        nbytes = send( fdid , buffer, size, 0 );      // flags

        if( nbytes != size )
        {
            printf("\n[client_chat error] cannot send => return to main\n");
            return;
        }

        ///////// display server prompt to wait remote server message ///////
        printf("\n[remote server] ");

        // receive remote server message
        nbytes = recv( fdid , buffer , BUF_SIZE , 0 ); 

        if( nbytes < 0 )
        {
            printf("\n\n[client_chat error] cannot receive => return to main\n" );
            return;
        }
        else if( nbytes == 0 )
        {
            printf("\n\n[client_chat] receive EOF => return to main\n" );
            return;
        }
        
        // display remote server message
        printf("%s\n", buffer );
    } 
}  // end client_chat()


////////////////////////////
void server_chat( int fdid )
{
    char   buffer[BUF_SIZE];   // string buffer (for send and receive)
    int    size;               // string length (including NUL)
    int    nbytes;             // number of characters actually sent or received

    while( 1 )
    {
        //////// display client prompt to wait remote client message ///////
        printf("\n[remote client] ");

        // receive remote client message
        nbytes = recv( fdid , buffer , BUF_SIZE , 0 ); 

        if( nbytes < 0 )
        {
            printf("\n\n[server_chat error] cannot receive => return to main\n" );
            return;
        }
        else if( nbytes == 0 )
        {
            printf("\n\n[server_chat] receive EOF => return to main\n" );
            return;
        }
        
        // display remote client message
        printf("%s\n", buffer );

        //////// display server prompt to wait local server message //////
        printf("\n[local server] ");

        // build local server message
        size = get_string( buffer , BUF_SIZE );

        // exit chat function when server message is the "exit" string
        if( strncmp( "exit" , buffer , 4 ) == 0 )
        {
            printf("\n[server_chat] local message is <exit> => return to main\n" );
            return;
        }

        // send local server message
        nbytes = send( fdid , buffer , size , 0 );

        if( nbytes != size )
        {
            printf("\n[server_chat error] cannot send => return to main\n");
            return;
        }
    } 
}  // end server_chat()


///////////////////////
int main( int     argc,
          char ** argv )
{
    int                  pid;          // process identifier
    int                  fdid;         // file index of local socket
    int                  error;

    sockaddr_in_t        local_sin;   // local socket internet address
    sockaddr_in_t        remote_sin;  // remote socket internet address

    unsigned long long   start_cycle;

    int                  addr_length = sizeof(sockaddr_t);

    // get  start cycle
    get_cycle( &start_cycle );

    // get PID
    pid = getpid();

    // get arguments
    if( argc != 4 )
    {
        printf("\n  usage : udp_chat is_server local_addr remote_addr port\n");
        exit( 0 );
    }

    int  is_server    = atoi( argv[0] );
    int  local_addr   = atoi( argv[1] );
    int  remote_addr  = atoi( argv[2] );
    int  port         = atoi( argv[3] );

    if( is_server )
    printf("\n[udp_chat] SERVER / pid %x / cycle %d"
    "  local_addr %x  / remote_addr %x / port %x\n",
    pid, (unsigned int)start_cycle, local_addr,  remote_addr, port );
    else           
    printf("\n[udp_chat] CLIENT / pid %x / cycle %d"
    "  local_addr %x  / remote_addr %x / port %x\n",
    pid, (unsigned int)start_cycle, local_addr,  remote_addr, port );

    // initialize local_sin
    local_sin.sin_domain = HTONS( AF_INET );
    local_sin.sin_addr   = HTONL( local_addr );
    local_sin.sin_port   = HTONS( port );

    // initialize remote_sin
    remote_sin.sin_domain = HTONS( AF_INET );
    remote_sin.sin_addr   = HTONL( remote_addr );
    remote_sin.sin_port   = HTONS( port );

    // 1. create local UDP socket
    fdid = socket( AF_INET,
                   SOCK_DGRAM,
                   0 );

    if( fdid < 0 )
    {
        printf("\n[udp_chat error] cannot create socket\n");
        exit( 0 );
    }
    else
    {
        printf("\n[udp_chat] created socket[%x,%d]\n", pid, fdid );
    }

    // 2. bind local socket
    error = bind( fdid,
                  (sockaddr_t *)(&local_sin),
                  sizeof(sockaddr_t) );
    if( error )
    {
        printf("\n[udp_chat error] bind failure on socketi[%x,%d]\n", pid, fdid );
        exit( 0 );
    }
    else
    {
        printf("\n[udp_chat] socket[%x,%d] bound : [%x,%x]\n",
        pid, fdid, local_sin.sin_addr, (unsigned int)local_sin.sin_port );
    }


    // 3. connect local socket to remote socket
    error = connect( fdid,
                     (sockaddr_t *)(&remote_sin),
                     addr_length );
    if( error )
    {
        printf("\n[server_chat error] cannot connect to remote => return to main\n" );
        exit( 0 );
    }
    else
    {
        printf("\n[server_chat] successfully connected to remote client\n");
    }

    // 4. call chat function
    if( is_server ) server_chat( fdid );
    else            client_chat( fdid );

    // 5. close local socket
    close( fdid );

    printf("\n[udp_chat] closed socket[%x,%d]\n", pid, fdid );

    exit(0);

    return 0;
}
