/*
 * printk.h - Kernel Log & debug messages API definition.
 *
 * authors  Alain Greiner (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

///////////////////////////////////////////////////////////////////////////////////
// The printk.c and printk.h files define the functions used by the kernel
// to build, or display on terminal TXT0, formated strings.
// In case ofdisplay, it calls synchronously the TXT0 driver, without descheduling.
//
// The supported formats are defined below :
//   %c : single ascii character (8 bits)
//   %b : exactly 2 hexadecimal digits (8 bits)
//   %d : up to 10 digits decimal integer (32 bits)
//   %u : up to 10 digits unsigned decimal (32 bits) 
//   %x : up to 8 digits hexadecimal integer (32 bits) 
//   %X : exactly 8 digits hexadecimal integer (32 bits) 
//   %l : up to 16 digits hexadecimal integer (64 bits)
//   %L : exactly 16 digits hexadecimal integer (64 bits)
//   %s : NUL terminated character string
///////////////////////////////////////////////////////////////////////////////////

#ifndef _PRINTK_H
#define _PRINTK_H

#include <hal_kernel_types.h>
#include <stdarg.h>

#include <hal_special.h> 

/**********************************************************************************
 * These functions display a formated string defined by the <format,...>
 * argument on the kernel terminal TXT0, with or without taking the TXT0 lock. 
 **********************************************************************************
 * Implementation note:
 * It uses a buffer allocated in the stack, defined by CONFIG_PRINTK_BUFFER_SIZE.
 * It calls the snprintk() function to build a printable string in this buffer, 
 * and calls directly the dev_txt_sync_write() driver function without using the
 * TXT server thread.
 * It displays a [PANIC] message on kernel TXT0 terminal if the formated string
 * exceeds the buffer size, or if the format is undefined. 
 **********************************************************************************
 * @ format     : formatted string.
 *********************************************************************************/
void printk       ( char * format, ... );
void nolock_printk( char * format, ... );

/**********************************************************************************
 * This function displays an [ASSERT] message on kernel TXT0 terminal
 * if Boolean expression <expr> is false. It prints a detailed message including:
 * - the calling core [cxy,lpid]
 * - the calling thread[pid,trdid]
 * - the current cycle
 * - the <func_name> argument
 * - the <string> argument
 **********************************************************************************
 * @ func_name  : calling function name.
 * @ expr       : Boolean expression to be checked.
 * @ format     : formated message argument.
 *********************************************************************************/
void assert( const char   * func_name,
             bool_t         expr,
             char         * format , ... );

/**********************************************************************************
 * This function build a formated string in a buffer defined by the <buffer> 
 * and <buf_size> arguments, from the format defined by the <format,...> argument.
 * This function set the NUL terminating character in target <buffer>,
 * but the returned length does not include this NUL character.
 **********************************************************************************
 * @ buffer     : pointer on target buffer (allocated by caller).
 * @ buf_size   : target buffer length (number of bytes).
 * @ format     : format respecting the printf syntax.
 * @ returns  string length (not including NUL) if success / -1 if error.
 *********************************************************************************/
int32_t snprintk( char       * buffer,
                  uint32_t     buf_size,
                  char       * format, ... );

/**********************************************************************************
 * These functions displays a non-formated string on TXT0 terminal.
 * They are actually used for low level debug, and call directly the TXT driver,
 * without using the TXT server thread.
 **********************************************************************************
 * @ string   : non-formatted, NUL terminated string.
 *********************************************************************************/
void puts( const char * string );
void nolock_puts( const char * string );

/**********************************************************************************
 * These functions display a 32 bits value in hexadecimal on TXT0 terminal.
 * They are actually used for low level debug, and call directly the TXT driver,
 * without using the TXT server thread.
 **********************************************************************************
 * @ val   : 32 bits unsigned value.
 *********************************************************************************/
void putx( uint32_t val );
void nolock_putx( uint32_t val );

/**********************************************************************************
 * These functions display a 32 bits signed value in decimal on TXT0 terminal.
 * They are actually used for low level debug, and call directly the TXT driver,
 * without using the TXT server thread.
 **********************************************************************************
 * @ val   : 32 bits signed value.
 *********************************************************************************/
void putd( int32_t val );
void nolock_putd( int32_t val );

/**********************************************************************************
 * These functions display a 64 bits value in hexadecimal on TXT0 terminal.
 * They are actually used low level debug, and call directly the TXT driver,
 * without using the TXT server thread.
 **********************************************************************************
 * @ val   : 64 bits unsigned value.
 *********************************************************************************/
void putl( uint64_t val );
void nolock_putl( uint64_t val );

/**********************************************************************************
 * This debug function displays on the TXT0 terminal the content of an
 * array of bytes defined by <buffer> and <size> arguments (16 bytes per line).
 * The <string> argument is displayed before the buffer content.
 * The line format is an address followed by 16 (hexa) bytes.
 **********************************************************************************
 * @ string   : buffer name or identifier.
 * @ buffer   : local pointer on bytes array.
 * @ size     : number of bytes to display.
 *********************************************************************************/
void putb( char     * string,
           uint8_t  * buffer,
           uint32_t   size );



#endif  // _PRINTK_H

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

