/*
 * sys_thread_sleep.c - block the calling thread on SLEEP, with or without alarm
 * 
 * Author    Alain Greiner    (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_special.h>
#include <scheduler.h>
#include <thread.h>
#include <printk.h>
#include <syscalls.h>

///////////////////////////////////////////////////////////////////////////////////////
// This static function implements the alarm handler used to wake-up a thread
// when the amarm rings after after a sleep( seconds ) syscall.
///////////////////////////////////////////////////////////////////////////////////////
// @ thread_xp  : extended pointer on blocked thread.
///////////////////////////////////////////////////////////////////////////////////////
static void __attribute__((noinline)) sleep_alarm_handler( xptr_t thread_xp )
{
    // stop the alarm
    alarm_stop( thread_xp );

    // unblock the thread
    thread_unblock( thread_xp , THREAD_BLOCKED_SLEEP );

}  // end sleep_alarm_handler()

////////////////////////////////////////
int sys_thread_sleep( uint32_t seconds )
{
    cycle_t    ncycles;    // number of cycles to sleep

    thread_t  * this      = CURRENT_THREAD;
    xptr_t      thread_xp = XPTR( local_cxy , this );

    cycle_t     tm_start = hal_get_cycles();

#if DEBUG_SYS_THREAD_SLEEP
if( DEBUG_SYS_THREAD_SLEEP < (uint32_t)tm_start )
printk("\n[%s] thread[%x,%x] enter / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, (uint32_t)tm_start );
#endif

    if( seconds == 0 )   // sleep without alarm
    {
 
#if DEBUG_SYS_THREAD_SLEEP
if( DEBUG_SYS_THREAD_SLEEP < tm_start )
printk("\n[%s] thread[%x,%x] blocks on <SLEEP> without alarm / cycle %d\n",
__FUNCTION__ , this->process->pid, this->trdid, (uint32_t)tm_start );
#endif
        // threads blocks and deschedules
        thread_block( thread_xp , THREAD_BLOCKED_SLEEP );
        sched_yield("sleep without alarm");
    }
    else                // sleep with alarm
    {
        // translate seconds to ncycles
        ncycles = seconds * LOCAL_CLUSTER->sys_clk;

        // register & start the calling thread alarm
        alarm_start( thread_xp,
                     tm_start + ncycles,
                     &sleep_alarm_handler,
                     thread_xp );
  
#if DEBUG_SYS_THREAD_SLEEP
if( DEBUG_SYS_THREAD_SLEEP < tm_start )
printk("\n[DBG] %s : thread[%x,%x] blocks on <SLEEP> for %d seconds / cycle %d\n",
__FUNCTION__ , this->process->pid,  this->trdid, seconds, (uint32_t)tm_start );
#endif
        // thread blocks & deschedules 
        thread_block( thread_xp , THREAD_BLOCKED_SLEEP );
        sched_yield("sleep with alarm");
    }

#if DEBUG_SYS_THREAD_SLEEP
if( DEBUG_SYS_THREAD_SLEEP < tm_end )
printk("\n[%s] thread[%x,%x] resume / cycle %d\n",
__FUNCTION__ , this->process->pid, this->trdid, (uint32_t)tm_end );
#endif

	return 0;

}  // end sys_thread_sleep()
