/*
 * hal_init.c - C initialization procedure for x86.
 *
 * Copyright (c) 2017 Maxime Villard
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_boot.h>
#include <hal_multiboot.h>
#include <hal_segmentation.h>
#include <hal_acpi.h>
#include <hal_lapic.h>
#include <hal_internal.h>
#include <hal_remote.h>

#include <memcpy.h>
#include <thread.h>
#include <string.h>
#include <process.h>
#include <printk.h>
#include <vmm.h>
#include <core.h>
#include <cluster.h>
#include <chdev.h>

#include <boot_info.h>

void kernel_init(boot_info_t *info);

static void gdt_create();
static void idt_create();
void cpu_attach();

size_t mytest __in_kdata = 0;

struct multiboot_info mb_info __in_kdata;
char mb_loader_name[PAGE_SIZE] __in_kdata;
uint8_t mb_mmap[PAGE_SIZE] __in_kdata;

/* -------------------------------------------------------------------------- */

static void
dump_memmap()
{
	size_t mmap_length = mb_info.mi_mmap_length;
	uint8_t *mmap_addr = (uint8_t *)&mb_mmap;
	size_t i;

	if (!(mb_info.mi_flags & MULTIBOOT_INFO_HAS_MMAP))
		x86_panic("No mmap");

	i = 0;
	while (i < mmap_length) {
		struct multiboot_mmap *mm;

		mm = (struct multiboot_mmap *)(mmap_addr + i);

		x86_printf("-> [%Z, %Z] %s\n", mm->mm_base_addr,
		    mm->mm_base_addr + mm->mm_length,
		    (mm->mm_type == 1) ? "ram" : "rsv" );

		i += mm->mm_size + 4;
	}
}

static void init_bootinfo_core(boot_core_t *core)
{
	memset(core, 0, sizeof(boot_core_t));

	core->gid = hal_lapic_gid();
	core->lid = 0;
	core->cxy = 0;
}

static void init_bootinfo_txt(boot_device_t *dev)
{
	memset(dev, 0, sizeof(boot_device_t));

	dev->base = 0xB8000;
	dev->type = (DEV_FUNC_TXT << 16) | IMPL_TXT_X86;
	dev->channels = 1;
	dev->param0 = 0;
	dev->param1 = 0;
	dev->param2 = 0;
	dev->param3 = 0;

#ifdef NOTYET
    uint32_t    irqs;    /*! number of input IRQs                    */
    boot_irq_t  irq[32]; /*! array of input IRQS (PIC and ICU only)  */
#endif
}

static void init_bootinfo(boot_info_t *info)
{
	extern uint64_t __kernel_data_start;
	extern uint64_t __kernel_end;

	memset(info, 0, sizeof(boot_info_t));

	info->signature = 0;

	info->paddr_width = 0;
	info->x_width = 1;
	info->y_width = 1;
	info->x_size = 1;
	info->y_size = 1;
	info->io_cxy = 0;

	info->ext_dev_nr = 1;
	init_bootinfo_txt(&info->ext_dev[0]);

	info->cxy = 0;
	info->cores_nr = 1;
	init_bootinfo_core(&info->core[0]);

	info->rsvd_nr = 0;
	/* rsvd XXX */

	/* dev_ XXX */

	info->pages_offset = 0;
	info->pages_nr = 0;

	info->kernel_code_start = (intptr_t)(KERNTEXTOFF - KERNBASE);
	info->kernel_code_end = (intptr_t)(&__kernel_data_start - KERNBASE) - 1;
	info->kernel_data_start = (intptr_t)(&__kernel_data_start - KERNBASE);
	info->kernel_code_end = (intptr_t)(&__kernel_end - KERNBASE) - 1;
}

void init_x86_64(paddr_t firstpa)
{
	boot_info_t btinfo;

	x86_printf("[+] init_x86_64 called\n");

	/* Create the global structures */
	gdt_create();
	idt_create();

	/* Attach cpu0 */
	cpu_attach();
	x86_printf("[+] cpu_attach called\n");

	x86_printf("[+] bootloader: '%s'\n", mb_loader_name);

	dump_memmap();
	x86_printf("[+] dump finished\n");

	hal_gpt_init(firstpa);
	x86_printf("[+] hal_gpt_init called\n");

	hal_acpi_init();
	x86_printf("[+] hal_acpi_init called\n");

	hal_gpt_bootstrap_reset();
	x86_printf("[+] hal_gpt_bootstrap_reset called\n");

	hal_lapic_init();
	x86_printf("[+] hal_lapic_init called\n");

	hal_tls_init_cpu0();
	x86_printf("[+] hal_tls_init_cpu0 called\n");

	x86_printf("-> mytest = %z\n", mytest);

	xptr_t myptr = XPTR(0, &mytest);
	hal_remote_sb(myptr, 1);
	x86_printf("-> mytest = %z\n", hal_remote_lb(myptr));

	init_bootinfo(&btinfo);
	kernel_init(&btinfo);
	x86_printf("[+] kernel_init called\n");


	sti();
	while (1);

	int m = 0;
	int v = 1 / m;

	char *buf = NULL;
	*buf = (char)0x01;

	x86_printf("ALIVE!\n");

	while (1);
}

/* -------------------------------------------------------------------------- */

uint8_t gdtstore[PAGE_SIZE] __in_kdata;
uint8_t idtstore[PAGE_SIZE] __in_kdata;
struct tss cpu0_tss __in_kdata;
uint8_t cpu0_intr_stack[STKSIZE] __in_kdata;
uint8_t cpu0_dbfl_stack[STKSIZE] __in_kdata;
uint8_t cpu0_nmfl_stack[STKSIZE] __in_kdata;

static void
setregion(struct region_descriptor *rd, void *base, uint16_t limit)
{
	rd->rd_limit = limit;
	rd->rd_base = (uint64_t)base;
}

/* -------------------------------------------------------------------------- */

static void
gdt_set_memseg(struct gdt_memseg *sd, void *base, size_t limit,
	int type, int dpl, int gran, int is64)
{
	sd->sd_lolimit = (unsigned)limit;
	sd->sd_lobase = (unsigned long)base;
	sd->sd_type = type;
	sd->sd_dpl = dpl;
	sd->sd_p = 1;
	sd->sd_hilimit = (unsigned)limit >> 16;
	sd->sd_avl = 0;
	sd->sd_long = is64;
	sd->sd_def32 = 0;
	sd->sd_gran = gran;
	sd->sd_hibase = (unsigned long)base >> 24;
}

static void
gdt_set_sysseg(struct gdt_sysseg *sd, void *base, size_t limit,
	int type, int dpl, int gran)
{
	memset(sd, 0, sizeof *sd);
	sd->sd_lolimit = (unsigned)limit;
	sd->sd_lobase = (uint64_t)base;
	sd->sd_type = type;
	sd->sd_dpl = dpl;
	sd->sd_p = 1;
	sd->sd_hilimit = (unsigned)limit >> 16;
	sd->sd_gran = gran;
	sd->sd_hibase = (uint64_t)base >> 24;
}

static void gdt_create()
{
	memset(&gdtstore, 0, PAGE_SIZE);

	/* Flat segments */
	gdt_set_memseg(GDT_ADDR_MEM(gdtstore, GDT_KCODE_SEL), 0,
	    0xfffff, SDT_MEMERA, SEL_KPL, 1, 1);
	gdt_set_memseg(GDT_ADDR_MEM(gdtstore, GDT_KDATA_SEL), 0,
	    0xfffff, SDT_MEMRWA, SEL_KPL, 1, 1);
	gdt_set_memseg(GDT_ADDR_MEM(gdtstore, GDT_UCODE_SEL), 0,
	    0xfffff, SDT_MEMERA, SEL_UPL, 1, 1);
	gdt_set_memseg(GDT_ADDR_MEM(gdtstore, GDT_UDATA_SEL), 0,
	    0xfffff, SDT_MEMRWA, SEL_UPL, 1, 1);
}

void cpu_load_gdt()
{
	struct region_descriptor region;
	setregion(&region, &gdtstore, PAGE_SIZE - 1);
	lgdt(&region);
}

/* -------------------------------------------------------------------------- */

static void
idt_set_seg(struct idt_seg *seg, void *func, int ist, int type, int dpl, int sel)
{
	seg->gd_looffset = (uint64_t)func & 0xffff;
	seg->gd_selector = sel;
	seg->gd_ist = ist;
	seg->gd_type = type;
	seg->gd_dpl = dpl;
	seg->gd_p = 1;
	seg->gd_hioffset = (uint64_t)func >> 16;
	seg->gd_zero = 0;
	seg->gd_xx1 = 0;
	seg->gd_xx2 = 0;
	seg->gd_xx3 = 0;
}

static void idt_create()
{
	extern uint64_t x86_traps[], x86_intrs[], x86_rsvd;
	struct idt_seg *idt;
	size_t i;

	idt = (struct idt_seg *)&idtstore;

	/* First, put a dead entry */
	for (i = 0; i < NIDT; i++) {
		idt_set_seg(&idt[i], (void *)&x86_rsvd, 0,
		    SDT_SYS386IGT, SEL_KPL, GDT_FIXED_SEL(GDT_KCODE_SEL, SEL_KPL));
	}

	/* General exceptions */
	for (i = CPUVEC_MIN; i < CPUVEC_MAX; i++) {
		idt_set_seg(&idt[i], (void *)x86_traps[i - CPUVEC_MIN], 0,
		    SDT_SYS386IGT, SEL_KPL, GDT_FIXED_SEL(GDT_KCODE_SEL, SEL_KPL));
	}

	/* LAPIC interrupts */
	for (i = LAPICVEC_MIN; i < LAPICVEC_MAX; i++) {
		idt_set_seg(&idt[i], (void *)x86_intrs[i - LAPICVEC_MIN], 0,
		    SDT_SYS386IGT, SEL_KPL, GDT_FIXED_SEL(GDT_KCODE_SEL, SEL_KPL));
	}
}

void cpu_load_idt()
{
	struct region_descriptor region;
	setregion(&region, &idtstore, PAGE_SIZE - 1);
	lidt(&region);
}

/* -------------------------------------------------------------------------- */

/*
 * The gdt bitmap must be per-cluster.
 */
int tss_alloc(struct tss *tss)
{
	int slot;

	/* Once we have proper SMP support, we will change that */
	slot = GDT_CPU0TSS_SEL;

	gdt_set_sysseg(GDT_ADDR_SYS(gdtstore, slot), tss,
	    sizeof(*tss) - 1, SDT_SYS386TSS, SEL_KPL, 0);

	return GDT_DYNAM_SEL(slot, SEL_KPL);
}

void cpu_create_tss()
{
	struct tss *tss = &cpu0_tss;
	int sel;

	/* Create the tss */
	memset(tss, 0, sizeof(*tss));
	tss->tss_iobase = IOMAP_INVALOFF << 16;
	tss->tss_ist[0] = (uint64_t)cpu0_intr_stack + STKSIZE;
	tss->tss_ist[1] = (uint64_t)cpu0_dbfl_stack + STKSIZE;
	tss->tss_ist[2] = (uint64_t)cpu0_nmfl_stack + STKSIZE;
	sel = tss_alloc(tss);

	/* Load it */
	ltr(sel);
}

/* -------------------------------------------------------------------------- */

void cpu_attach()
{
	cpu_load_gdt();
	cpu_load_idt();
	cpu_create_tss();
}

