/********************************************************************************/
/*	File : reset.S								*/
/*	Author : Alain Greiner							*/
/*	Date : 26/04/2012							*/
/********************************************************************************/
/* This boot code is for a multi-cluster, multi-processor architecture,		*/
/* running one or several multi-tasks software application(s) defined in the	*/
/* the mapping_info data-structure.						*/
/* It uses the mapping_info data structure to build the page tables, the tasks	*/
/* contexts, and to initialize the peripherals.					*/
/* Processor 0 is in charge of building all pages tables, all tasks contexts	*/
/* and to initialize all peripherals. Other processors are waiting until the 	*/	
/* mapping_info signature has been modified by processor 0.			*/
/* The entry point is 0xbfc00000, but the actual boot code starts at address	*/
/* 0xbfc00500, and a minimal boot exception handler is implemented at address	*/
/* 0xbfc0380.									*/
/********************************************************************************/
		
#include <giet_config.h>
#include <mips32_registers.h>

#define EXCEP_ORG    		0x380
#define START_ORG    		0x500

#define OUT_MAPPING_SIGNATURE	0xBABEF00D

	.section .boot,"ax",@progbits
	.align	2
       	.set noreorder

/********************************************************/
/* reset entry point                                    */
/* (address 0xBFC00000 imposed by the hardware)         */
/********************************************************/
boot_reset:
        j	boot_start
	nop

/*******************************************************/
/* handling exceptions in the boot phase               */
/* (address 0xBFC00380 imposed by the hardware         */
/*******************************************************/
	.align	2
        .org	EXCEP_ORG
boot_excep:
        la	a0,	boot_error_string
        jal	boot_tty_puts
        nop
        mfc0	a0,	CP0_TIME
	jal	boot_tty_putw
	nop
        la	a0,	boot_lf_string
        jal	boot_tty_puts
        nop

	la	a0,	boot_epc_string
        jal	boot_tty_puts
        nop
        mfc0	a0,	CP0_EPC
	jal	boot_tty_putw
        nop
        la	a0,	boot_lf_string
        jal	boot_tty_puts
        nop

	la	a0,	boot_cr_string
        jal	boot_tty_puts
        nop
        mfc0	a0,	CP0_CR
	jal	boot_tty_putw
        nop
        la	a0,	boot_lf_string
        jal	boot_tty_puts
        nop

	la	a0,	boot_sr_string
        jal	boot_tty_puts
        nop
        mfc0	a0,	CP0_SR
	jal	boot_tty_putw
        nop
        la	a0,	boot_lf_string
        jal	boot_tty_puts
        nop

	la	a0,	boot_bar_string
        jal	boot_tty_puts
        nop
        mfc0	a0,	CP0_BAR
	jal	boot_tty_putw
        nop
        la	a0,	boot_lf_string
        jal	boot_tty_puts
        nop

	j	boot_exit
	nop

/*******************************************/
/* actual starting point for the boot code */
/*******************************************/
        .align 	2
	.org	START_ORG

boot_start:
        /* get the procid */
        mfc0	k0,	CP0_PROCID
	andi	k0,	k0,	0x3FF	/* no more than 1024 processors... */

	/* Only processor 0 does init */
	bne	k0,	zero,	boot_wait_signature
        nop

	/* temporary stack for procesor 0 : 16K */
	la	sp,	seg_boot_stack_base
	addiu	sp,	sp,	0x4000	

        /* plat-form initialisation */
	jal	boot_init
        nop

boot_wait_signature:
	la	k0,	seg_boot_mapping_base
        cache	0x11,	0(k0)			/* invalidate local cache copy */
        lw	k0,	0(k0)			/* k0 <= mapping_info[0] */
	li	k1, 	OUT_MAPPING_SIGNATURE
        bne	k1,	k0,	boot_wait_signature
        nop
	
        /* All processors initialize SR / PTPR / SP / EPC / MODE */
        /* and jump to user code.				 */

        /* get the procid */
        mfc0	s0,	CP0_PROCID
	andi	s0,	s0,	0x3FF		/* no more than 1024 processors... */

        /* get the scheduler address s2 */
        li	k0,	256
        li	k1,	GIET_NB_TASKS_MAX
        mul	s2,	k1,	k0		/* s2 <= sizeof(context_array) */
        addiu	k0,	s2,	8		/* k0 <= sizeof(scheduler_t) */
        mul	k0,	k0,	s0		/* k0 <= proc_id*sizeof(scheduler_t) */
        la	k1,	_scheduler
        addu	s1,	k1,	k0		/* s1 <= &_scheduler[proc_id] */

        /* test number of tasks */
        addu	k1,	s1,	s2		/* k1 <= &tasks */
	lw	k1,	0(k1)			/* k1 <= tasks */
        beq	k1, zero,	boot_exit
	nop

        /* load SP */
	li	k1,	CTX_SP_ID
        sll	k1,	k1, 	2
	addu	k1,	s1,	k1
        lw	k1,	0(k1)	
        move	sp,	k1 			/* sp <= ctx[SP]  */

        /* load SR */
	li	k1,	CTX_SR_ID
        sll	k1,	k1, 	2
	addu	k1,	s1,	k1
        lw	k1,	0(k1)	
        mtc0	k1,	CP0_SR 			/* sr <= ctx[SR]  */

        /* load PTPR */
	li	k1,	CTX_PTPR_ID
        sll	k1,	k1, 	2
	addu	k1,	s1,	k1
        lw	k1,	0(k1)	
        mtc2	k1,	CP2_PTPR 		/* ptpr <= ctx[PTPR]  */

        /* load EPC */
	li	k1,	CTX_EPC_ID
        sll	k1,	k1, 	2
	addu	k1,	s1,	k1
        lw	k1,	0(k1)	
        mtc0	k1,	CP0_EPC 		/* epc <= ctx[EPC]  */

	/* activates MMU */
	li	k1,	0xF
	mtc2	k1,	CP2_MODE             	/* load MODE register */

	/* jump to user's code in user mode */
	eret

boot_error_string:	.asciiz "\n[BOOT] Fatal Error at cycle"
boot_sp_string:		.asciiz "    SP   = "
boot_sr_string:		.asciiz "    SR   = "
boot_cr_string:		.asciiz "    CR   = "
boot_epc_string:	.asciiz "    EPC  = "
boot_ptpr_string:	.asciiz "    PTPR = "
boot_bar_string:	.asciiz "    BAR  = "
boot_lf_string:		.asciiz "\n"

	.set	reorder


