///////////////////////////////////////////////////////////////////////////////////
// File     : ctx_handler.c
// Date     : 01/04/2012
// Authors  : alain greiner & joel porquet
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////////////
// The ctx_handler.h and ctx_handler.c files are part of the GIET nano-kernel.
// This code is used to support context switch when several tasks are executing
// in time multiplexing on a single processor.
// The tasks must be statically allocated to a processor in the boot phase, and
// there is one private scheduler per processor:  NB_CLUSTERS * NB_PROCS
// Each sheduler contains up to NB_TASKS_MAX contexts.
////////////////////////////////////////////////////////////////////////////////////
// A task context is an array of 64 words = 256 bytes. 
// It contains copies of processor registers, when the task is not running,
// and some general informations associated to the task.
// - It contains GPR[i], generally stored in slot (i). $0, *26 & $27 are not saved.
// - It contains HI & LO registers.
// - It contains CP0 registers: EPC, SR, CR.
// - It contains CP2 registers : PTPR and MODE.
// - It contains the TTY index for the terminal allocated to the task.
// ctx[0] <- SR   ctx[8] <- $8    ctx[16]<- $16   ctx[24]<- $24   ctx[32]<- EPC
// ctx[1] <- $1   ctx[9] <- $9    ctx[17]<- $17   ctx[25]<- $25   ctx[33]<- CR
// ctx[2] <- $2   ctx[10]<- $10   ctx[18]<- $18   ctx[26]<- LO    ctx[34]<- TTY
// ctx[3] <- $3   ctx[11]<- $11   ctx[19]<- $19   ctx[27]<- HI    ctx[35]<- PTPR
// ctx[4] <- $4   ctx[12]<- $12   ctx[20]<- $20   ctx[28]<- $28   ctx[36]<- MODE
// ctx[5] <- $5   ctx[13]<- $13   ctx[21]<- $21   ctx[29]<- $29   ctx[37]<- reserved
// ctx[6] <- $6   ctx[14]<- $14   ctx[22]<- $22   ctx[30]<- $30   ctx[38]<- reserved
// ctx[7] <- $7   ctx[15]<- $15   ctx[23]<- $23   ctx[31]<- $31   ctx[39]<- reserved
/////////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <ctx_handler.h>
#include <drivers.h>
#include <common.h>

extern void _task_switch(unsigned int *, unsigned int *);

/////////////////////////////////////////////////////////////////////////////////
//	Global variables
/////////////////////////////////////////////////////////////////////////////////

static_scheduler_t _scheduler[NB_CLUSTERS * NB_PROCS];

/////////////////////////////////////////////////////////////////////////////////
//  _ctx_select() 
// The scheduling policy is round-robin : for each processor, the task index is
// incremented, modulo the number of tasks allocated to the processor.
/////////////////////////////////////////////////////////////////////////////////
unsigned int _ctx_select( unsigned int curr_task_id,
                          unsigned int proc_id )
{
    unsigned int tasks = _scheduler[proc_id].tasks; 
    return (curr_task_id + 1) % tasks;
} 

////////////////////////////////////////////////////////////////////////////////////
// _ctx_switch()
// This function performs a context switch between the running task
// and  another task, selected by the _ctx_select() function.
// It use the global variable scheduler[] : array indexed by the procid,
// that has NB_CLUSTERS * NB_PROCS entries.
// The return address contained in $31 is saved in the _current task context
// (in the ctx[31] slot), and the function actually returns to the address
// contained in the ctx[31] slot of the new task context.
////////////////////////////////////////////////////////////////////////////////////
void _ctx_switch()
{
    unsigned int*	curr_context;	// pointer on the current task context
    unsigned int*	next_context;	// pointer on the next task context

    unsigned int	curr_task_id;
    unsigned int	next_task_id;
    unsigned int	tasks;
    unsigned int 	proc_id;

    proc_id      = _procid();
    curr_task_id = _scheduler[proc_id].current;
    tasks        = _scheduler[proc_id].tasks;

    // if no more than one task, return
    if ( tasks <= 1 ) return;

    /* save _current task context */

    curr_context = &_scheduler[proc_id].context[curr_task_id][0];

    asm volatile(
    "add     $27,    %0,  $0       \n"  /* $27<= &context[curr_task_id] */
    "mfc0    $26,    $12           \n"  /* $26 <= SR */
    "sw      $26,    0*4($27)      \n"  /* ctx[0] <= SR */
    ".set noat                     \n"
    "sw      $1,     1*4($27)      \n"  /* ctx[1] <= $1 */
    ".set at                       \n"
    "sw      $2,     2*4($27)      \n"  /* ctx[2] <= $2 */
    "sw      $3,     3*4($27)      \n"  /* ctx[3] <= $3 */
    "sw      $4,     4*4($27)      \n"  /* ctx[4] <= $4 */
    "sw      $5,     5*4($27)      \n"  /* ctx[5] <= $5 */
    "sw      $6,     6*4($27)      \n"  /* ctx[6] <= $6 */
    "sw      $7,     7*4($27)      \n"  /* ctx[7] <= $7 */
    "sw      $8,     8*4($27)      \n"  /* ctx[8] <= $8 */
    "sw      $9,     9*4($27)      \n"  /* ctx[9] <= $9 */
    "sw      $10,    10*4($27)     \n"  /* ctx[10] <= $10 */
    "sw      $11,    11*4($27)     \n"  /* ctx[11] <= $11 */
    "sw      $12,    12*4($27)     \n"  /* ctx[12] <= $12 */
    "sw      $13,    13*4($27)     \n"  /* ctx[13] <= $13 */
    "sw      $14,    14*4($27)     \n"  /* ctx[14] <= $14 */
    "sw      $15,    15*4($27)     \n"  /* ctx[15] <= $15 */
    "sw      $16,    16*4($27)     \n"  /* ctx[16] <= $16 */
    "sw      $17,    17*4($27)     \n"  /* ctx[17] <= $17 */
    "sw      $18,    18*4($27)     \n"  /* ctx[18] <= $18 */
    "sw      $19,    19*4($27)     \n"  /* ctx[19] <= $19 */
    "sw      $20,    20*4($27)     \n"  /* ctx[20] <= $20 */
    "sw      $21,    21*4($27)     \n"  /* ctx[21] <= $21 */
    "sw      $22,    22*4($27)     \n"  /* ctx[22] <= $22 */
    "sw      $23,    23*4($27)     \n"  /* ctx[23] <= $23 */
    "sw      $24,    24*4($27)     \n"  /* ctx[24] <= $24 */
    "sw      $25,    25*4($27)     \n"  /* ctx[25] <= $25 */
    "mflo    $26                   \n"
    "sw      $26,    26*4($27)     \n"  /* ctx[26] <= LO */
    "mfhi    $26                   \n"
    "sw      $26,    27*4($27)     \n"  /* ctx[27] <= H1 */
    "sw      $28,    28*4($27)     \n"  /* ctx[28] <= $28 */
    "sw      $29,    29*4($27)     \n"  /* ctx[29] <= $29 */
    "sw      $30,    30*4($27)     \n"  /* ctx[30] <= $30 */
    "sw      $31,    31*4($27)     \n"  /* ctx[31] <= $31 */
    "mfc0    $26,    $14           \n"
    "sw      $26,    32*4($27)     \n"  /* ctx[32] <= EPC */
    "mfc0    $26,    $13           \n"
    "sw      $26,    33*4($27)     \n"  /* ctx[33] <= CR */
    "mfc2    $26,    $0	           \n"
    "sw      $26,    35*4($27)     \n"  /* ctx[35] <= PTPR */
    "mfc2    $26,    $1	           \n"
    "sw      $26,    36*4($27)     \n"  /* ctx[36] <= MODE */
    :
    : "r" (curr_context)
    );

    // select next task and update scheduler state

    next_task_id = _ctx_select( curr_task_id, proc_id );
    _scheduler[proc_id].current = next_task_id;

#if GIET_DEBUG_SWITCH
unsigned int time = _proctime();
_tty_puts( "\n[GIET] Context switch for processor ");
_tty_putw( proc_id );
_tty_puts( " at cycle ");
_tty_putw( time );
_tty_puts("\n");
_tty_puts( " - tasks        = ");
_tty_putw( tasks );
_tty_puts("\n");
_tty_puts( " - curr_task_id = ");
_tty_putw( curr_task_id );
_tty_puts("\n");
_tty_puts( " - next_task_id = ");
_tty_putw( next_task_id );
_tty_puts("\n");
#endif

    /* restore next task context */

    next_context = &_scheduler[proc_id].context[next_task_id][0];

    asm volatile(
    "add     $27,    %0,  $0       \n"  /* $27<= &context[next] */
    "lw      $26,    35*4($27)     \n"
    "mtc2    $26,    $0            \n"  /* restore PTPR */
    "lw      $26,    36*4($27)     \n"
    "mtc2    $26,    $1            \n"  /* restore MODE */
    "lw      $26,    0*4($27)      \n"
    "mtc0    $26,    $12           \n"  /* restore SR */
    ".set noat                     \n"
    "lw      $1,     1*4($27)      \n"  /* restore $1 */
    ".set at                       \n"
    "lw      $2,     2*4($27)      \n"  /* restore $2 */
    "lw      $3,     3*4($27)      \n"  /* restore $3 */
    "lw      $4,     4*4($27)      \n"  /* restore $4 */
    "lw      $5,     5*4($27)      \n"  /* restore $5 */
    "lw      $6,     6*4($27)      \n"  /* restore $6 */
    "lw      $7,     7*4($27)      \n"  /* restore $7 */
    "lw      $8,     8*4($27)      \n"  /* restore $8 */
    "lw      $9,     9*4($27)      \n"  /* restore $9 */
    "lw      $10,    10*4($27)     \n"  /* restore $10 */
    "lw      $11,    11*4($27)     \n"  /* restore $11 */
    "lw      $12,    12*4($27)     \n"  /* restore $12 */
    "lw      $13,    13*4($27)     \n"  /* restore $13 */
    "lw      $14,    14*4($27)     \n"  /* restore $14 */
    "lw      $15,    15*4($27)     \n"  /* restore $15 */
    "lw      $16,    16*4($27)     \n"  /* restore $16 */
    "lw      $17,    17*4($27)     \n"  /* restore $17 */
    "lw      $18,    18*4($27)     \n"  /* restore $18 */
    "lw      $19,    19*4($27)     \n"  /* restore $19 */
    "lw      $20,    20*4($27)     \n"  /* restore $20 */
    "lw      $21,    21*4($27)     \n"  /* restore $21 */
    "lw      $22,    22*4($27)     \n"  /* restore $22 */
    "lw      $23,    23*4($27)     \n"  /* restore $23 */
    "lw      $24,    24*4($27)     \n"  /* restore $24 */
    "lw      $25,    25*4($27)     \n"  /* restore $25 */
    "lw      $26,    26*4($27)     \n"
    "mtlo    $26                   \n"  /* restore LO */
    "lw      $26,    27*4($27)     \n"
    "mthi    $26                   \n"  /* restore HI */
    "lw      $28,    28*4($27)     \n"  /* restore $28 */
    "lw      $29,    29*4($27)     \n"  /* restore $29 */
    "lw      $30,    30*4($27)     \n"  /* restore $30 */
    "lw      $31,    31*4($27)     \n"  /* restore $31 */
    "lw      $26,    32*4($27)     \n"
    "mtc0    $26,    $14           \n"  /* restore EPC */
    "lw      $26,    33*4($27)     \n"
    "mtc0    $26,    $13           \n"  /* restore CR */
    "eret                          \n"  /* returns to user code */
    :
    : "r" (next_context)
    );
} // end task_switch()

