/* -*- c++ -*-
 *
 * SOCLIB_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute vSO and/or modify vSO
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that vSO will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * SOCLIB_LGPL_HEADER_END
 *
 * Copyright (c) UPMC, Lip6, SoC
 *         Mohamed Lamine KARAOUI <Mohamed.Karaoui@lip6.fr>, 2012
 */

#include <string.h>
#include <cassert>
#include <sstream>
#include <fstream>
#include <ios>
#include <iostream>
#include <cstdarg>
#include <cassert>
#include <iomanip>


#include "exception.h"
#include "mover.h"


#define MOVER_DEBUG

Mover::Mover( const std::string &filename, 
        const size_t pageSize)
        :m_path(filename),
        m_pathHandler(filename),
        m_ginit(false),
        m_generator(new elfpp::object())
{
    
    PSeg::setPageSize(pageSize);

    load_bin(m_path);

#ifdef MOVER_DEBUG
    std::cout << "Binary file path: " << m_path << std::endl;
    //print_mapping_info(m_data);
#endif

    buildMap(m_data);
    
#ifdef MOVER_DEBUG
    std::cout << "parsing done" << std::endl ;
#endif

#ifdef MOVER_DEBUG
    print_mapping();
    std::cout << *this << std::endl;
#endif

    m_psegh.check();
#ifdef MOVER_DEBUG
    std::cout << "checking done" << std::endl ;
#endif

}

void Mover::print( std::ostream &o ) const
{
    std::cout << "All sections:" << std::endl;
    FOREACH( sect, m_generator->get_section_table() )
    {
        assert(&*sect != NULL);
        std::cout << *sect << std::endl; 
    }
}

void Mover::print_mapping() const
{
    std::cout << m_psegh << std::endl;
}

//TODO:delete
elfpp::section* Mover::get_sect_by_addr(elfpp::object *loader, unsigned int addr)
{
    FOREACH( sect, loader->get_section_table() )
    {
        assert(&*sect != NULL);
        elfpp::sh_flags_e eflags = sect->get_flags();
        if ( !(eflags & elfpp::SHF_ALLOC) )
            continue;
        if(sect->get_load_address() == addr) //load_addr ?
        {
            return (&*sect);
        }
    }
    return NULL;
}
elfpp::section* Mover::get_sect_by_name(elfpp::object *loader, std::string name)
{
#ifdef MOVER_DEBUG
    std::cout << "get_sect_by_name " << name << std::endl;
#endif
    FOREACH( sect, loader->get_section_table() )
    {
        assert(&*sect != NULL);
        elfpp::sh_flags_e eflags = sect->get_flags();
        if ( !(eflags & elfpp::SHF_ALLOC) )
            continue;

#ifdef MOVER_DEBUG
        std::cout << "Trying " << sect->get_name() << std::endl;
#endif
        if(!(sect->get_name()).compare(name)) 
        {
            return (&*sect);
        }
    }
    return NULL;
}

void Mover::buildSoft(const std::string &filename)
{
    if(!m_ginit)
        throw soclib::exception::RunTimeError(std::string("Can't get generator! ") );

    m_generator->write(filename);

}

void* Mover::load_bin(std::string filename)
{
    
#ifdef MOVER_DEBUG
    std::cout  << "Trying to load the binary blob from file '" << m_path << "'" << std::endl;
#endif

    std::ifstream input(m_path.c_str(), std::ios_base::binary|std::ios_base::in);

    if ( ! input.good() )
        throw soclib::exception::RunTimeError(std::string("Can't open the file: ") + m_path);

    input.seekg( 0, std::ifstream::end );
    m_size = input.tellg();
    input.seekg( 0, std::ifstream::beg );

    //m_data = new void*[m_size];
    m_data = std::malloc(m_size);
    if ( !m_data )
        throw soclib::exception::RunTimeError("malloc failed... No memory space");

    input.read( (char*)m_data, m_size );
    
    return m_data;
}


Mover::~Mover()
{
    //std::cout << "Deleted Mover " << *this << std::endl;
    std::free(m_data);
    std::map<std::string, elfpp::object*>::iterator l;
    for(l = m_loaders.begin(); l != m_loaders.end(); l++)
    {
        delete (*l).second;
    }
    //delete m_generator;
}


/////////////////////////////////////////////////////////////////////////////
// various mapping_info data structure access functions
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
mapping_pseg_t* Mover::get_pseg_base( mapping_header_t* header )
{
    return   (mapping_pseg_t*)    ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters);
}
/////////////////////////////////////////////////////////////////////////////
mapping_vspace_t* Mover::get_vspace_base( mapping_header_t* header )
{
    return   (mapping_vspace_t*)  ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs);
}
/////////////////////////////////////////////////////////////////////////////
mapping_vseg_t* Mover::get_vseg_base( mapping_header_t* header )
{
    return   (mapping_vseg_t*)    ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs +
                                  MAPPING_VSPACE_SIZE*header->vspaces);
}
/////////////////////////////////////////////////////////////////////////////
mapping_vobj_t* Mover::get_vobj_base( mapping_header_t* header )
{
    return   (mapping_vobj_t*)    ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs +
                                  MAPPING_VSPACE_SIZE*header->vsegs +
                                  MAPPING_VSPACE_SIZE*header->vspaces);
}

/////////////////////////////////////////////////////////////////////////////
// print the content of the mapping_info data structure 
////////////////////////////////////////////////////////////////////////
void Mover::print_mapping_info(void* desc)
{
    mapping_header_t*   header = (mapping_header_t*)desc;  

    mapping_pseg_t*	    pseg    = get_pseg_base( header );;
    mapping_vspace_t*	vspace  = get_vspace_base ( header );;
    mapping_vseg_t*	    vseg    = get_vseg_base ( header );
    mapping_vobj_t*	    vobj    = get_vobj_base ( header );

    // header
    std::cout << std::hex << "mapping_info" << std::endl
              << " + signature = " << header->signature << std::endl
              << " + name = " << header->name << std::endl
              << " + clusters = " << header->clusters << std::endl
              << " + psegs = " << header->psegs << std::endl
             << " + ttys = " << header->ttys  << std::endl
             << " + vspaces = " << header->vspaces  << std::endl
             << " + globals = " << header->globals  << std::endl
             << " + vsegs = " << header->vsegs  << std::endl
             << " + tasks = " << header->tasks  << std::endl;

    // psegs
    for ( size_t pseg_id = 0 ; pseg_id < header->psegs ; pseg_id++ )
    {
        std::cout << "pseg " << pseg_id << std::endl
         << " + name = " << pseg[pseg_id].name << std::endl 
         << " + base = " << pseg[pseg_id].base << std::endl 
         << " + length = " << pseg[pseg_id].length << std::endl ;
    }

    // globals
    for ( size_t vseg_id = 0 ; vseg_id < header->globals ; vseg_id++ )
    {
        std::cout << "global vseg: " << vseg_id << std::endl
         << " + name = " << vseg[vseg_id].name << std::endl 
         << " + vbase = " << vseg[vseg_id].vbase << std::endl 
         << " + length = " << vseg[vseg_id].length << std::endl 
         << " + mode = " << (size_t)vseg[vseg_id].mode << std::endl 
         << " + ident = " << (bool)vseg[vseg_id].ident << std::endl 
         << " + psegname" << pseg[vseg[vseg_id].psegid].name << std::endl;
        for(size_t vobj_id = vseg[vseg_id].vobj_offset ; vobj_id < vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs ; vobj_id++ )
        {
            std::cout << "\t vobjs     = "<<     vobj[vobj_id].name << std::endl;
            std::cout<<"\t name     =" << vobj[vobj_id].name <<std::endl;
            std::cout<<"\t type     =" << vobj[vobj_id].type <<std::endl;
            std::cout<<"\t length   =" << vobj[vobj_id].length <<std::endl;
            std::cout<<"\t align    =" << vobj[vobj_id].align <<std::endl;
            std::cout<<"\t binpath  =" << vobj[vobj_id].binpath <<std::endl;
            std::cout<<"\t \n";
        }
    }


    // vspaces
    for ( size_t vspace_id = 0 ; vspace_id < header->vspaces ; vspace_id++ )
    {
        std::cout << "***vspace: " << vspace_id << "***" << std::endl
         << " + name = " <<  vspace[vspace_id].name  << std::endl 
         << " + ttys = " <<  vspace[vspace_id].ttys  << std::endl;

        for ( size_t vseg_id = vspace[vspace_id].vseg_offset ; 
              vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs) ; 
              vseg_id++ )
        {
            std::cout << "private vseg: ";
            std::cout <<  vseg_id  << std::endl
             << " + name = " <<  vseg[vseg_id].name  << std::endl
             << " + vbase = " <<  vseg[vseg_id].vbase  << std::endl
             << " + length = " <<  vseg[vseg_id].length  << std::endl
             << " + mode = " <<  (size_t)vseg[vseg_id].mode  << std::endl
             << " + ident = " <<  (bool)vseg[vseg_id].ident  << std::endl
             << " + psegname = " << pseg[vseg[vseg_id].psegid].name  << std::endl << std::endl;
            for(size_t vobj_id = vseg[vseg_id].vobj_offset ; vobj_id < vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs ; vobj_id++ )
            {
                std::cout<<"\t\t vobjs     =" << vobj[vobj_id].name <<std::endl;
                std::cout<<"\t\t name     =" << vobj[vobj_id].name <<std::endl;
                std::cout<<"\t\t type     =" << vobj[vobj_id].type <<std::endl;
                std::cout<<"\t\t length   =" << vobj[vobj_id].length <<std::endl;
                std::cout<<"\t\t align    =" << vobj[vobj_id].align <<std::endl;
                std::cout<<"\t\t binpath  =" << vobj[vobj_id].binpath <<std::endl;
            }
        }

    }
} // end print_mapping_info()

///////////////////////////////////////////////////////////////////////////
void Mover::pseg_map( mapping_pseg_t* pseg) 
{
    std::string name(pseg->name);
    m_psegh.m_pSegs.push_back(PSeg(name, pseg->base, pseg->length));
}


///////////////////////////////////////////////////////////////////////////
void Mover::vseg_map( mapping_vseg_t* vseg) 
{

    mapping_vobj_t*     vobj   = get_vobj_base( (mapping_header_t*) m_data ); 
    PSeg *ps = &(m_psegh.get(vseg->psegid));// get physical segment pointer(PSegHandler::get)
    elfpp::section* sect = NULL;
    size_t cur_vaddr;
    size_t cur_paddr;
    bool first = true;
    bool aligned = false;

    VSeg   *vSO = new VSeg;

    vSO->m_name = std::string(vseg->name);
    vSO->m_vma = vseg->vbase;
    vSO->m_lma = ps->nextLma();

    cur_vaddr = vseg->vbase;
    cur_paddr = ps->nextLma();

    mapping_vobj_t* cur_vobj;
    size_t simple_size = 0; //for debug
    
#ifdef MOVER_DEBUG
    std::cout << "--------------------vseg_map "<< vseg->name <<"---------------------" << std::endl;
#endif
    
    for ( size_t vobj_id = vseg->vobj_offset ; vobj_id < (vseg->vobj_offset + vseg->vobjs) ; vobj_id++ )
    {
        cur_vobj = &vobj[vobj_id];

#ifdef MOVER_DEBUG
        std::cout << "current vobj("<< vobj_id <<"): " << cur_vobj->name << " (" <<cur_vobj->vaddr << ")" 
                        << " size: "<< cur_vobj->length << " type: " <<  cur_vobj->type << std::endl;
#endif

        if(cur_vobj->type == ELF)
        {
            if(!first) 
                throw soclib::exception::RunTimeError(std::string("elf vobj type, must be placed first in a vseg"));

            size_t elf_size;
            std::string filePath(m_pathHandler.getFullPath(std::string(cur_vobj->binpath)));
#ifdef MOVER_DEBUG
            std::cout << "Handling: " << filePath << " ..." << std::endl;
#endif
            if(!filePath.compare(m_path))    //local blob: map_info
            {
#ifdef MOVER_DEBUG
                std::cout << "Found the vseg of the mapping info" << std::endl;
#endif
                /**creating a new section */
                sect = new elfpp::section(*m_generator, elfpp::SHT_PROGBITS);

                sect->set_name(std::string(cur_vobj->name));
                sect->set_flags(elfpp::SHF_ALLOC | elfpp::SHF_WRITE);
                sect->set_size(this->m_size);
                sect->set_content(this->m_data);                

                elf_size = this->m_size;       
                assert((elf_size >0) and "MAPPING INFO file is empty !?");
            }
            else
            { 
#ifdef MOVER_DEBUG
                std::cout << "Found an ELF vseg" << std::endl;
#endif
                if(m_loaders.count(filePath) == 0 )
                    m_loaders[filePath] = new elfpp::object(filePath);
                elfpp::object* loader = m_loaders[filePath];//TODO:free!?

                //sect =  get_sect_by_name(loader, std::string(cur_vobj->name));
                //assert(( sect->get_vaddr() == cur_vaddr) and "Vaddr doesn't match!");
                sect =  get_sect_by_addr(loader, cur_vaddr);
                assert(sect and "No section found");

                sect->set_name(std::string(cur_vobj->name));

                elf_size = sect->get_size();
                assert((elf_size >0) and "ELF section empty ?");

                if(!m_ginit)
                {
                    /** Initailising the header of the generator from the first binary
                    * we suppose the header is the same for all the binarys
                    * */
                    m_generator->copy_info(*loader);
                    m_ginit=true;
                }
            }

            if(elf_size > cur_vobj->length)
                std::cout << "Warning, specified elf type vobj ("<<
                cur_vobj->name  <<") size is "<< cur_vobj->length << ", the actual size is "
                << elf_size  << std::endl;

            cur_vobj->length = elf_size;//set the true size of this ELF vobj

            vSO->m_file = filePath;
            vSO->m_loadable = true;        
        }
        first = false;

        //aligning the vobj->paddr if necessary
        if(cur_vobj->align)
        {
            cur_paddr = PSeg::align(cur_paddr, cur_vobj->align);
            aligned = true;
        }

        cur_vaddr += cur_vobj->length;
        cur_paddr += cur_vobj->length;
        simple_size += cur_vobj->length;
    }

    assert((cur_vaddr >= vseg->vbase ));
    assert((cur_paddr >= ps->nextLma() ));

    vSO->m_length = (cur_paddr - ps->nextLma()); //pageAlign is done by the psegs

#ifdef MOVER_DEBUG
    if(aligned)
    {
        std::cout << "vseg base "<< std::hex << ps->nextLma()
        <<(ps->nextLma()+simple_size)  <<" size " << std::dec << simple_size <<
        std::endl; 

        std::cout << "vseg aligned to: base: " << std::hex << ps->nextLma()
        <<" to "<< std::hex << ps->nextLma()+vSO->m_length<< " size " << std::dec <<
        vSO->m_length << std::endl;
    }
#endif
     
    vSO->m_ident = vseg->ident;      

    //set the lma
    if ( vseg->ident != 0 )            // identity mapping required
        ps->addIdent( *vSO );
    else
        ps->add( *vSO );
    
    if(!sect)
        return;

    //std::cout << "section: "<< *sect <<"\n seted to: " << (*vSO) << std::endl;
    sect->set_vaddr((*vSO).lma());
    m_generator->add_section(*(new elfpp::section(*sect)));

} // end vseg_map()


/////////////////////////////////////////////////////////////////////
void Mover::buildMap(void* desc)
{
    mapping_header_t*   header = (mapping_header_t*)desc;  

    mapping_vspace_t*   vspace = get_vspace_base( header );     
    mapping_pseg_t*     pseg   = get_pseg_base( header ); 
    mapping_vseg_t*     vseg   = get_vseg_base( header );

    // get the psegs
#ifdef MOVER_DEBUG
std::cout << "\n******* Storing Pseg information *********\n" << std::endl;
#endif
    for ( size_t pseg_id = 0 ; pseg_id < header->psegs ; pseg_id++ )
    {
        pseg_map( &pseg[pseg_id]);
    }

    // map global vsegs
#ifdef MOVER_DEBUG
std::cout << "\n******* mapping global vsegs *********\n" << std::endl;
#endif
    for ( size_t vseg_id = 0 ; vseg_id < header->globals ; vseg_id++ )
    {
        vseg_map( &vseg[vseg_id]);
    }

    // second loop on virtual spaces to map private vsegs
    for (size_t vspace_id = 0 ; vspace_id < header->vspaces ; vspace_id++ )
    {

#ifdef MOVER_DEBUG
std::cout << "\n******* mapping all vsegs of " << vspace[vspace_id].name << " *********\n" << std::endl;
#endif
            
        for ( size_t vseg_id = vspace[vspace_id].vseg_offset ; 
              vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs) ; 
              vseg_id++ )
        {
            vseg_map( &vseg[vseg_id]); 
        }
    } 

} // end buildMap()



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

