////////////////////////////////////////////////////////////////////////////
// File     : xml_driver.c
// Date     : 04/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////
// This program translate a binary file containing a MAPPING_INFO 
// data structure to an xml file.
////////////////////////////////////////////////////////////////////////////

#include  <stdlib.h>
#include  <fcntl.h>
#include  <unistd.h>
#include  <stdio.h>
#include  <string.h>
#include  <stdint.h>
#include  <mapping_info.h>

/////////////////////////////////////////////////////
void buildXml( mapping_header_t* header, FILE* fpout)
{

    const char* 
    vobj_type[] = 
    { 
        "ELF",
        "PTAB",   //page table 
        "PERI",   //hardware component
        "MWMR",   //MWMR channel
        "LOCK",   //Lock
        "BUFFER", //Any "no intialiasation needed" objects (stacks...)
        "BARRIER" //Barrier
    };
    const char* mode_str[] = { "____",
                               "___U", 
                               "__W_", 
                               "__WU", 
                               "_X__", 
                               "_X_U", 
                               "_XW_", 
                               "_XWU", 
                               "C___", 
                               "C__U", 
                               "C_W_", 
                               "C_WU", 
                               "CX__", 
                               "CX_U", 
                               "CXW_", 
                               "CXWU" };

    unsigned int		vspace_id;
    unsigned int		cluster_id;
    unsigned int		pseg_id;
    unsigned int		vseg_id;
    unsigned int		vobj_id;
    unsigned int		task_id;

    mapping_cluster_t*	cluster;
    mapping_pseg_t*	pseg;
    mapping_vspace_t*	vspace;
    mapping_vseg_t*	vseg;
    mapping_vobj_t*	vobj;
    mapping_task_t*	task;

    // computes the base adresss for clusters array, 
    cluster = (mapping_cluster_t*)((char*)header +
                                  MAPPING_HEADER_SIZE );

    // computes the base adresss for psegs array, 
    pseg    = (mapping_pseg_t*)   ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters );

    // computes the base adresss for vspaces array, 
    vspace  = (mapping_vspace_t*) ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs );

    // computes the base adresss for vsegs array, 
    vseg    = (mapping_vseg_t*)   ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs +
                                  MAPPING_VSPACE_SIZE*header->vspaces );

    // computes the base adresss for vobjs array, 
    vobj    = (mapping_vobj_t*)   ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs +
                                  MAPPING_VSPACE_SIZE*header->vspaces +
                                  MAPPING_VSEG_SIZE*header->vsegs );

    // computes the base address for tasks array 
    task    = (mapping_task_t*)   ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs +
                                  MAPPING_VSPACE_SIZE*header->vspaces +
                                  MAPPING_VOBJ_SIZE*header->vobjs +
                                  MAPPING_VSEG_SIZE*header->vsegs );

    fprintf( fpout, "<?xml version = \"1.0\"?>\n\n");

    ///////////////////////// header /////////////////////////////////////////////

    fprintf( fpout, "<mapping_info signature = \"0x%x\"\n", header->signature);
    fprintf( fpout, "              name      = \"%s\"\n", header->name);
    fprintf( fpout, "              clusters  = \"%d\"\n", header->clusters);
    fprintf( fpout, "              psegs     = \"%d\"\n", header->psegs);
    fprintf( fpout, "              ttys      = \"%d\"\n", header->ttys);
    fprintf( fpout, "              fbs       = \"%d\"\n", header->fbs);
    fprintf( fpout, "              vspaces   = \"%d\"\n", header->vspaces);
    fprintf( fpout, "              globals   = \"%d\" >\n\n", header->globals);

    ///////////////////// clusters ///////////////////////////////////////////////

    fprintf( fpout, "    <clusterset>\n" );
    for ( cluster_id = 0 ; cluster_id < header->clusters ; cluster_id++ )
    {
        fprintf( fpout, "        <cluster index  = \"%d\"\n",      cluster_id);
        fprintf( fpout, "                 procs  = \"%d\" />\n\n", cluster[cluster_id].procs);
    }
    fprintf( fpout, "    </clusterset>\n" );

    //////////////////// psegs ///////////////////////////////////////////////////

    fprintf( fpout, "    <psegset>\n" );
    for ( pseg_id = 0 ; pseg_id < header->psegs ; pseg_id++ )
    {
        fprintf( fpout, "        <pseg    name   = \"%s\"\n",        pseg[pseg_id].name);
        fprintf( fpout, "                 base   = \"0x%x\"\n",      pseg[pseg_id].base);
        fprintf( fpout, "                 length = \"0x%x\" />\n\n", pseg[pseg_id].length);
    }
    fprintf( fpout, "    </psegset>\n" );


    /////////////////// globals /////////////////////////////////////////////////

    fprintf( fpout, "    <globalset>\n" );
    for ( vseg_id = 0 ; vseg_id < header->globals ; vseg_id++ )
    {
        unsigned int pseg_id = vseg[vseg_id].psegid; 

        fprintf( fpout, "        <vseg    name     = \"%s\"\n",      vseg[vseg_id].name);
        fprintf( fpout, "                 vbase    = \"0x%x\"\n",    vseg[vseg_id].vbase);
        fprintf( fpout, "                 mode     = \"%s\"\n", mode_str[vseg[vseg_id].mode]);
        fprintf( fpout, "                 psegname = \"%s\"\n",      pseg[pseg_id].name);
        fprintf( fpout, "                 ident    = \"%d\" >\n",    vseg[vseg_id].ident);
        for ( vobj_id = vseg[vseg_id].vobj_offset;
              vobj_id < (vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs); 
              vobj_id++ )
        {
            fprintf( fpout, "                 <vobj name     = \"%s\"\n",    vobj[vobj_id].name);
            fprintf( fpout, "                       type     = \"%s\"\n",    vobj_type[vobj[vobj_id].type]);
            fprintf( fpout, "                       length   = \"0x%x\"\n",  vobj[vobj_id].length);
            fprintf( fpout, "                       align    = \"%d\"\n",    vobj[vobj_id].align);
            fprintf( fpout, "                       init     = \"%d\" \n",   vobj[vobj_id].init);
            fprintf( fpout, "                       binpath  = \"%s\" />\n", vobj[vobj_id].binpath);
        }
        fprintf( fpout, "        </vseg>\n\n");
    }
    fprintf( fpout, "    </globalset>\n" );

    //////////////////// vspaces ////////////////////////////////////////////////

    fprintf( fpout, "    <vspaceset>\n" );
    for ( vspace_id = 0 ; vspace_id < header->vspaces ; vspace_id++ )
    {
        unsigned int func_id = vspace[vspace_id].vobj_offset + vspace[vspace_id].start_offset; 
        fprintf( fpout, "        <vspace  name      = \"%s\"\n",     vspace[vspace_id].name); 
        fprintf( fpout, "                 startname = \"%s\" >\n\n", vobj[func_id].name); 

        for ( vseg_id = vspace[vspace_id].vseg_offset ;
              vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs) ; vseg_id++ )
        {
            unsigned int pseg_id = vseg[vseg_id].psegid; 

            fprintf( fpout, "                 <vseg name      = \"%s\"\n",      vseg[vseg_id].name);
            fprintf( fpout, "                       vbase     = \"0x%x\"\n",    vseg[vseg_id].vbase);
            fprintf( fpout, "                       mode      = \"%s\"\n", mode_str[vseg[vseg_id].mode]);
            fprintf( fpout, "                       psegname  = \"%s\"\n",      pseg[pseg_id].name);
            fprintf( fpout, "                       ident     = \"%d\" >\n",    vseg[vseg_id].ident);

            for ( vobj_id = vseg[vseg_id].vobj_offset ;
                  vobj_id < (vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs) ; 
                  vobj_id++ )
            {
                fprintf( fpout, "                       <vobj name     = \"%s\"\n",    vobj[vobj_id].name);
                fprintf( fpout, "                             type     = \"%s\" \n",   vobj_type[vobj[vobj_id].type]);
                fprintf( fpout, "                             length   = \"0x%x\" \n", vobj[vobj_id].length);
                fprintf( fpout, "                             align    = \"%d\" \n",   vobj[vobj_id].align);
                fprintf( fpout, "                             init     = \"%d\" \n",   vobj[vobj_id].init);
                fprintf( fpout, "                             binpath  = \"%s\" />\n", vobj[vobj_id].binpath);
            }
            fprintf( fpout, "                 </vseg>\n\n");
        }
        for ( task_id = vspace[vspace_id].task_offset ;
              task_id < (vspace[vspace_id].task_offset + vspace[vspace_id].tasks) ; 
              task_id++ )
        {
            unsigned int vobj_id = task[task_id].vobjlocid + vspace[vspace_id].vobj_offset; 

            fprintf( fpout, "                 <task name      = \"%s\"\n",      task[task_id].name);
            fprintf( fpout, "                       clusterid = \"%d\"\n",      task[task_id].clusterid);
            fprintf( fpout, "                       proclocid = \"%d\"\n",      task[task_id].proclocid);
            fprintf( fpout, "                       stackname = \"%s\"\n",      vobj[vobj_id].name);
            fprintf( fpout, "                       startid   = \"%d\"\n",      task[task_id].startid);
            fprintf( fpout, "                       usetty    = \"%d\"\n",      task[task_id].use_tty);
            fprintf( fpout, "                       usefb     = \"%d\" />\n\n", task[task_id].use_fb);
        }
        fprintf( fpout, "        </vspace>\n\n");
    }
    fprintf( fpout, "    </vspaceset>\n" );
    fprintf( fpout, "</mapping_info>\n");
} // end buildXml()

/////////////////////////////////////
int  main ( int argc, char* argv[] )
{
    if ( argc < 2 ) 
    {
        printf("Usage: bin2xml <input_file_path> <output_file_path>\n");
        return 1;
    }

    unsigned int      bin[0x10000];		// 64 K int = 256 Kbytes 

    int fdin = open( argv[1], O_RDONLY );
    if (fdin < 0) 
    {
        perror("open");
        exit(1);
    }

    FILE* fpout = fopen( argv[2], "w" );
    if (fpout == NULL) 
    {
        perror("open");
        exit(1);
    }

    unsigned int length = read(fdin, bin, 0x40000);

    if ( length <= 0 )
    {
        perror("read");
        exit(1);
    }

    if ( bin[0] == IN_MAPPING_SIGNATURE )
    {
        buildXml( (mapping_header_t*)bin, fpout );
    } 
    else
    {
        printf("[ERROR] Wrong file format\n");
        exit(1);
    }
    return 0;
} // end main()
