////////////////////////////////////////////////////////////////////////////
// File     : mapping_info.h
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////
// The MAPPING_INFO data structure can be used with the GIET.
// It contains the mapping directive for one or several virtual spaces.
// Ech virtual space contains a variable number of virtual segments
// and a variable number of tasks. The number of virtual space can be one.
//
// The mapping table data structure is organised as the concatenation of
// a fixed size header, and 6 variable size arrays:
//
// - mapping_header_t   header              (MAPPING_HEADER_SIZE)
// - mapping_cluster_t  cluster[clusters]   (MAPPING_CLUSTER_SIZE * clusters)
// - mapping_pseg_t     pseg[psegs]         (MAPPING_PSEG_SIZE * psegs)
// - mapping_vspace_t   vspace[vspaces]     (MAPPING_VSPACE_SIZE * vspaces)
// - mapping_vseg_t     vseg[vsegs]         (MAPPING_VSEG_SIZE * vsegs)
// - mapping_vseg_t     vobj[vsegs]         (MAPPING_VOBJ_SIZE * vsegs)
// - mapping_task_t     task[tasks]         (MAPPING_TASK_SIZE * tasks)
//
// The number of clusters and the number of vspaces are defined in the header.
// The number of psegs are defined in each cluster.
// The number of vsegs, tasks ans mwmrs are defined in each vspace.
//
// It is intended to be stored in the boot ROM at address MAPPING_BOOT_BASE. 
// For each cluster, the base address of the first pseg descriptor
// is defined by a pseg_offset relative to MAPPING_BOOT_BASE.
// For each vspace, the base address of the first vseg descriptor
// is defined by a vseg_offset relative to MAPPING_BOOT_BASE.
// For each vspace, the base address of the first task desciptor
// is defined by a task_offset relative to MAPPING_BOOT_BASE.
// For each vspace, the base address of the first mwmr desciptor
// is defined by a mwmr_offset relative to MAPPING_BOOT_BASE.
////////////////////////////////////////////////////////////////////////////

#ifndef _MAPPING_INFO_H_
#define _MAPPING_INFO_H_

#define MAPPING_HEADER_SIZE     sizeof(mapping_header_t)
#define MAPPING_CLUSTER_SIZE    sizeof(mapping_cluster_t)
#define MAPPING_VSPACE_SIZE     sizeof(mapping_vspace_t)
#define MAPPING_VSEG_SIZE	    sizeof(mapping_vseg_t)
#define MAPPING_VOBJ_SIZE	    sizeof(mapping_vobj_t)
#define MAPPING_PSEG_SIZE	    sizeof(mapping_pseg_t)
#define MAPPING_TASK_SIZE	    sizeof(mapping_task_t)

#define C_MODE_MASK     0b1000      // cacheable
#define X_MODE_MASK     0b0100      // executable
#define W_MODE_MASK     0b0010      // writable
#define U_MODE_MASK     0b0001      // user access

#define IN_MAPPING_SIGNATURE    0xDEADBEEF
#define OUT_MAPPING_SIGNATURE   0xBABEF00D

enum 
{
    VOBJ_TYPE_ELF     = 0,     // loadable code object
    VOBJ_TYPE_PTAB    = 1,     // page table 
    VOBJ_TYPE_PERI    = 2,     // hardware component
    VOBJ_TYPE_MWMR    = 3,     // MWMR channel
    VOBJ_TYPE_LOCK    = 4,     // Lock
    VOBJ_TYPE_BUFFER  = 5,     // Any "no intialiasation needed" objects (stacks...)
    VOBJ_TYPE_BARRIER = 6,     // Barrier
};


///////////////////////////////
typedef struct mapping_header_s
{
    unsigned int    signature;      // must contain MAPPING_SIGNATURE
	unsigned int	clusters;	    // number of clusters
	unsigned int	psegs;	        // number of psegs
    unsigned int    ttys;           // number of TTY terminals 
    unsigned int    fbs;            // number of Frame Buffer DMA channels
	unsigned int	globals;		// number of vsegs mapped in all vspaces
	unsigned int	vspaces;		// number of virtual spaces
	unsigned int	vsegs;  		// total number of virtual segments (for all vspaces)
	unsigned int	vobjs;  		// total number of virtual objects (for all vspaces)
	unsigned int	tasks;  		// total number of tasks (for all vspaces)
    char            name[32];       // mapping name
} mapping_header_t;

////////////////////////////////
typedef struct mapping_cluster_s
{
    unsigned int    procs;          // number of processors in cluster
} mapping_cluster_t;

/////////////////////////////
typedef struct mapping_pseg_s 
{
    char            name[32];       // pseg name (unique in a cluster)
	unsigned int    base;           // base address in physical space
	unsigned int	length;         // size (bytes)
    unsigned int    next_base;      // first free page base address
} mapping_pseg_t;

///////////////////////////////
typedef struct mapping_vspace_s
{
    char            name[32];       // virtual space name
    unsigned int    start_offset;   // offset of the vobj containing the start vector
	unsigned int	vsegs;		    // number of vsegs in vspace
	unsigned int	vobjs;		    // number of vobjs in vspace
	unsigned int	tasks;		    // number of tasks in vspace
    unsigned int    vseg_offset;    // index of first vseg in vspace 
    unsigned int    vobj_offset;    // index of first vobjs in vspace
    unsigned int    task_offset;    // index of first task in vspace
} mapping_vspace_t;

/////////////////////////////
typedef struct mapping_vseg_s 
{
	char            name[32];       // vseg name (unique in vspace)
	unsigned int    vbase;          // base address in virtual space (hexa)
	unsigned int    pbase;          // base address in physical space (hexa)
	unsigned int	length;         // size (bytes)
	unsigned int    psegid;	        // physical segment index
	unsigned int    mode;	        // C-X-W-U flags
    unsigned int    ident;          // identity mapping if non zero
	unsigned int	vobjs;		    // number of vobjs in vseg
    unsigned int    vobj_offset;    // index of first vobjs in vseg
} mapping_vseg_t;

/////////////////////////////
typedef struct mapping_task_s 
{
	char            name[32];       // task name (unique in vspace)
	unsigned int	clusterid;	    // physical cluster index
	unsigned int	proclocid;      // processor local index (inside cluster)
    unsigned int    vobjlocid;      // stack vobj index in vspace
    unsigned int    startid;        // index in start_vector 
    unsigned int    use_tty;        // TTY terminal required
    unsigned int    use_fb;         // DMA channel to frame buffer required
} mapping_task_t;

/////////////////////////////
typedef struct mapping_vobj_s 
{
    char            name[32];       // vobj name (unique in a vspace)
    char            binpath[64];    // path for the binary code ("*.elf")
	unsigned int    type;           // type of vobj
	unsigned int	length;         // size (bytes)
	unsigned int	align;          // required alignement (logarithm of 2)
	unsigned int	vaddr;          // virtual base addresse of the vobj
	unsigned int	paddr;          // physical base addresse of the vobj
	unsigned int	init;           // init value (number of participants for a barrier)
} mapping_vobj_t;

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

