///////////////////////////////////////////////////////////////////////////////////
// File     : ctx_handler.c
// Date     : 01/04/2012
// Authors  : alain greiner & joel porquet
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////////////
// The ctx_handler.h and ctx_handler.c files are part of the GIET nano-kernel.
// This code is used to support context switch when several tasks are executing
// in time multiplexing on a single processor.
// The tasks must be statically allocated to a processor in the boot phase, and
// there is one private scheduler per processor:  NB_CLUSTERS * NB_PROCS
// Each sheduler contains up to NB_TASKS_MAX contexts.
////////////////////////////////////////////////////////////////////////////////////
// A task context is an array of 64 words = 256 bytes. 
// It contains copies of processor registers, when the task is not running,
// and some general informations associated to the task.
//
// - It contains GPR[i], generally stored in slot (i). $0, *26 & $27 are not saved.
// - It contains HI & LO registers.
// - It contains CP0 registers: EPC, SR, CR.
// - It contains CP2 registers : PTPR and MODE.
// - It contains the TTY global index, the FBDMA global index, the virtual base 
//   address of the page table (PTAB), and the task global index (TASK).
//
// ctx[0]<- SR|ctx[8] <- $8 |ctx[16]<- $16|ctx[24]<- $24|ctx[32]<- EPC |ctx[40]<- TTY
// ctx[1]<- $1|ctx[9] <- $9 |ctx[17]<- $17|ctx[25]<- $25|ctx[33]<- CR  |ctx[41]<- FBDMA
// ctx[2]<- $2|ctx[10]<- $10|ctx[18]<- $18|ctx[26]<- LO |ctx[34]<- *** |ctx[42]<- PTAB
// ctx[3]<- $3|ctx[11]<- $11|ctx[19]<- $19|ctx[27]<- HI |ctx[35]<- PTPR|ctx[43]<- TASK
// ctx[4]<- $4|ctx[12]<- $12|ctx[20]<- $20|ctx[28]<- $28|ctx[36]<- MODE|ctx[44]<- ***
// ctx[5]<- $5|ctx[13]<- $13|ctx[21]<- $21|ctx[29]<- SP |ctx[37]<- *** |ctx[45]<- ***
// ctx[6]<- $6|ctx[14]<- $14|ctx[22]<- $22|ctx[30]<- $30|ctx[38]<- *** |ctx[46]<- ***
// ctx[7]<- $7|ctx[15]<- $15|ctx[23]<- $23|ctx[31]<- RA |ctx[39]<- *** |ctx[47]<- ***
/////////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <drivers.h>
#include <common.h>
#include <ctx_handler.h>
#include <mapping_info.h>
#include <sys_handler.h>

extern void _task_switch(unsigned int *, unsigned int *);

/////////////////////////////////////////////////////////////////////////////////
//	Global variables : array of schedulers (one scheduler per processor)
/////////////////////////////////////////////////////////////////////////////////

__attribute__((section (".kdata"))) static_scheduler_t _scheduler[NB_CLUSTERS * NB_PROCS];

/////////////////////////////////////////////////////////////////////////////////
//	_ctx_switch()
// This function performs a context switch between the running task
// and  another task, using a round-robin sheduling policy.
// It use the global variable scheduler[] : array indexed by the procid,
// that contains NB_CLUSTERS * NB_PROCS entries.
// The return address contained in $31 is saved in the _current task context
// (in the ctx[31] slot), and the function actually returns to the address
// contained in the ctx[31] slot of the new task context. To perform the
// actual switch, it calls the _task_switch function written in assembly language.
/////////////////////////////////////////////////////////////////////////////////
void _ctx_switch()
{
    unsigned char curr_task_id;
    unsigned char next_task_id;

    unsigned int *curr_context;
    unsigned int *next_context;

    unsigned int proc_id   = _procid();
    unsigned int tasks     = _scheduler[proc_id].tasks;

    // return if only one task  */
    if ( tasks <= 1) return;
 
    // compute the task context base address for the current task
    curr_task_id = _scheduler[proc_id].current;
    curr_context = &(_scheduler[proc_id].context[curr_task_id][0]);
    
    // select the next task using a round-robin scheduling policy
    next_task_id = (curr_task_id + 1) % tasks;
    
    // compute the task context base address for the next task
    next_context = &(_scheduler[proc_id].context[next_task_id][0]);

#if GIET_DEBUG_SWITCH
_get_lock( &_tty_put_lock );
_puts( "\n[GIET] Context switch for processor ");
_putw( proc_id );
_puts( " at cycle ");
_putw( _proctime() );
_puts("\n");
_puts( " - tasks        = ");
_putw( tasks );
_puts("\n");
_puts( " - curr_task_id = ");
_putw( curr_task_id );
_puts("\n");
_puts( " - next_task_id = ");
_putw( next_task_id );
_puts("\n");
_release_lock( &_tty_put_lock );
#endif

    //  update the scheduler state, and makes the task switch
    _scheduler[proc_id].current = next_task_id;
    _task_switch( curr_context, next_context );

} // end _ctx_switch

