/* -*- c++ -*-
 *
 * SOCLIB_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute vSO and/or modify vSO
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that vSO will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * SOCLIB_LGPL_HEADER_END
 *
 * Copyright (c) UPMC, Lip6, SoC
 *         Mohamed Lamine KARAOUI <Mohamed.Karaoui@lip6.fr>, 2012
 */

#include <string.h>
#include <cassert>
#include <sstream>
#include <fstream>
#include <ios>
#include <iostream>
#include <cstdarg>
#include <cassert>
#include <iomanip>


#include "exception.h"
#include "memo.h"

//#define MOVER_DEBUG

MeMo::MeMo( const std::string &filename, 
        const size_t pageSize)
        :m_path(filename),
        m_pathHandler(filename),
        m_ginit(false),
        m_generator(new elfpp::object())
{
    
    PSeg::setPageSize(pageSize);

    /* loading map_info blob */
    m_data = std::malloc(bin_size(m_path));
    if ( !m_data )
        throw exception::RunTimeError("malloc failed... No memory space");

    m_size = load_bin(m_path, m_data);

    /* checking signature */
    mapping_header_t*   header = (mapping_header_t*)m_data; 
    //assert((IN_MAPPING_SIGNATURE == header->signature) and "wrong signature");
    if((IN_MAPPING_SIGNATURE != header->signature))
    {
        std::cout  << "wrong signature " << std::hex <<header->signature << ", should be:"<< IN_MAPPING_SIGNATURE << std::endl;
        exit(1);
    }
    
    

#ifdef MOVER_DEBUG
    std::cout << "Binary file path: " << m_path << std::endl;
    print_mapping_info(m_data);
#endif

    buildMap(m_data);
    
#ifdef MOVER_DEBUG
    std::cout << "parsing done" << std::endl ;
#endif

#ifdef MOVER_DEBUG
    print_mapping();
    std::cout << *this << std::endl;
#endif

    m_psegh.check();
#ifdef MOVER_DEBUG
    std::cout << "checking done" << std::endl ;
#endif

}

void MeMo::print( std::ostream &o ) const
{
    std::cout << "All sections:" << std::endl;
    FOREACH( sect, m_generator->get_section_table() )
    {
        assert(&*sect != NULL);
        std::cout << *sect << std::endl; 
    }
}

void MeMo::print_mapping() const
{
    std::cout << m_psegh << std::endl;
}

elfpp::section* MeMo::get_sect_by_addr(elfpp::object *loader, unsigned int addr)
{
    FOREACH( sect, loader->get_section_table() )
    {
        assert(&*sect != NULL);
        elfpp::sh_flags_e eflags = sect->get_flags();
        if ( !(eflags & elfpp::SHF_ALLOC) )
            continue;
        if(sect->get_load_address() == addr) //load_addr ?
        {
            return (&*sect);
        }
    }
    return NULL;
}


void MeMo::buildSoft(const std::string &filename)
{
    if(!m_ginit)
        throw exception::RunTimeError(std::string("Can't get generator! ") );

    m_generator->write(filename);

}

/* Load the content of filename in buffer, and send the size */
////////////////////////////////////////////////////////
size_t MeMo::load_bin(std::string filename, void* buffer)
{

#ifdef MOVER_DEBUG
    std::cout  << "Trying to load the binary blob from file '" << filename << "'" << std::endl;
#endif

    std::ifstream input(filename.c_str(), std::ios_base::binary|std::ios_base::in);

    if ( ! input.good() )
        throw exception::RunTimeError(std::string("Can't open the file: ") + filename);

    input.seekg( 0, std::ifstream::end );
    size_t size = input.tellg();
    input.seekg( 0, std::ifstream::beg );

    if ( !buffer )
        throw exception::RunTimeError("Empty buffer!");

    input.read( (char*)buffer, size );

    input.close();

    return size;
}


/* get the size of the content of the filename */
//////////////////////////////////////////
size_t MeMo::bin_size(std::string filename)
{

#ifdef MOVER_DEBUG
    std::cout  << "Trying to get the size of the binary blob '" << filename << "'" << std::endl;
#endif

    std::ifstream input(filename.c_str(), std::ios_base::binary|std::ios_base::in);

    if ( ! input.good() )
        throw exception::RunTimeError(std::string("Can't open the file: ") + filename);

    input.seekg( 0, std::ifstream::end );
    size_t size = input.tellg();
    input.seekg( 0, std::ifstream::beg );

    input.close();

    return size;
}


/////////////
MeMo::~MeMo()
{
    //std::cout << "Deleted MeMo " << *this << std::endl;
    //std::free(m_data);//should be done by the elfpp driver since we passed the pointer
    std::map<std::string, elfpp::object*>::iterator l;
    for(l = m_loaders.begin(); l != m_loaders.end(); l++)
    {
        delete (*l).second;
    }
    //delete m_generator;
}


/////////////////////////////////////////////////////////////////////////////
// various mapping_info data structure access functions
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
mapping_pseg_t* MeMo::get_pseg_base( mapping_header_t* header )
{
    return   (mapping_pseg_t*)    ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters);
}
/////////////////////////////////////////////////////////////////////////////
mapping_vspace_t* MeMo::get_vspace_base( mapping_header_t* header )
{
    return   (mapping_vspace_t*)  ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs);
}
/////////////////////////////////////////////////////////////////////////////
mapping_vseg_t* MeMo::get_vseg_base( mapping_header_t* header )
{
    return   (mapping_vseg_t*)    ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs +
                                  MAPPING_VSPACE_SIZE*header->vspaces);
}
/////////////////////////////////////////////////////////////////////////////
mapping_vobj_t* MeMo::get_vobj_base( mapping_header_t* header )
{
    return   (mapping_vobj_t*)    ((char*)header +
                                  MAPPING_HEADER_SIZE +
                                  MAPPING_CLUSTER_SIZE*header->clusters +
                                  MAPPING_PSEG_SIZE*header->psegs +
                                  MAPPING_VSPACE_SIZE*header->vspaces +
                                  MAPPING_VSEG_SIZE*header->vsegs);
}

/////////////////////////////////////////////////////////////////////////////
// print the content of the mapping_info data structure 
////////////////////////////////////////////////////////////////////////
void MeMo::print_mapping_info(void* desc)
{
    mapping_header_t*   header = (mapping_header_t*)desc;  

    mapping_pseg_t*	    pseg    = get_pseg_base( header );;
    mapping_vspace_t*	vspace  = get_vspace_base ( header );;
    mapping_vseg_t*	    vseg    = get_vseg_base ( header );
    mapping_vobj_t*	    vobj    = get_vobj_base ( header );

    // header
    std::cout << std::hex << "mapping_info" << std::dec << std::endl
              << " + signature = " << header->signature << std::endl
              << " + name      = " << header->name      << std::endl
              << " + clusters  = " << header->clusters  << std::endl
              << " + psegs     = " << header->psegs     << std::endl
              << " + ttys      = " << header->ttys      << std::endl
              << " + fbs       = " << header->fbs       << std::endl
              << " + vspaces   = " << header->vspaces   << std::endl
              << " + globals   = " << header->globals   << std::endl
              << " + vsegs     = " << header->vsegs     << std::endl
              << " + vobjs     = " << header->vsegs     << std::endl
              << " + tasks     = " << header->tasks     << std::endl;

    // psegs
    for ( size_t pseg_id = 0 ; pseg_id < header->psegs ; pseg_id++ )
    {
        std::cout << "pseg " << pseg[pseg_id].name << std::hex << std::endl 
                  << " + base   = " << pseg[pseg_id].base   << std::endl 
                  << " + length = " << pseg[pseg_id].length << std::endl ;
    }

    // globals
    for ( size_t vseg_id = 0 ; vseg_id < header->globals ; vseg_id++ )
    {
        std::cout << std::endl;
        std::cout << "global vseg "   << vseg[vseg_id].name << std::hex << std::endl 
                  << " + vbase    = " << vseg[vseg_id].vbase << std::endl 
                  << " + length   = " << vseg[vseg_id].length << std::endl 
                  << " + mode     = " << (size_t)vseg[vseg_id].mode << std::endl 
                  << " + ident    = " << (bool)vseg[vseg_id].ident << std::endl 
                  << " + psegname = " << pseg[vseg[vseg_id].psegid].name << std::endl;
        for( size_t vobj_id = vseg[vseg_id].vobj_offset ; 
                    vobj_id < vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs ; 
                    vobj_id++ )
        {
            std::cout << "\t vobj "        << vobj[vobj_id].name    << std::endl
                      << "\t + index   = " << std::dec << vobj_id   << std::endl
                      << "\t + type    = " << vobj[vobj_id].type    << std::endl
                      << "\t + length  = " << vobj[vobj_id].length  << std::endl
                      << "\t + align   = " << vobj[vobj_id].align   << std::endl
                      << "\t + binpath = " << vobj[vobj_id].binpath << std::endl
                      << "\t + init    = " << vobj[vobj_id].init    << std::endl;
        }
    }

    // vspaces
    for ( size_t vspace_id = 0 ; vspace_id < header->vspaces ; vspace_id++ )
    {
        std::cout << "***vspace "  << vspace[vspace_id].name   << std::endl 
                  << " + vsegs = " <<  vspace[vspace_id].vsegs << std::endl 
                  << " + vobjs = " <<  vspace[vspace_id].vobjs << std::endl 
                  << " + tasks = " <<  vspace[vspace_id].tasks << std::endl;

        for ( size_t vseg_id = vspace[vspace_id].vseg_offset ; 
              vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs) ; 
              vseg_id++ )
        {
            std::cout << "\t vseg " << vseg[vseg_id].name  << std::endl
                      << "\t + vbase    = " << vseg[vseg_id].vbase             << std::endl
                      << "\t + length   = " << vseg[vseg_id].length            << std::endl
                      << "\t + mode     = " << (size_t)vseg[vseg_id].mode      << std::endl
                      << "\t + ident    = " << (bool)vseg[vseg_id].ident       << std::endl
                      << "\t + psegname = " << pseg[vseg[vseg_id].psegid].name << std::endl;
            for(size_t vobj_id = vseg[vseg_id].vobj_offset ; 
                       vobj_id < vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs ; 
                       vobj_id++ )
            {
                std::cout << "\t\t vobj "      << vobj[vobj_id].name      << std::endl
                          << "\t\t + index   = " << std::dec << vobj_id   << std::endl
                          << "\t\t + type    = " << vobj[vobj_id].type    << std::endl
                          << "\t\t + length  = " << vobj[vobj_id].length  << std::endl
                          << "\t\t + align   = " << vobj[vobj_id].align   << std::endl
                          << "\t\t + binpath = " << vobj[vobj_id].binpath << std::endl
                          << "\t\t + init    = " << vobj[vobj_id].init    << std::endl;
            }
        }

    }
} // end print_mapping_info()

//////////////////////////////////////////
void MeMo::pseg_map( mapping_pseg_t* pseg) 
{
    std::string name(pseg->name);
    m_psegh.m_pSegs.push_back(PSeg(name, pseg->base, pseg->length));
}


//////////////////////////////////////////
void MeMo::vseg_map( mapping_vseg_t* vseg)
{
    mapping_vobj_t*     vobj   = get_vobj_base( (mapping_header_t*) m_data );
    PSeg *ps = &(m_psegh.get(vseg->psegid));// get physical segment pointer(PSegHandler::get)
    elfpp::section* sect = NULL;
    size_t cur_vaddr;
    size_t cur_paddr;
    bool first = true;
    bool aligned = false;

    VSeg   *vSO = new VSeg;

    vSO->m_name = std::string(vseg->name);
    vSO->m_vma = vseg->vbase;
    vSO->m_lma = ps->nextLma();

    cur_vaddr = vseg->vbase;
    cur_paddr = ps->nextLma();

    mapping_vobj_t* cur_vobj;
    size_t simple_size = 0; //for debug

#ifdef MOVER_DEBUG
    std::cout << "--------------------vseg_map "<< vseg->name <<"---------------------" << std::endl;
#endif

    for ( size_t vobj_id = vseg->vobj_offset ; vobj_id < (vseg->vobj_offset + vseg->vobjs) ; vobj_id++ )
    {
        cur_vobj = &vobj[vobj_id];

#ifdef MOVER_DEBUG
        std::cout << std::hex << "current vobj("<< vobj_id <<"): " << cur_vobj->name << " (" <<cur_vobj->vaddr << ")"
                        << " size: "<< cur_vobj->length << " type: " <<  cur_vobj->type << std::endl;
#endif
        if(cur_vobj->type == VOBJ_TYPE_BLOB)
        {
            size_t blob_size;
            std::string filePath(m_pathHandler.getFullPath(std::string(cur_vobj->binpath)));

#ifdef MOVER_DEBUG
            std::cout << std::hex << "Handling: " << filePath << " ..." << std::endl;
#endif

            if(!filePath.compare(m_path))    //local blob: map_info
            {
#ifdef MOVER_DEBUG
                std::cout << "Found the vseg of the mapping info" << std::endl;
#endif
                blob_size = this->m_size;
                assert((blob_size >0) and "MAPPING INFO file is empty !?");
            }
            else
            {
#ifdef MOVER_DEBUG
                std::cout << "Found an BLOB vseg" << std::endl;
#endif
                blob_size = bin_size(filePath);
            }


            /**creating a new section */
            sect = new elfpp::section(*m_generator, elfpp::SHT_PROGBITS);

            sect->set_name(std::string(cur_vobj->name));
            sect->set_flags(elfpp::SHF_ALLOC | elfpp::SHF_WRITE);
            sect->set_size(blob_size);//do the malloc for the get_content fonction

            assert(sect->get_content());//check allocation

            if(!filePath.compare(m_path))    //local blob: map_info
                //memcpy(sect->get_content(), m_data, sect->get_size());
                /* this way the modification of the elf size are propageted to the giet */
                sect->set_content(this->m_data);
            else
                load_bin(filePath, sect->get_content());


            if(blob_size > cur_vobj->length)
            {
                std::cout << std::hex << "!!! Warning, specified blob type vobj ("<<
                cur_vobj->name  <<") size is "<< cur_vobj->length << ", the actual size is "
                << blob_size  << " !!!" <<std::endl;
                assert(0 and "blob vobj length smaller than the actual content" );//???
            }

            cur_vobj->length = blob_size;//set the true size of this BLOB vobj

            vSO->m_file = filePath;
            vSO->m_loadable = true;
        }
        else if(cur_vobj->type == VOBJ_TYPE_ELF)
        {
            if(!first)
                throw exception::RunTimeError(std::string("elf vobj type, must be placed first in a vseg"));

            size_t elf_size;
            std::string filePath(m_pathHandler.getFullPath(std::string(cur_vobj->binpath)));
#ifdef MOVER_DEBUG
            std::cout << "Handling: " << filePath << " ..." << std::endl;
            std::cout << "Found an ELF vseg" << std::endl;
#endif
            if(m_loaders.count(filePath) == 0 )
                m_loaders[filePath] = new elfpp::object(filePath);
            elfpp::object* loader = m_loaders[filePath];//TODO:free!?

            sect =  get_sect_by_addr(loader, cur_vaddr);
            assert(sect and "No section found");

            sect->set_name(std::string(cur_vobj->name));


            elf_size = sect->get_size();
            assert((elf_size > 0) and "ELF section empty ?");

            if(!m_ginit)
            {
                /** Initailising the header of the generator from the first binary,
                ** we suppose that the header is the same for all the binarys **/
                m_generator->copy_info(*loader);
                m_ginit=true;
            }

            if(elf_size > cur_vobj->length)
            {
                std::cout << "Warning, specified elf type vobj ("<<
                cur_vobj->name  <<") size is "<< cur_vobj->length << ", the actual size is "
                << elf_size  << std::endl;
                //assert((elf_size < cur_vobj->length) and "elf vobj length smaller than the actual content" );//???
                assert((0) and "elf vobj length smaller than the actual content" );//???
            }

            cur_vobj->length = elf_size;//set the true size of this ELF vobj

            vSO->m_file = filePath;
            vSO->m_loadable = true;
        }

        //aligning the vobj->paddr if necessary
        //
        if(cur_vobj->align)
        {
            cur_paddr = PSeg::align(cur_paddr, cur_vobj->align);
            aligned = true;
        }

        cur_vaddr += cur_vobj->length;
        cur_paddr += cur_vobj->length;
        simple_size += cur_vobj->length;
        first = false;
    }

    assert((cur_vaddr >= vseg->vbase ));
    assert((cur_paddr >= ps->nextLma() ));

    vSO->m_length = (cur_paddr - ps->nextLma()); //pageAlign is done by the psegs

#ifdef MOVER_DEBUG
    if(aligned)
    {
        std::cout << "vseg base "<< std::hex << ps->nextLma()
        <<(ps->nextLma()+simple_size)  <<" size " << std::dec << simple_size <<
        std::endl;

        std::cout << "vseg aligned to: base: " << std::hex << ps->nextLma()
        <<" to "<< std::hex << ps->nextLma()+vSO->m_length<< " size " << std::dec <<
        vSO->m_length << std::endl;
    }
#endif

    vSO->m_ident = vseg->ident;

    //set the lma
    if ( vseg->ident != 0 )            // identity mapping required
        ps->addIdent( *vSO );
    else
        ps->add( *vSO );

    if(!sect)
        return;

#ifdef MOVER_DEBUG
    std::cout << "section: "<< *sect <<"\n seted to: " << (*vSO) << std::endl;
#endif

    sect->set_vaddr((*vSO).lma());
    m_generator->add_section(*(new elfpp::section(*sect)));

} // end vseg_map()
 

///////////////////////////////
void MeMo::buildMap(void* desc)
{
    mapping_header_t*   header = (mapping_header_t*)desc;  

    mapping_vspace_t*   vspace = get_vspace_base( header );     
    mapping_pseg_t*     pseg   = get_pseg_base( header ); 
    mapping_vseg_t*     vseg   = get_vseg_base( header );

    // get the psegs

#ifdef MOVER_DEBUG
std::cout << "\n******* Storing Pseg information *********\n" << std::endl;
#endif

    for ( size_t pseg_id = 0 ; pseg_id < header->psegs ; pseg_id++ )
    {
        pseg_map( &pseg[pseg_id]);
    }

    // map global vsegs

#ifdef MOVER_DEBUG
std::cout << "\n******* mapping global vsegs *********\n" << std::endl;
#endif

    for ( size_t vseg_id = 0 ; vseg_id < header->globals ; vseg_id++ )
    {
        vseg_map( &vseg[vseg_id]);
    }

    // loop on virtual spaces to map private vsegs
    for (size_t vspace_id = 0 ; vspace_id < header->vspaces ; vspace_id++ )
    {

#ifdef MOVER_DEBUG
std::cout << "\n******* mapping all vsegs of " << vspace[vspace_id].name << " *********\n" << std::endl;
#endif
            
        for ( size_t vseg_id = vspace[vspace_id].vseg_offset ; 
              vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs) ; 
              vseg_id++ )
        {
            vseg_map( &vseg[vseg_id]); 
        }
    } 

} // end buildMap()



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

