///////////////////////////////////////////////////////////////////////////////////////
// File     : xml_parser.c
// Date     : 04/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////////
// This program translate an xml file containing a MAPPING_INFO data structure 
// to a binary file that can be directly loaded in the boot ROM and used by the GIET.
// The C strcutures are defined in the mapping_info.h file.
///////////////////////////////////////////////////////////////////////////////////////

#include  <stdlib.h>
#include  <fcntl.h>
#include  <sys/types.h>
#include  <sys/stat.h>
#include  <unistd.h>
#include  <stdio.h>
#include  <string.h>
#include  <libxml/xmlreader.h>
#include  <mapping_info.h>

#define MAX_CLUSTERS	1024
#define MAX_PSEGS		4096
#define MAX_VSPACES		1024
#define MAX_TASKS		4096
#define MAX_MWMRS		4096
#define MAX_VSEGS		4096
#define MAX_VOBJS		8192

#define XML_PARSER_DEBUG	0

///////////////////////////////////////////////////////////////////////////////////
// 	global variables used to store and index the data structures
///////////////////////////////////////////////////////////////////////////////////

mapping_header_t*	header;
mapping_cluster_t*	cluster[MAX_CLUSTERS];		// cluster array
mapping_pseg_t*		pseg[MAX_PSEGS];		// pseg array
mapping_vspace_t*	vspace[MAX_VSPACES];		// vspace array
mapping_vseg_t*		vseg[MAX_VSEGS];		// vseg array
mapping_vobj_t*		vobj[MAX_VOBJS];		// vobj array
mapping_task_t*		task[MAX_TASKS];		// task array

unsigned int		cluster_index  = 0;
unsigned int		vspace_index   = 0;
unsigned int		global_index   = 0;
unsigned int		pseg_index     = 0;
unsigned int		vseg_index     = 0;
unsigned int		vseg_loc_index = 0;
unsigned int		task_index     = 0;
unsigned int		task_loc_index = 0;
unsigned int		vobj_index     = 0;
unsigned int		vobj_loc_index = 0;
unsigned int		vobj_count     = 0;

unsigned int		tty_index      = 1;
unsigned int		fb_index       = 0;
            
//////////////////////////////////////////////////
unsigned int getIntValue( xmlTextReaderPtr reader, 
                          const char*      attributeName, 
                          unsigned int*    ok )
{
    unsigned int 	value = 0;
    unsigned int 	i;
    char	c;

    char* string = (char*)xmlTextReaderGetAttribute(reader, (const xmlChar*)attributeName);

    if ( string == NULL )  // missing argument
    {
        *ok = 0;
        return 0;
    }
    else
    {
        if ( (string[0] == '0') && ((string[1] == 'x') || (string[1] == 'X')) )  // Hexa
        {
            for ( i = 2 ; (string[i] != 0) && (i < 10) ; i++ )
            {
                c = string[i];
                if      ((c >= '0') && (c <= '9')) value = (value<<4) + string[i] - 48;
                else if ((c >= 'a') && (c <= 'f')) value = (value<<4) + string[i] - 87;
                else if ((c >= 'A') && (c <= 'F')) value = (value<<4) + string[i] - 55;
                else    
                {
                    *ok = 0;
                    return 0;
                }
            }
        }
        else 								// Decimal
        {
            for ( i = 0 ; (string[i] != 0) && (i < 9) ; i++ )
            {
                c = string[i];
                if ((c >= '0') && (c <= '9')) value = (value*10) + string[i] - 48;
                else 
                {
                    *ok = 0;
                    return 0;
                }
            }
        }
        *ok = 1;
        return value; 
    }
} // end getIntValue()

///////////////////////////////////////////////
char* getStringValue ( xmlTextReaderPtr reader, 
                       const char* 	attributeName, 
                       unsigned int* 	ok )
{
    char* string = (char*)xmlTextReaderGetAttribute(reader, (const xmlChar*)attributeName);

    if ( string == NULL )  // missing argument
    {
        *ok = 0;
        return NULL;
    }
    else
    {
        *ok = 1;
        return string;
    }
} // end getStringValue()

//////////////////////////
int getPsegId( char* str )
{
    unsigned int pseg_id;

    for ( pseg_id = 0 ; pseg_id < header->psegs ; pseg_id++ )
    {
        if ( strcmp(pseg[pseg_id]->name, str) == 0 ) return pseg_id;
    }
    return -1;
}

//////////////////////////////////////////
int getVobjLocId( unsigned int	vspace_id,
                  char*		str )
{
    unsigned int vobj_id;
    unsigned int vobj_min = vspace[vspace_id]->vobj_offset;
    unsigned int vobj_max = vobj_min + vobj_loc_index;

    for ( vobj_id = vobj_min ; vobj_id < vobj_max ; vobj_id++ )
    {
        if ( strcmp(vobj[vobj_id]->name, str) == 0 )
        {
             return (vobj_id - vobj_min);
        }
    }
    return -1;
}

//////////////////////////////////////////
void  taskNode ( xmlTextReaderPtr reader )
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( task_index >= MAX_TASKS )
    {
        printf("[XML ERROR] The number of tasks is larger than %d\n", MAX_TASKS);
    }

#if XML_PARSER_DEBUG
printf("   task %d\n", task_loc_index);
#endif

    task[task_index] = (mapping_task_t*)malloc(sizeof(mapping_task_t));

    ////////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      name = %s\n", str);
#endif
        strncpy( task[task_index]->name, str, 31 );
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for task (%d,%d)\n", 
               vspace_index, task_loc_index);
        exit(1);
    }

    ///////// get clusterid attribute
    value = getIntValue(reader,"clusterid", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      clusterid = %x\n", value);
#endif
        if ( value >= header->clusters )
        {
            printf("[XML ERROR] <clusterid> too large for task (%d,%d)\n",
                   vspace_index, task_loc_index);
            exit(1);
        }
        task[task_index]->clusterid = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <clusterid> attribute for task (%d,%d)\n",
               vspace_index, task_loc_index);
        exit(1);
    }

    ////////// get proclocid attribute
    value = getIntValue(reader,"proclocid", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      proclocid = %x\n", value);
#endif
        if ( value >= cluster[task[task_index]->clusterid]->procs )
        {
            printf("[XML ERROR] <proclocid> too large for task (%d,%d)\n",
                   vspace_index, task_loc_index);
            exit(1);
        }
        task[task_index]->proclocid = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <locprocid> attribute for task (%d,%d)\n", 
                vspace_index, task_loc_index);
        exit(1);
    }

    ////////// get stackname attribute
    str = getStringValue(reader, "stackname" , &ok);
    if ( ok )
    {
        int index = getVobjLocId( vspace_index, str );
        if ( index >= 0 ) 
        {
#if XML_PARSER_DEBUG
printf("      stackname = %s\n", str);
printf("      stackid   = %d\n", index);
#endif
            task[task_index]->vobjlocid = index;
        }
        else              
        {
            printf("[XML ERROR] illegal or missing <stackname> for task (%d,%d)\n", 
                    vspace_index, task_loc_index);
            exit(1);
        }
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <stackname> for task (%d,%d)\n", 
                vspace_index, task_loc_index);
        exit(1);
    }

    ////////// get startid  attribute
    value = getIntValue(reader,"startid", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      startid = %x\n", value);
#endif
        task[task_index]->startid = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <startid> attribute for task (%d,%d)\n", 
                vspace_index, task_loc_index);
        exit(1);
    }

    /////////// get use_tty  attribute (optionnal : 0 if missing)
    value = getIntValue(reader,"usetty", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      usetty = %x\n", value);
#endif
        if ( (value != 0) && (tty_index >= header->ttys) )
        {
            printf("[XML ERROR] The tty index is too large for task (%d,%d)\n", 
                vspace_index, task_loc_index);
            exit(1);
        }
        task[task_index]->use_tty = value;
        if (value != 0) tty_index++;
    }  
    else
    {
        task[task_index]->use_tty = 0;
    }

    /////////// get use_fb  attribute (optionnal : 0 if missing)
    value = getIntValue(reader,"usefb", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      usefb = %x\n", value);
#endif
        if ( (value != 0) && (fb_index >= header->fbs) )
        {
            printf("[XML ERROR] The fb channel index is too large for task (%d,%d)\n", 
                vspace_index, task_loc_index);
            exit(1);
        }
        task[task_index]->use_fb = value;
        if (value != 0) fb_index++;
    }  
    else
    {
        task[task_index]->use_fb = 0;
    }

    task_index++;
    task_loc_index++;
} // end taskNode()

//////////////////////////////////////////
void  vobjNode ( xmlTextReaderPtr reader )
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( vobj_index >= MAX_VOBJS )
    {
        printf("[XML ERROR] The number of vobjs is larger than %d\n", MAX_VSEGS);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("      vobj %d\n", vobj_loc_index);
#endif

    vobj[vobj_index] = (mapping_vobj_t*)malloc(sizeof(mapping_vobj_t));

    ///////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        name = %s\n", str);
#endif
        strncpy( vobj[vobj_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for vobj (%d,%d)\n", 
                vseg_index, vobj_loc_index);
        exit(1);
    }

    //////// get type attribute
    char *type_str = getStringValue(reader, "type", &ok);
#if XML_PARSER_DEBUG
printf("        type = %s\n", type_str);
#endif
    if (ok && (strcmp(type_str, "ELF") == 0))
    {
        vobj[vobj_index]->type = VOBJ_TYPE_ELF;

        //check that this vobj is the first in vseg
        if(vobj_count != 0) 
        {
            printf("[XML ERROR] an ELF vobj must be alone in a vseg (%d,%d)\n", 
                    vspace_index, vobj_loc_index);
            exit(1);
        }
    }
    else if (ok && (strcmp(type_str, "BLOB")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_BLOB;
    else if (ok && (strcmp(type_str, "PTAB")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_PTAB;
    else if (ok && (strcmp(type_str, "PERI")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_PERI;
    else if (ok && (strcmp(type_str, "MWMR")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_MWMR;
    else if (ok && (strcmp(type_str, "LOCK")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_LOCK;
    else if (ok && (strcmp(type_str, "BUFFER")  == 0)) vobj[vobj_index]->type = VOBJ_TYPE_BUFFER;
    else if (ok && (strcmp(type_str, "BARRIER") == 0)) vobj[vobj_index]->type = VOBJ_TYPE_BARRIER;
    else
    {
        printf("[XML ERROR] illegal or missing <type> attribute for vobj (%d,%d)\n", 
                vspace_index, vobj_loc_index);
        exit(1);
    }

    ////////// get length attribute 
    value = getIntValue(reader,"length", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        length = %d\n", value);
#endif
        vobj[vobj_index]->length = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <length> attribute for vobj (%d,%d)\n", 
                vspace_index, vobj_loc_index);
        exit(1);
    }

    ////////// get align attribute (optional : 0 if missing)
    value = getIntValue(reader,"align", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        align = %d\n", value);
#endif
        vobj[vobj_index]->align = value;
    }  
    else
    {
        vobj[vobj_index]->align = 0;
    }

    ////////// get binpath attribute (optional : '\0' if missing)
    str = getStringValue(reader, "binpath", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        binpath = %s\n", str);
#endif
        strncpy(vobj[vobj_index]->binpath, str, 63);
    }  
    else
    {
        vobj[vobj_index]->binpath[0] = '\0';
    }
    
    ////////// get init attribute (mandatory for mwmr and barrier)
    value = getIntValue(reader,"init", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        init  = %d\n", value);
#endif
        vobj[vobj_index]->init = value;
    }  
    else
    {
        if(vobj[vobj_index]->type == VOBJ_TYPE_MWMR || vobj[vobj_index]->type == VOBJ_TYPE_BARRIER )
        {
            printf("[XML ERROR] illegal or missing <value> attribute for vobj (%d,%d), of type: %s\n", 
                    vspace_index, vobj_loc_index, type_str);
            exit(1);
        }
        else
            vobj[vobj_index]->init = 0;
    }

    vobj_index++;
    vobj_count++;
    vobj_loc_index++;
} // end vobjNode()

//////////////////////////////////////////
void  vsegNode ( xmlTextReaderPtr reader )
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

	vobj_count = 0;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( vseg_index >= MAX_VSEGS )
    {
        printf("[XML ERROR] The number of vsegs is larger than %d\n", MAX_VSEGS);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("    vseg %d\n", vseg_loc_index);
#endif

    vseg[vseg_index] = (mapping_vseg_t*)malloc(sizeof(mapping_vseg_t));
    
    ////////// set vobj_offset attributes
    vseg[vseg_index]->vobj_offset = vobj_index;
#if XML_PARSER_DEBUG
printf("      vobj_offset = %d\n", vobj_index);
#endif

    ///////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      name = %s\n", str);
#endif
        strncpy( vseg[vseg_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for vseg (%d,%d)\n", 
                vspace_index, vseg_loc_index);
        exit(1);
    }

    /////////// get vbase attribute
    value = getIntValue(reader,"vbase", &ok);
    if ( ok ) 
    {
#if XML_PARSER_DEBUG
printf("      vbase = 0x%x\n", value);
#endif
        vseg[vseg_index]->vbase = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <vbase> attribute for vseg (%d,%d)\n", 
                vspace_index, vseg_loc_index);
        exit(1);
    }

    ////////// get ident attribute (optional : 0 if missing)
    value = getIntValue(reader,"ident", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      ident = %d\n", value);
#endif
        vseg[vseg_index]->ident = value;
    }  
    else
    {
        vseg[vseg_index]->ident = 0;
    }

    ////////// get psegname attribute
    str = getStringValue(reader,"psegname", &ok);
    if ( ok )
    {
        int index = getPsegId( str );
        if ( index >= 0 ) 
        {
#if XML_PARSER_DEBUG
printf("      psegname = %s\n", str);
printf("      psegid   = %d\n", index);
#endif
            vseg[vseg_index]->psegid = index;
        }
        else              
        {
            printf("[XML ERROR] illegal or missing <psegname> for vseg %d\n", 
                     vseg_loc_index);
            exit(1);
        }
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <psegname> for vseg %d\n", 
                 vseg_loc_index);
        exit(1);
    }

    //////// get mode attribute
    str = getStringValue(reader,"mode", &ok);
#if XML_PARSER_DEBUG
printf("      mode = %s\n", str);
#endif
    if      (ok && (strcmp(str, "CXWU") == 0)) vseg[vseg_index]->mode = 0xF;
    else if (ok && (strcmp(str, "CXW_") == 0)) vseg[vseg_index]->mode = 0xE;
    else if (ok && (strcmp(str, "CX_U") == 0)) vseg[vseg_index]->mode = 0xD;
    else if (ok && (strcmp(str, "CX__") == 0)) vseg[vseg_index]->mode = 0xC;
    else if (ok && (strcmp(str, "C_WU") == 0)) vseg[vseg_index]->mode = 0xB;
    else if (ok && (strcmp(str, "C_W_") == 0)) vseg[vseg_index]->mode = 0xA;
    else if (ok && (strcmp(str, "C__U") == 0)) vseg[vseg_index]->mode = 0x9;
    else if (ok && (strcmp(str, "C___") == 0)) vseg[vseg_index]->mode = 0x8;
    else if (ok && (strcmp(str, "_XWU") == 0)) vseg[vseg_index]->mode = 0x7;
    else if (ok && (strcmp(str, "_XW_") == 0)) vseg[vseg_index]->mode = 0x6;
    else if (ok && (strcmp(str, "_X_U") == 0)) vseg[vseg_index]->mode = 0x5;
    else if (ok && (strcmp(str, "_X__") == 0)) vseg[vseg_index]->mode = 0x4;
    else if (ok && (strcmp(str, "__WU") == 0)) vseg[vseg_index]->mode = 0x3;
    else if (ok && (strcmp(str, "__W_") == 0)) vseg[vseg_index]->mode = 0x2;
    else if (ok && (strcmp(str, "___U") == 0)) vseg[vseg_index]->mode = 0x1;
    else if (ok && (strcmp(str, "____") == 0)) vseg[vseg_index]->mode = 0x0;
    else
    {
        printf("[XML ERROR] illegal or missing <mode> attribute for vseg (%d,%d)\n", 
                vspace_index, vseg_loc_index);
        exit(1);
    }
    
    ////////// set the length attribute to 0
    vseg[vseg_index]->length = value;

    ////////// get vobjs
    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag,"vobj") == 0    ) vobjNode(reader);
        else if ( strcmp(tag,"#text"  ) == 0 ) { }
        else if ( strcmp(tag,"vseg") == 0  ) 
        {
            //  checking source file consistency?
            vseg[vseg_index]->vobjs = vobj_count;
            vseg_index++;
            vseg_loc_index++;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end vsegNode()

//////////////////////////////////////////
void vspaceNode( xmlTextReaderPtr reader )
{
    char*    		str;
    unsigned int 	ok;

    vobj_loc_index = 0;
	vseg_loc_index = 0;
	task_loc_index = 0;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    // checking source file consistency
    if ( vspace_index >= header->vspaces )
    {
        printf("[XML ERROR] The vspace index is too large : %d\n", 
                 vspace_index);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("\n  vspace %d\n", vspace_index);
#endif

    vspace[vspace_index] = (mapping_vspace_t*)malloc(sizeof(mapping_vspace_t));

    ////////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  name = %s\n", str);
#endif
        strncpy(vspace[vspace_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for vspace %d\n", 
                 vspace_index);
        exit(1);
    }

    ////////// set vseg_offset and task_offset attributes
    vspace[vspace_index]->vseg_offset = vseg_index;
    vspace[vspace_index]->vobj_offset = vobj_index;
    vspace[vspace_index]->task_offset = task_index;
    
#if XML_PARSER_DEBUG
printf("  vseg_offset = %d\n", vseg_index);
printf("  vobj_offset = %d\n", vobj_index);
printf("  task_offset = %d\n", task_index);
#endif

    ////////// get startname attribute
    str = getStringValue(reader, "startname", &ok);
    if ( ok )
    {
        //used after parsing the vobjs
    }
    else
    {
        printf("[XML ERROR] illegal or missing <startname> attribute for vspace %d\n", 
                 vspace_index);
        exit(1);
    }

    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag,"vseg") == 0    ) vsegNode(reader);
        else if ( strcmp(tag,"task") == 0    ) taskNode(reader);
        else if ( strcmp(tag,"#text"  ) == 0 ) { }
        else if ( strcmp(tag,"vspace") == 0  ) 
        {
            vspace[vspace_index]->vobjs = vobj_loc_index;
            vspace[vspace_index]->tasks = task_loc_index ;
            vspace[vspace_index]->vsegs = vseg_loc_index ;

            // get index of the vobj containing the start vector
            int index =  getVobjLocId( vspace_index, str );
            if(index == -1)
            {
                printf("[XML ERROR] vobj containing the start vector not found %s\n",str);
                exit(-1);
            }
            else
            {
                vspace[vspace_index]->start_offset = index;
#if XML_PARSER_DEBUG
printf("  startname = %s\n", str);
printf("  startid   = %d\n", index);
#endif
            }

            // checking startid values for all tasks in vspace
            int task_id;
            int task_min = vspace[vspace_index]->task_offset;
            int task_max = task_min + vspace[vspace_index]->tasks; 
            for ( task_id = task_min ; task_id < task_max ; task_id++ )
            {
                if ( task[task_id]->startid >= vspace[vspace_index]->tasks )
                {
                    printf("[XML ERROR] <startid> too large for task (%d,%d)\n", 
                           vspace_index, task_id );
                    exit(1);
                }
            }

            vspace_index++;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end vspaceNode()

//////////////////////////////////////////
void  psegNode ( xmlTextReaderPtr reader )
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( pseg_index >= MAX_PSEGS )
    {
        printf("[XML ERROR] The number of psegs is larger than %d\n", MAX_PSEGS);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("    pseg %d\n", pseg_index);
#endif

    pseg[pseg_index] = (mapping_pseg_t*)malloc(sizeof(mapping_pseg_t));

    // get name attribute
    str = getStringValue( reader, "name", &ok );
#if XML_PARSER_DEBUG
printf("    name = %s\n", str);
#endif
    if ( ok ) 
    {
        strncpy(pseg[pseg_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> for pseg %d\n", pseg_index);
        exit(1);
    }

    // get base attribute
    value = getIntValue( reader, "base", &ok );
#if XML_PARSER_DEBUG
printf("    base = %x\n", value);
#endif
    if ( ok ) 
    {
        pseg[pseg_index]->base = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <base> for pseg %d\n", pseg_index);
        exit(1);
    }

    // get length attribute
    value = getIntValue( reader, "length", &ok );
#if XML_PARSER_DEBUG
printf("    length = %x\n\n", value);
#endif
    if ( ok )
    {
        pseg[pseg_index]->length = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <length> for pseg %d\n", pseg_index);
        exit(1);
    }

    pseg_index++;
} // end psegNode()

/////////////////////////////////////////////
void  clusterNode ( xmlTextReaderPtr reader )
{
    unsigned int ok;
    unsigned int value;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    // checking source file consistency
    if ( cluster_index >= header->clusters )
    {
        printf("[XML ERROR] The cluster index is too large : %d\n", cluster_index);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("  cluster %d :\n", cluster_index);
#endif

    cluster[cluster_index] = (mapping_cluster_t*)malloc(sizeof(mapping_cluster_t));

    // get index attribute (optional)
    value = getIntValue(reader,"index",&ok);
#if XML_PARSER_DEBUG
printf("    index = %d\n", value);
#endif
    if ( ok && (value != cluster_index) )
    {
            printf("[XML ERROR] wrong cluster index / expected value is %d", 
                cluster_index);
            exit(1);
    }

    // get procs attribute
    value = getIntValue(reader,"procs",&ok);
#if XML_PARSER_DEBUG
printf("    procs = %d\n", value);
#endif
    if ( ok )
    {
        cluster[cluster_index]->procs = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <procs> attribute for cluster %d", 
                cluster_index);
        exit(1);
    }

    cluster_index++;
} // end clusterNode()

//////////////////////////////////////////////
void clusterSetNode( xmlTextReaderPtr reader )
{
    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("\n  clusters set\n");
#endif

    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag,"cluster") == 0 ) clusterNode(reader);
        else if ( strcmp(tag,"#text") == 0 ) { }
        else if ( strcmp(tag,"clusterset") == 0 ) 
        {
            // checking source file consistency
            if ( cluster_index != header->clusters )
            {
                printf("[XML ERROR] Wrong number of clusters\n");
                exit(1);
            }
            else
            {
                return;
            }
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in clusterset node : %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end clusterSetNode()

///////////////////////////////////////////
void psegSetNode( xmlTextReaderPtr reader )
{
    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("  psegs set\n");
#endif

    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag,"pseg") == 0    ) psegNode(reader);
        else if ( strcmp(tag,"#text") == 0 ) { }
        else if ( strcmp(tag,"psegset") == 0  ) 
        {
            // checking source file consistency
            if ( pseg_index != header->psegs )
            {
                printf("[XML ERROR] Wrong number of psegs\n");
                exit(1);
            }
            else
            {
                return;
            }
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in psegset node : %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end psegSetNode()

/////////////////////////////////////////////
void globalSetNode( xmlTextReaderPtr reader )
{
    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("  globals set\n");
#endif

    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag,"vseg") == 0    ) vsegNode(reader);
        else if ( strcmp(tag,"#text"  ) == 0 ) { }
        else if ( strcmp(tag,"globalset") == 0  ) 
        {
            // checking source file consistency
            if ( vseg_index != header->globals )
            {
                printf("[XML ERROR] Wrong number of global vsegs\n");
                exit(1);
            }
            else
            {
                vseg_loc_index = 0;
                return;
            }
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in globalset node : %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end globalSetNode()

/////////////////////////////////////////////
void vspaceSetNode( xmlTextReaderPtr reader )
{
    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("\n  vspaces set\n");
#endif

    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag,"vspace") == 0    ) vspaceNode(reader);
        else if ( strcmp(tag,"#text"  ) == 0 ) { }
        else if ( strcmp(tag,"vspaceset") == 0  ) 
        {
            // checking source file consistency
            if ( vspace_index != header->vspaces )
            {
                printf("[XML ERROR] Wrong number of vspaces\n");
                exit(1);
            }
            else
            {
                header->vsegs = vseg_index;
                header->vobjs = vobj_index;
                header->tasks = task_index;
                return;
            }
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in vspaceset node : %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end globalSetNode()

//////////////////////////////////////////
void headerNode(xmlTextReaderPtr reader )
{
    char*       	name;
    unsigned int 	value;
    unsigned int 	ok;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("mapping_info\n");
#endif

    header = (mapping_header_t*)malloc(sizeof(mapping_header_t));

    ////////// get name attribute
    name = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  name = %s\n", name);
#endif
	strncpy( header->name, name, 31);
    }      
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute in header\n");
        exit(1);
    }

    /////////// get clusters attribute
    value = getIntValue(reader, "clusters", &ok);
    if ( ok )
    {
        if ( value >= MAX_CLUSTERS )
        {
            printf("[XML ERROR] The number of clusters is larger than %d\n", MAX_CLUSTERS);
            exit(1);
        }
#if XML_PARSER_DEBUG
printf("  clusters = %d\n", value);
#endif
	header->clusters  = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <clusters> attribute in header\n");
        exit(1);
    }

    ////////// get psegs attribute
    value = getIntValue(reader, "psegs", &ok);
    if ( ok )
    {
        if ( value >= MAX_PSEGS )
        {
            printf("[XML ERROR] The number of psegs is larger than %d\n", MAX_PSEGS);
            exit(1);
        }
#if XML_PARSER_DEBUG
printf("  psegs = %d\n", value);
#endif
	header->psegs = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <psegs> attribute in header\n");
        exit(1);
    }

    ///////// get ttys attribute
    value = getIntValue(reader, "ttys", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  ttys = %d\n", value);
#endif
	header->ttys  = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <ttys> attribute in header\n");
        exit(1);
    }

    ///////// get fbs attribute
    value = getIntValue(reader, "fbs", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  fbs = %d\n", value);
#endif
	header->fbs  = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <fbs> attribute in header\n");
        exit(1);
    }

    ///////// get vspaces attribute
    value = getIntValue(reader, "vspaces", &ok);
    if ( ok )
    {
        if ( value >= MAX_VSPACES )
        {
            printf("[XML ERROR] The number of vspaces is larger than %d\n", MAX_VSPACES);
            exit(1);
        }
#if XML_PARSER_DEBUG
printf("  vspaces = %d\n", value);
#endif
	header->vspaces  = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <vspaces> attribute in mapping\n");
        exit(1);
    }

    ////////// get globals attribute
    value = getIntValue(reader, "globals", &ok);
    if ( ok )
    {
        if ( value >= MAX_VSEGS )
        {
            printf("[XML ERROR] The number of globals is larger than %d\n", MAX_VSEGS);
            exit(1);
        }
#if XML_PARSER_DEBUG
printf("  globals = %d\n", value);
#endif
	header->globals  = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <globals> attribute in mapping_info_header\n");
        exit(1);
    }

    header->signature = IN_MAPPING_SIGNATURE;

    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag,"clusterset")   == 0 ) clusterSetNode(reader);
        else if ( strcmp(tag,"psegset")      == 0 ) psegSetNode(reader);
        else if ( strcmp(tag,"globalset")   == 0 ) globalSetNode(reader);
        else if ( strcmp(tag,"vspaceset")    == 0 ) vspaceSetNode(reader);
        else if ( strcmp(tag,"#text")        == 0 ) { }
        else if ( strcmp(tag,"mapping_info") == 0 ) 
        {
#if XML_PARSER_DEBUG
printf("end mapping_info\n");
#endif
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in header node : %s\n",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end headerNode()

///////////////////////////
void  buildBin( int fdout )
{
    unsigned int    cluster_id;
    unsigned int    pseg_id;
    unsigned int    vspace_id;
    unsigned int    vseg_id;
    unsigned int    vobj_id;
    unsigned int    task_id;

    unsigned int    length;

    // write header to binary file
    length = write(fdout, (char*)header, sizeof(mapping_header_t));
    
#if XML_PARSER_DEBUG
printf("\n*** write header\n");
#endif

    if ( length != sizeof(mapping_header_t) )
    {
        perror("write header");
        exit(1);
    }

    // write clusters
    for ( cluster_id = 0 ; cluster_id < header->clusters ; cluster_id++ )
    {
        length = write(fdout, (char*)cluster[cluster_id], sizeof(mapping_cluster_t));
    
#if XML_PARSER_DEBUG
printf("\n*** write cluster %d\n", cluster_id);
#endif

        if ( length != sizeof(mapping_cluster_t) )
        {
            perror("write cluster");
            exit(1);
        }
    }

    // write psegs
    for ( pseg_id = 0 ; pseg_id < header->psegs ; pseg_id++ )
    {
        length = write(fdout, (char*)pseg[pseg_id], sizeof(mapping_pseg_t));
    
#if XML_PARSER_DEBUG
printf("write pseg %d\n", pseg_id);
#endif

        if ( length != sizeof(mapping_pseg_t) )
        {
            perror("write pseg");
            exit(1);
        }
    }

    // write vspaces
    for ( vspace_id = 0 ; vspace_id < header->vspaces ; vspace_id++ )
    {
        length = write(fdout, (char*)vspace[vspace_id], sizeof(mapping_vspace_t));
    
#if XML_PARSER_DEBUG
printf("write vspace %d\n", vspace_id);
printf("vspace->vobj_offset: %d\n", vspace[vspace_id]->vobj_offset);
printf("vspace->vobjs: %d\n", vspace[vspace_id]->vobjs);
printf("header->vobjs: %d\n", header->vobjs);
#endif

        if ( length != sizeof(mapping_vspace_t) )
        {
            perror("write vspace");
            exit(1);
        }
    }

    // write vsegs
    for ( vseg_id = 0 ; vseg_id < header->vsegs ; vseg_id++ )
    {
        length = write(fdout, (char*)vseg[vseg_id], sizeof(mapping_vseg_t));
    
#if XML_PARSER_DEBUG
printf("write vseg %d\n", vseg_id);
#endif

        if ( length != sizeof(mapping_vseg_t) )
        {
            perror("write vseg");
            exit(1);
        }
    }
    
    // write vobjs
    for ( vobj_id = 0 ; vobj_id < header->vobjs ; vobj_id++ )
    {
        length = write(fdout, (char*)vobj[vobj_id], sizeof(mapping_vobj_t));
    
#if XML_PARSER_DEBUG
printf("write vobj %d\n", vobj_id);
printf("write vobj name %s\n", vobj[vobj_id]->name);
printf("write vobj length %x\n", vobj[vobj_id]->length);
printf("write vobj type %d\n", vobj[vobj_id]->type);
#endif

        if ( length != sizeof(mapping_vobj_t) )
        {
            perror("write vobj");
            exit(1);
        }
    }

    // write tasks
    for ( task_id = 0 ; task_id < header->tasks ; task_id++ )
    {
        length = write(fdout, (char*)task[task_id], sizeof(mapping_task_t));
    
#if XML_PARSER_DEBUG
printf("write task %d\n", task_id);
#endif

        if ( length != sizeof(mapping_task_t) )
        {
            perror("write task");
            exit(1);
        }
    }
} // end buildBin()

/////////////////////////////////////
int  main ( int argc, char* argv[] )
{
    if ( argc < 3 ) 
    {
        printf("Usage: xml2bin <input_file_path> <output_file_path>\n");
        return 1;
    }

    int fdout = open( argv[2], (O_CREAT | O_RDWR), S_IRWXU );
    if ( fdout < 0)
    {
        perror("open");
        exit(1);
    }
    
    LIBXML_TEST_VERSION;

    int              status;
    xmlTextReaderPtr reader = xmlReaderForFile ( argv[1], NULL, 0 );

    if ( reader != NULL ) 
    {
        status = xmlTextReaderRead ( reader );
        while ( status == 1 ) 
        {
            const char* tag = (const char*)xmlTextReaderConstName(reader);
            if ( strcmp(tag,"mapping_info") == 0 )
            { 
                headerNode( reader );
                buildBin( fdout );
            }
            else
            {
                printf("[XML ERROR] Wrong file type: \"%s\"\n", argv[1]);
                return 1;
            }
            status = xmlTextReaderRead ( reader );
        }
        xmlFreeTextReader ( reader );

        if ( status != 0 ) 
        {
            printf("[XML ERROR] Wrong Syntax in \"%s\" file\n", argv[1]);
            return 1;
        }
    }
    return 0;
} // end main()
