////////////////////////////////////////////////////////////////////////////
// File     : mapping_info.h
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////
// The MAPPING_INFO data structure can be used with the GIET.
// It contains the mapping directive for one or several virtual spaces.
// Ech virtual space contains a variable number of virtual segments
// and a variable number of tasks. The number of virtual space can be one.
//
// The mapping table data structure is organised as the concatenation of
// a fixed size header, and 6 variable size arrays:
//
// - mapping_header_t   header              (MAPPING_HEADER_SIZE)
// - mapping_cluster_t  cluster[clusters]   (MAPPING_CLUSTER_SIZE * clusters)
// - mapping_pseg_t     pseg[psegs]         (MAPPING_PSEG_SIZE * psegs)
// - mapping_vspace_t   vspace[vspaces]     (MAPPING_VSPACE_SIZE * vspaces)
// - mapping_vseg_t     vseg[vsegs]         (MAPPING_VSEG_SIZE * vsegs)
// - mapping_vseg_t     vobj[vsegs]         (MAPPING_VOBJ_SIZE * vsegs)
// - mapping_task_t     task[tasks]         (MAPPING_TASK_SIZE * tasks)
//
// The number of clusters and the number of vspaces are defined in the header.
// The number of psegs are defined in each cluster.
// The number of vsegs, tasks ans mwmrs are defined in each vspace.
//
// It is intended to be stored in the boot ROM at address MAPPING_BOOT_BASE. 
// For each cluster, the base address of the first pseg descriptor
// is defined by a pseg_offset relative to MAPPING_BOOT_BASE.
// For each vspace, the base address of the first vseg descriptor
// is defined by a vseg_offset relative to MAPPING_BOOT_BASE.
// For each vspace, the base address of the first task desciptor
// is defined by a task_offset relative to MAPPING_BOOT_BASE.
// For each vspace, the base address of the first mwmr desciptor
// is defined by a mwmr_offset relative to MAPPING_BOOT_BASE.
////////////////////////////////////////////////////////////////////////////

#ifndef _MAPPING_INFO_H_
#define _MAPPING_INFO_H_

#define MAPPING_HEADER_SIZE     sizeof(mapping_header_t)
#define MAPPING_CLUSTER_SIZE    sizeof(mapping_cluster_t)
#define MAPPING_VSPACE_SIZE     sizeof(mapping_vspace_t)
#define MAPPING_VSEG_SIZE	    sizeof(mapping_vseg_t)
#define MAPPING_VOBJ_SIZE	    sizeof(mapping_vobj_t)
#define MAPPING_PSEG_SIZE	    sizeof(mapping_pseg_t)
#define MAPPING_TASK_SIZE	    sizeof(mapping_task_t)
#define MAPPING_PROC_SIZE	    sizeof(mapping_proc_t)
#define MAPPING_IRQ_SIZE	    sizeof(mapping_irq_t)
#define MAPPING_COPROC_SIZE	    sizeof(mapping_coproc_t)
#define MAPPING_CP_PORT_SIZE    sizeof(mapping_coproc_port_t)
#define MAPPING_CP_REG_SIZE	    sizeof(mapping_coproc_reg_t)

#define C_MODE_MASK     0b1000      // cacheable
#define X_MODE_MASK     0b0100      // executable
#define W_MODE_MASK     0b0010      // writable
#define U_MODE_MASK     0b0001      // user access

#define IN_MAPPING_SIGNATURE    0xDEADBEEF
#define OUT_MAPPING_SIGNATURE   0xBABEF00D

enum
{
    VOBJ_TYPE_ELF     = 0,     // loadable code/data object of elf files
    VOBJ_TYPE_BLOB    = 1,     // loadable blob object
    VOBJ_TYPE_PTAB    = 2,     // page table 
    VOBJ_TYPE_PERI    = 3,     // hardware component
    VOBJ_TYPE_MWMR    = 4,     // MWMR channel
    VOBJ_TYPE_LOCK    = 5,     // Lock
    VOBJ_TYPE_BUFFER  = 6,     // Any "no intialiasation needed" objects (stacks...)
    VOBJ_TYPE_BARRIER = 7,     // Barrier
};

enum
{
    PSEG_TYPE_RAM     = 0,
    PSEG_TYPE_ROM     = 1,
    PSEG_TYPE_PERI    = 2,
};

enum
{
    IRQ_TYPE_HARD     = 0,          // hardware interrupt (peripheral)
    IRQ_TYPE_SOFT     = 1,          // software interrupt (IPI)
};

enum
{
    ISR_SWITCH        = 0,
    ISR_IOC           = 1,
    ISR_FBDMA         = 2,
    ISR_TTY           = 3,
};

enum
{
    REG_TYPE_STATUS   = 0,          // status register
    REG_TYPE_CONFIG   = 1,          // config register
};

enum
{
    PORT_TO_COPROC    = 0,          // status register
    PORT_FROM_COPROC  = 1,          // config register
};

///////////////////////////////
typedef struct mapping_header_s
{
    unsigned int    signature;      // must contain MAPPING_SIGNATURE
	unsigned int	clusters;	    // number of clusters
    unsigned int    ttys;           // number of TTY terminals 
    unsigned int    fbs;            // number of FBDMA channels
	unsigned int	globals;		// number of vsegs mapped in all vspaces
	unsigned int	vspaces;		// number of virtual spaces
    unsigned int    psegs;          // total number of physical segments (for all clusters)
	unsigned int	vsegs;  		// total number of virtual segments (for all vspaces)
	unsigned int	vobjs;  		// total number of virtual objects (for all vspaces)
	unsigned int	tasks;  		// total number of tasks (for all vspaces)
	unsigned int	procs;  		// total number of procs
	unsigned int	irqs;  	    	// total number of irqs
	unsigned int	coprocs;  		// total number of coprocs
	unsigned int	cp_ports;  		// total number of cp_ports
	unsigned int	cp_regs;  		// total number of cp_regss
    char            name[32];       // mapping name
} mapping_header_t;

////////////////////////////////
typedef struct mapping_cluster_s
{
    unsigned int    procs;          // number of processors in cluster
    unsigned int    proc_offset;    // index of first proc in proc set
    unsigned int    coprocs;        // number of coprocessors in cluster
    unsigned int    coproc_offset;  // index of first coproc in coproc set
    unsigned int    psegs;          // number of psegs in cluster
    unsigned int    pseg_offset;    // index of first pseg in pseg set
} mapping_cluster_t;

/////////////////////////////
typedef struct mapping_pseg_s 
{
    char            name[32];       // pseg name (unique in a cluster)
	unsigned int    base;           // base address in physical space
	unsigned int	length;         // size (bytes)
    unsigned int    type;           // RAM / ROM / PERI
    unsigned int    cluster;        // index of cluster containing pseg
    unsigned int    next_base;      // first free page base address
} mapping_pseg_t;

///////////////////////////////
typedef struct mapping_vspace_s
{
    char            name[32];       // virtual space name
    unsigned int    start_offset;   // offset of the vobj containing the start vector
	unsigned int	vsegs;		    // number of vsegs in vspace
	unsigned int	vobjs;		    // number of vobjs in vspace
	unsigned int	tasks;		    // number of tasks in vspace
    unsigned int    vseg_offset;    // index of first vseg in vspace 
    unsigned int    vobj_offset;    // index of first vobjs in vspace
    unsigned int    task_offset;    // index of first task in vspace
} mapping_vspace_t;

/////////////////////////////
typedef struct mapping_vseg_s 
{
	char            name[32];       // vseg name (unique in vspace)
	unsigned int    vbase;          // base address in virtual space (hexa)
	unsigned int    pbase;          // base address in physical space (hexa)
	unsigned int	length;         // size (bytes)
	unsigned int    psegid;	        // physical segment global index
	unsigned int    mode;	        // C-X-W-U flags
    unsigned int    ident;          // identity mapping if non zero
	unsigned int	vobjs;		    // number of vobjs in vseg
    unsigned int    vobj_offset;    // index of first vobjs in vseg
} mapping_vseg_t;

/////////////////////////////
typedef struct mapping_task_s 
{
	char            name[32];       // task name (unique in vspace)
	unsigned int	clusterid;	    // physical cluster index
	unsigned int	proclocid;      // processor local index (inside cluster)
    unsigned int    vobjlocid;      // stack vobj index in vspace
    unsigned int    startid;        // index in start_vector 
    unsigned int    use_tty;        // TTY terminal required
    unsigned int    use_fb;         // DMA channel to frame buffer required
} mapping_task_t;

/////////////////////////////
typedef struct mapping_vobj_s 
{
    char            name[32];       // vobj name (unique in a vspace)
    char            binpath[64];    // path for the binary code ("*.elf")
	unsigned int    type;           // type of vobj
	unsigned int	length;         // size (bytes)
	unsigned int	align;          // required alignement (logarithm of 2)
	unsigned int	vaddr;          // virtual base addresse of the vobj
	unsigned int	paddr;          // physical base addresse of the vobj
	unsigned int	init;           // init value (used by barrier or mwmr channel)
} mapping_vobj_t;

/////////////////////////////
typedef struct mapping_proc_s
{
    unsigned int    irqs;           // number of IRQs allocated to processor
    unsigned int    irq_offset;     // index of first IRQ allocated to processor
} mapping_proc_t;

/////////////////////////////
typedef struct mapping_irq_s
{
    unsigned int    type;           // HW_IRQ  / SW_IRQ
    unsigned int    icuid;          // IRQ Index for the ICU component
    unsigned int    isr;            // Interrupt Service Routine Index 
    unsigned int    channel;        // Channel Index (for multi-cannels peripherals)

    unsigned int    cluster_id;     // physical cluster index
	unsigned int	proclocid;      // processor local index (inside cluster)
} mapping_irq_t;

///////////////////////////////
typedef struct mapping_coproc_s
{
    char            name[32];       // coprocessor name
    unsigned int    psegid;         // pseg index in cluster
    unsigned int    ports;          // number of MWMR ports used by coprocessor
    unsigned int    port_offset;    // index of first MWMR port used by coprocessor
    unsigned int    regs;           // number of config/status registers 
    unsigned int    reg_offset;     // index of first register 
} mapping_coproc_t;

///////////////////////////////////////
typedef struct mapping_coproc_port_s
{
    unsigned int    direction;      // TO_COPROC == 0 / FROM_COPROC == 1
    unsigned int    vspaceid;      // global index of the vspace containing the MWMR channel
    unsigned int    vobjlocid;        // local(to vspace) index of the vobj containing the MWMR channel
} mapping_coproc_port_t;

///////////////////////////////////
typedef struct mapping_coproc_reg_s
{
    char            name[32];       // register name
    unsigned int    type;           // STATUS = 0 / CONFIG == 1
    unsigned int    loc_id;         // register index in coprocessor (word offset)
    unsigned int    channel_id;     // channel index in coprocessor (page offset)
    unsigned int    value;          // initialisation value
}  mapping_coproc_reg_t;

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

