///////////////////////////////////////////////////////////////////////////////////////
// File     : ctx_handler.c
// Date     : 01/04/2012
// Authors  : alain greiner & joel porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////////
// The ctx_handler.h and ctx_handler.c files are part of the GIET-VM nano-kernel.
// This code is used to support context switch when several tasks are executing
// in time multiplexing on a single processor.
// The tasks must be statically allocated to a processor in the boot phase, and
// there is one private scheduler per processor.
// Each sheduler contains up to 15 task contexts.
///////////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <drivers.h>
#include <common.h>
#include <ctx_handler.h>
#include <mapping_info.h>
#include <sys_handler.h>

///////////////////////////////////////////////////////////////////////////////////////
// A task context is an array of 64 words = 256 bytes. 
// It contains copies of processor registers (when the task is preempted),
// and some general informations associated to the task.
//
// - It contains GPR[i], generally stored in slot (i). $0, *26 & $27 are not saved.
// - It contains HI & LO registers.
// - It contains CP0 registers: EPC, SR, CR, SCHED
// - It contains CP2 registers : PTPR and MODE.
// - It contains TTY global index, the FBDMA global index, the virtual base 
//   address of the page table (PTAB), and the task global index (TASK).
//
// ctx[0]<- SR|ctx[8] <- $8 |ctx[16]<- $16|ctx[24]<- $24|ctx[32]<- EPC |ctx[40]<- TTY
// ctx[1]<- $1|ctx[9] <- $9 |ctx[17]<- $17|ctx[25]<- $25|ctx[33]<- CR  |ctx[41]<- FBDMA
// ctx[2]<- $2|ctx[10]<- $10|ctx[18]<- $18|ctx[26]<- LO |ctx[34]<- *** |ctx[42]<- PTAB
// ctx[3]<- $3|ctx[11]<- $11|ctx[19]<- $19|ctx[27]<- HI |ctx[35]<- PTPR|ctx[43]<- TASK
// ctx[4]<- $4|ctx[12]<- $12|ctx[20]<- $20|ctx[28]<- $28|ctx[36]<- MODE|ctx[44]<- SCHED
// ctx[5]<- $5|ctx[13]<- $13|ctx[21]<- $21|ctx[29]<- SP |ctx[37]<- *** |ctx[45]<- TIMER
// ctx[6]<- $6|ctx[14]<- $14|ctx[22]<- $22|ctx[30]<- $30|ctx[38]<- *** |ctx[46]<- ***
// ctx[7]<- $7|ctx[15]<- $15|ctx[23]<- $23|ctx[31]<- RA |ctx[39]<- *** |ctx[47]<- ***
////////////////////////////////////////////////////////////////////////////////////////

extern void _task_switch(unsigned int*, unsigned int*);

/////////////////////////////////////////////////////////////////////////////////
//	_ctx_switch()
// This function performs a context switch between the running task
// and  another task, using a round-robin sheduling policy.
//
// It desactivate the DTLB, to directly access the scheduler using 
// the physical address stored in register CP0_SCHED.
// All the context switch procedure is executed with interrupts masked.
//
// The return address contained in $31 is saved in the current task context
// (in the ctx[31] slot), and the function actually returns to the address
// contained in the ctx[31] slot of the next task context.
/////////////////////////////////////////////////////////////////////////////////
void _ctx_switch()
{
    unsigned int			tasks;
    unsigned int 			curr_task_id;
    unsigned int 			next_task_id;
    unsigned int			*curr_ctx_paddr;
    unsigned int			*next_ctx_paddr;

    // get scheduler physical address
    static_scheduler_t*	psched = (static_scheduler_t*)_get_sched();

    // get number of tasks allocated to scheduler
    tasks = _get_tasks_number();

    // no switch if only one task
    if ( tasks > 1) 
    {
        // compute the context physical address for the current task
        curr_task_id = _get_current_task_id();
        curr_ctx_paddr = &(psched->context[curr_task_id][0]);
    
        // select the next task using a round-robin policy
        next_task_id = (curr_task_id + 1) % tasks;
    
        // compute the context physical address for the next task
        next_ctx_paddr = &(psched->context[next_task_id][0]);

        // update the scheduler state
        _set_current_task_id( next_task_id );

#if GIET_DEBUG_SWITCH
_get_lock( &_tty_put_lock );
_puts( "\n[GIET] Context switch for processor ");
_putw( _procid() );
_puts( " at cycle ");
_putw( _proctime() );
_puts("\n");
_puts( " - tasks        = ");
_putw( tasks );
_puts("\n");
_puts( " - curr_task_id = ");
_putw( curr_task_id );
_puts("\n");
_puts( " - next_task_id = ");
_putw( next_task_id );
_puts("\n");
_release_lock( &_tty_put_lock );
#endif

        // makes the task switch
        _task_switch( curr_ctx_paddr, next_ctx_paddr );

/*
        asm volatile( "ori		$27,	$0,		0xB	\n"
                      "mtc2		$27,	$1			\n"		// desactivate DTLB 

                      "add      $27,    %0,     $0  \n"		// $27 <= &curr_ctx 

                      "mfc0     $26,    $12         \n"		// $26 <= SR 
                      "sw       $26,    0*4($27)    \n"		// ctx[0] <= SR 
                      ".set noat                    \n"
                      "sw       $1,     1*4($27)    \n"		// ctx[1] <= $1 
                      ".set at                      \n"
                      "sw       $2,     2*4($27)    \n"		// ctx[2] <= $2 
                      "sw       $3,     3*4($27)    \n"		// ctx[3] <= $3 
                      "sw       $4,     4*4($27)    \n"		// ctx[4] <= $4 
                      "sw       $5,     5*4($27)    \n"		// ctx[5] <= $5 
                      "sw       $6,     6*4($27)    \n"		// ctx[6] <= $6 
                      "sw       $7,     7*4($27)    \n"		// ctx[7] <= $7 
                      "sw       $8,     8*4($27)    \n"		// ctx[8] <= $8 
                      "sw       $9,     9*4($27)    \n"		// ctx[9] <= $9 
                      "sw       $10,    10*4($27)   \n"		// ctx[10] <= $10 
                      "sw       $11,    11*4($27)   \n"		// ctx[11] <= $11 
                      "sw       $12,    12*4($27)   \n"		// ctx[12] <= $12 
                      "sw       $13,    13*4($27)   \n"		// ctx[13] <= $13 
                      "sw       $14,    14*4($27)   \n"		// ctx[14] <= $14 
                      "sw       $15,    15*4($27)   \n"		// ctx[15] <= $15 
                      "sw       $16,    16*4($27)   \n"		// ctx[16] <= $16 
                      "sw       $17,    17*4($27)   \n"		// ctx[17] <= $17 
                      "sw       $18,    18*4($27)   \n"		// ctx[18] <= $18 
                      "sw       $19,    19*4($27)   \n"		// ctx[19] <= $19 
                      "sw       $20,    20*4($27)   \n"		// ctx[20] <= $20 
                      "sw       $21,    21*4($27)   \n"		// ctx[21] <= $21 
                      "sw       $22,    22*4($27)   \n"		// ctx[22] <= $22 
                      "sw       $23,    23*4($27)   \n"		// ctx[23] <= $23 
                      "sw       $24,    24*4($27)   \n"		// ctx[24] <= $24 
                      "sw       $25,    25*4($27)   \n"		// ctx[25] <= $25 
                      "mflo     $26                 \n"
                      "sw       $26,    26*4($27)   \n"		// ctx[26] <= LO 
                      "mfhi     $26                 \n"
                      "sw       $26,    27*4($27)   \n"		// ctx[27] <= H1 
                      "sw       $28,    28*4($27)   \n"		// ctx[28] <= $28 
                      "sw       $29,    29*4($27)   \n"		// ctx[29] <= $29 
                      "sw       $30,    30*4($27)   \n"		// ctx[30] <= $30 
                      "sw       $31,    31*4($27)   \n"		// ctx[31] <= $31 
                      "mfc0     $26,    $14         \n"
                      "sw       $26,    32*4($27)   \n"		// ctx[32] <= EPC 
                      "mfc0     $26,    $13         \n"
                      "sw       $26,    33*4($27)   \n"		// ctx[33] <= CR 
                      "mfc2     $26,    $0          \n"
                      "sw       $26,    35*4($27)   \n"		// ctx[35] <= PTPR 

                      "add      $27,    %1,     $0	\n"		// $27<= &next_ctx

                      "lw       $26,    35*4($27)	\n"
                      "mtc2     $26,    $0          \n"		// restore PTPR 
                      "lw       $26,    0*4($27)	\n"
                      "mtc0     $26,    $12			\n"		// restore SR
                      ".set noat                    \n"
                      "lw       $1,     1*4($27)	\n"		// restore $1
                      ".set at                      \n"
	                  "lw       $2,     2*4($27)    \n"		// restore $2 
                      "lw       $3,     3*4($27)    \n"		// restore $3
                      "lw       $4,     4*4($27)    \n"		// restore $4
                      "lw       $5,     5*4($27)    \n"		// restore $5
                      "lw       $6,     6*4($27)    \n"		// restore $6
                      "lw       $7,     7*4($27)    \n"		// restore $7
                      "lw       $8,     8*4($27)    \n"		// restore $8
                      "lw       $9,     9*4($27)    \n"		// restore $9
                      "lw       $10,    10*4($27)   \n"		// restore $10
                      "lw       $11,    11*4($27)   \n"		// restore $11
                      "lw       $12,    12*4($27)   \n"		// restore $12
                      "lw       $13,    13*4($27)   \n"		// restore $13
                      "lw       $14,    14*4($27)   \n"		// restore $14
                      "lw       $15,    15*4($27)   \n"		// restore $15
                      "lw       $16,    16*4($27)   \n"		// restore $16
                      "lw       $17,    17*4($27)   \n"		// restore $17 
                      "lw       $18,    18*4($27)   \n"		// restore $18 
                      "lw       $19,    19*4($27)   \n"		// restore $19 
                      "lw       $20,    20*4($27)   \n"		// restore $20 
                      "lw       $21,    21*4($27)   \n"		// restore $21 
                      "lw       $22,    22*4($27)   \n"		// restore $22 
                      "lw       $23,    23*4($27)   \n"		// restore $23 
                      "lw       $24,    24*4($27)   \n"		// restore $24 
                      "lw       $25,    25*4($27)   \n"		// restore $25 
                      "lw       $26,    26*4($27)   \n"
                      "mtlo     $26                 \n"		// restore LO 
                      "lw       $26,    27*4($27)   \n"
                      "mthi     $26                 \n"		// restore HI 
                      "lw       $28,    28*4($27)   \n"		// restore $28 
                      "lw       $29,    29*4($27)   \n"		// restore $29 
                      "lw       $30,    30*4($27)   \n"		// restore $30 
                      "lw       $31,    31*4($27)   \n"		// restore $31 
                      "lw       $26,    32*4($27)   \n"
                      "mtc0     $26,    $14         \n"		// restore EPC 
                      "lw       $26,    33*4($27)   \n"
                      "mtc0     $26,    $13         \n"		// restore CR 

                      "ori		$27,	$0,		0xF \n"
                      "mtc2		$27,	$1          \n"		// activate DTLB
                      :
                      : "r"(curr_ctx_paddr), "r"(next_ctx_paddr)
                      : "$1" ,            "$4" ,"$5" ,"$6" ,"$7" ,"$8" ,"$9" ,"$10",
                        "$11","$12","$13","$14","$15","$16","$17","$18","$19","$20",
                        "$21","$22","$23","$24","$25","$26","$27",      "$29",
                        "$31" );
*/
    }
} // end _ctx_switch

