/********************************************************************************/
/*	File : reset.S                                                              */
/*	Author : Alain Greiner	& Mohamed karaoui                                   */
/*	Date : 03/06/2012							                                */
/********************************************************************************/
/* This boot code is for a multi-cluster, multi-processor architecture,		    */
/* running one or several multi-tasks software application(s) defined in    	*/
/* the mapping_info data-structure.						                        */
/* Procesor 0 uses the mapping_info data structure to statically initialize		*/
/* the kernel structures:														*/
/* - build page tables for all vspaces											*/
/* - initializes the vobjs not initialized by GCC								*/
/* - initialize the schedulers and task contexts for all processeurs			*/
/* - initialize the peripherals													*/				
/* Other processors are waiting until the mapping_info signature has been       */
/* modified by processor 0.														*/
/*																				*/
/* Implementation note:															*/
/* The entry point is 0xbfc00000, but the actual boot code starts at address	*/
/* 0xbfc00500, and a minimal boot exception handler is implemented at address	*/
/* 0xbfc0380.									                                */
/********************************************************************************/
		
#include <giet_config.h>
#include "../sys/mips32_registers.h"

#define EXCEP_ORG    		0x380
#define START_ORG    		0x500

#define OUT_MAPPING_SIGNATURE	0xBABEF00D

	.section .boot,"ax",@progbits
	.align	2
   	.set noreorder

/********************************************************/
/* reset entry point                                    */
/* (address 0xBFC00000 imposed by the hardware)         */
/********************************************************/
boot_reset:
    j	boot_start
	nop

/*******************************************************/
/* handling exceptions in the boot phase               */
/* (address 0xBFC00380 imposed by the hardware         */
/*******************************************************/
	.align	2
    .org	EXCEP_ORG

boot_excep:

    # get the lock protecting TTY0 
    la		k0, boot_tty0_lock
    ll		k1, 0(k0)
    bnez	k1, boot_excep
    li		k1, 1
    sc      k1, 0(k0)
    beqz	k1, boot_excep
    nop

    # display error messages on TTY0  
    la		a0,	boot_error_string
    jal		boot_puts
    nop
    mfc0	a0,	CP0_TIME
	jal		boot_putd
	nop
    la		a0,	boot_lf_string
    jal		boot_puts
    nop

	la		a0,	boot_pid_string
    jal		boot_puts
    nop
    mfc0	k0,	CP0_PROCID
    andi	a0,	k0,	0xFFF
	jal		boot_putd
    nop
    la		a0,	boot_lf_string
    jal		boot_puts
    nop

	la		a0,	boot_epc_string
    jal		boot_puts
    nop
    mfc0	a0,	CP0_EPC
	jal		boot_putx
    nop
    la		a0,	boot_lf_string
    jal		boot_puts
    nop

	la		a0,	boot_cr_string
    jal		boot_puts
    nop
    mfc0	a0,	CP0_CR
	jal		boot_putx
    nop
    la		a0,	boot_lf_string
    jal		boot_puts
    nop

	la		a0,	boot_sr_string
    jal		boot_puts
    nop
    mfc0	a0,	CP0_SR
	jal		boot_putx
    nop
    la		a0,	boot_lf_string
    jal		boot_puts
    nop

	la		a0,	boot_bar_string
    jal		boot_puts
    nop
    mfc0	a0,	CP0_BVAR
	jal		boot_putx
    nop
    la		a0,	boot_lf_string
    jal		boot_puts
    nop

    # release the lock  
    la		k0,	boot_tty0_lock
    li		k1,	0
    sw		k1, 0(k0)

    # exit 
	j	boot_exit
	nop

/*******************************************/
/* actual starting point for the boot code */
/*******************************************/
    .align 	2
    .org	START_ORG

boot_start:

    # get the procid 
    mfc0	k0,	CP0_PROCID
    andi	k0,	k0,	0xFFF	# no more than 4096 processors

    # Only processor 0 does init 
    bne		k0,	zero,	boot_wait_signature
    nop

    # Processor 0 get a temporary stack 
    la	    sp,	seg_boot_stack_base
    addiu	sp,	sp,	0x3000				# SP <= seg_boot_stack + 12K 

    # Processor 0 initialises all kernel structures 
    jal	boot_init
    nop
 
    # jump to kernel_init 
    j		boot_to_kernel_init
    nop

boot_wait_signature:

    # all other processors are waiting signature change 
    la		k0, seg_mapping_base
    cache   0x11,   0(k0)				# invalidate local cache copy 
    lw		k0, 0(k0)					# k0 <= mapping_info[0]       
    li		k1, OUT_MAPPING_SIGNATURE
    bne		k1, k0, boot_wait_signature
    nop
	
    # all other processors initialise SP register: temporary stack of 256 bytes 
    la      sp, seg_boot_stack_base
    addiu   sp, sp, 0x3100				
    mfc0	k0, CP0_PROCID
    andi    k0, k0, 0xFFF
    sll     k0, k0, 8					
    addu    sp, sp, k0		# SP <= seg_boot_stack_base + 12K + (pid+1)*256 

boot_to_kernel_init:

    # all processors initialise SCHED register with boot_schedulers_paddr[pid] 
    mfc0	k0, CP0_PROCID
    andi    k0, k0, 0xFFF
    sll		k0, k0, 2		# k0 <= 4*pid 
    la		k1, boot_schedulers_paddr
    addu    k1, k1, k0		# k1 <= &boot_scheduler_paddr[pid] 
    lw		k1,	0(k1)		
    mtc0	k1, CP0_SCHED

    # all processors initialize PTPR register with boot_ptabs_paddr[0] 
    la	    k1,	boot_ptabs_paddr
    lw	    k1,	0(k1)	
    srl     k1, k1, 13
    mtc2	k1,	CP2_PTPR 	

	# all processors activate MMU 
    li      k1,	0xF
    mtc2    k1,	CP2_MODE 

    # all processors jump to kernel_init 
    la	    k0,	seg_kernel_init_base
    j       k0
    nop

boot_error_string:	.asciiz "\n[BOOT] Fatal Error at cycle "
boot_pid_string:	.asciiz "    PID  = "
boot_sr_string:		.asciiz "    SR   = "
boot_cr_string:		.asciiz "    CR   = "
boot_epc_string:	.asciiz "    EPC  = "
boot_bar_string:	.asciiz "    BAR  = "
boot_lf_string:		.asciiz "\n"

boot_tty0_lock:		.word	0

	.set	reorder


