///////////////////////////////////////////////////////////////////////////////////////
// File     : xml_parser.c
// Date     : 14/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////////
// This program translate a "map.xml" source file to a binary file "map.bin" that 
// can be directly loaded in the boot ROM and used by the GIET-VM operating system.
//
// This map.xml file contains :
// 1) the multi-cluster/multi-processors hardware architecture description
// 2) the various multi-threaded software applications
// 3) the mapping directives bor both the tasks and the virtual segments.
// The corresponding C structures are defined in the "mapping_info.h" file.
// 
// This program also generates the "hard_config.h" and the "giet_vsegs.ld" files,
// required  to compile the GIET-VM code. 
///////////////////////////////////////////////////////////////////////////////////////

#include  <stdlib.h>
#include  <fcntl.h>
#include  <sys/types.h>
#include  <sys/stat.h>
#include  <unistd.h>
#include  <stdio.h>
#include  <string.h>
#include  <assert.h>
#include  <libxml/xmlreader.h>
#include  <mapping_info.h>
#include  <irq_handler.h>

#define MAX_CLUSTERS	1024
#define MAX_PSEGS		4096
#define MAX_VSPACES		1024
#define MAX_TASKS		4096
#define MAX_MWMRS		4096
#define MAX_VSEGS		4096
#define MAX_VOBJS		8192
#define MAX_PROCS		1024
#define MAX_IRQS		8192
#define MAX_COPROCS		4096
#define MAX_CP_PORTS	8192
#define MAX_PERIPHS		8192

#define XML_PARSER_DEBUG   0

///////////////////////////////////////////////////////////////////////////////////
// 	global variables used to store and index the data structures
///////////////////////////////////////////////////////////////////////////////////

mapping_header_t*       header;
mapping_cluster_t*      cluster[MAX_CLUSTERS];	// cluster array
mapping_pseg_t*	        pseg[MAX_PSEGS];		// pseg array
mapping_vspace_t*       vspace[MAX_VSPACES];	// vspace array
mapping_vseg_t*	        vseg[MAX_VSEGS];		// vseg array
mapping_vobj_t*	        vobj[MAX_VOBJS];		// vobj array
mapping_task_t*	        task[MAX_TASKS];		// task array
mapping_proc_t*	        proc[MAX_PROCS];		// proc array
mapping_irq_t*	        irq[MAX_IRQS];			// irq array
mapping_coproc_t*       coproc[MAX_COPROCS];	// coproc array
mapping_cp_port_t*	    cp_port[MAX_CP_PORTS];	// coproc port array
mapping_periph_t*       periph[MAX_PERIPHS];    // peripheral array

// Index for the various arrays

unsigned int		cluster_index  = 0;
unsigned int		vspace_index   = 0;
unsigned int		global_index   = 0;
unsigned int		pseg_index     = 0;		

unsigned int		proc_index     = 0;
unsigned int		proc_loc_index = 0;

unsigned int		irq_index      = 0;
unsigned int		irq_loc_index  = 0;

unsigned int		coproc_index     = 0;
unsigned int		coproc_loc_index = 0;

unsigned int		cp_port_index     = 0;
unsigned int		cp_port_loc_index = 0;

unsigned int		periph_index     = 0;
unsigned int		periph_loc_index = 0;

unsigned int		vseg_index     = 0;
unsigned int		vseg_loc_index = 0;

unsigned int		task_index     = 0;
unsigned int		task_loc_index = 0;

unsigned int		vobj_index     = 0;
unsigned int		vobj_loc_index = 0;
unsigned int		vobj_count     = 0;


/////////////////////////
// found peripheral
/////////////////////////
char found_timer = 0;
char found_icu = 0;
char found_xicu = 0;
char found_dma = 0;


//////////////////////////////////
//needed to generate map_config.ld
//////////////////////////////////
unsigned int cluster_y              = 0;
unsigned int cluster_x              = 0; 
unsigned int nb_proc_max            = 0; // max number of processors per cluster 
unsigned int nb_timer_channel_max   = 0; // max number of user timer
unsigned int nb_dma_channel_max     = 0;
unsigned int nb_tty_channel         = 0;
unsigned int nb_ioc_channel         = 0;
unsigned int nb_nic_channel         = 0;
unsigned int io_mmu_active          = 0;
unsigned int use_xicu               = 0xFFFFFFFF;


//////////////////////////////////
//needed to generate giet_vseg.ld
//////////////////////////////////

//kernel and boot code
unsigned int kernel_code_base    = 0x80000000;	/* kernel code */
unsigned int kernel_data_base    = 0x80010000;	/* system cacheable data */
unsigned int kernel_uncdata_base = 0x80080000;	/* system uncacheable data */
unsigned int kernel_init_base    = 0x80090000;	/* system init entry */ 

unsigned int boot_code_base      = 0xBFC00000;   /* boot code */
unsigned int boot_stack_base     = 0xBFC08000;	/* boot temporary stack */ 
unsigned int boot_mapping_base   = 0xBFC0C000;	/* mapping_info blob */

//periphs
unsigned int tim_base_offset = 0xFFFFFFFF;
unsigned int tty_base_offset = 0xFFFFFFFF;
unsigned int dma_base_offset = 0xFFFFFFFF;
unsigned int ioc_base_offset = 0xFFFFFFFF;
unsigned int nic_base_offset = 0xFFFFFFFF;
unsigned int fbf_base_offset = 0xFFFFFFFF;
unsigned int icu_base_offset = 0xFFFFFFFF;
unsigned int gcd_base_offset = 0xFFFFFFFF;
unsigned int iob_base_offset = 0xFFFFFFFF;


//////////////////////////////////////////////////////////////////////
// This arrray is useful to build a temporary list of vobj references. 
// The struct vobj_ref_s is formed by a vspace_name and a vobj_name.
// This array is used to set the attribute vobj_id of a cp_port
// once all the vspace have been parsed.
/////////////////////////////////////////////////////////////////////
typedef struct vobj_ref_s
{
    char vspace_name[32];
    char vobj_name[32];
}vobj_ref_t;

vobj_ref_t*	        cp_port_vobj_ref[MAX_CP_PORTS];	


//////////////////////////////////////////////////
unsigned int getIntValue( xmlTextReaderPtr reader, 
                          const char*      attributeName, 
                          unsigned int*    ok )
{
    unsigned int 	value = 0;
    unsigned int 	i;
    char	c;

    char* string = (char*)xmlTextReaderGetAttribute(reader, (const xmlChar*)attributeName);

    if ( string == NULL )  // missing argument
    {
        *ok = 0;
        return 0;
    }
    else
    {
        if ( (string[0] == '0') && ((string[1] == 'x') || (string[1] == 'X')) )  // Hexa
        {
            for ( i = 2 ; (string[i] != 0) && (i < 10) ; i++ )
            {
                c = string[i];
                if      ((c >= '0') && (c <= '9')) value = (value<<4) + string[i] - 48;
                else if ((c >= 'a') && (c <= 'f')) value = (value<<4) + string[i] - 87;
                else if ((c >= 'A') && (c <= 'F')) value = (value<<4) + string[i] - 55;
                else    
                {
                    *ok = 0;
                    return 0;
                }
            }
        }
        else 								// Decimal
        {
            for ( i = 0 ; (string[i] != 0) && (i < 9) ; i++ )
            {
                c = string[i];
                if ((c >= '0') && (c <= '9')) value = (value*10) + string[i] - 48;
                else 
                {
                    *ok = 0;
                    return 0;
                }
            }
        }
        *ok = 1;
        return value; 
    }
} // end getIntValue()

///////////////////////////////////////////////
char* getStringValue ( xmlTextReaderPtr reader, 
                       const char* 	attributeName, 
                       unsigned int* 	ok )
{
    char* string = (char*)xmlTextReaderGetAttribute(reader, (const xmlChar*)attributeName);

    
    if ( string == NULL )  // missing argument
    {
        *ok = 0;
        return NULL;
    }
    else
    {
        //we read only string smaller than 32 byte
        if(strlen(string) > 32)
        {
            printf("[XML ERROR] all strings must be less than 32 bytes\n");
            exit(1);
        }

        *ok = 1;
        return string;
    }
} // end getStringValue()

///////////////////////////////////////
int getPsegId( unsigned int	cluster_id,
               char*        pseg_name )
{
    unsigned int pseg_id;
    unsigned int pseg_min = cluster[cluster_id]->pseg_offset;
    unsigned int pseg_max = pseg_min + cluster[cluster_id]->psegs;

    for ( pseg_id = pseg_min ; pseg_id < pseg_max ; pseg_id++ )
    {
        if ( strcmp(pseg[pseg_id]->name, pseg_name) == 0 ) 
        {
            return pseg_id;
        }
    }
    return -1;
}

////////////////////////////////////////////
int getVspaceId( char*         vspace_name)
{
    unsigned int vspace_id;

    for( vspace_id = 0; vspace_id < vspace_index ; vspace_id++)
    {
        if( !strcmp(vspace[vspace_id]->name, vspace_name))
        {
            return vspace_id;
        }
    }
    return -1;
}

////////////////////////////////////////////
int getVobjLocId( unsigned int	vspace_id,
                  char*		    vobj_name,
                  unsigned int  vspace_max)
{
    unsigned int vobj_id;
    unsigned int vobj_min = vspace[vspace_id]->vobj_offset;
    unsigned int vobj_max = vobj_min + vspace_max;

    for ( vobj_id = vobj_min ; vobj_id < vobj_max ; vobj_id++ )
    {
        if ( strcmp(vobj[vobj_id]->name, vobj_name) == 0 )
        {
             return (vobj_id - vobj_min);
        }
    }
    return -1;
}

/////////////////////////////////////////
void taskNode ( xmlTextReaderPtr reader )
/////////////////////////////////////////
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( task_index >= MAX_TASKS )
    {
        printf("[XML ERROR] The number of tasks is larger than %d\n", MAX_TASKS);
    }

#if XML_PARSER_DEBUG
printf("   task %d\n", task_loc_index);
#endif

    task[task_index] = (mapping_task_t*)malloc(sizeof(mapping_task_t));

    ////////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      name = %s\n", str);
#endif
        strncpy( task[task_index]->name, str, 31 );
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for task (%d,%d)\n", 
               vspace_index, task_loc_index);
        exit(1);
    }

    ///////// get clusterid attribute
    value = getIntValue(reader,"clusterid", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      clusterid = %x\n", value);
#endif
        if ( value >= header->clusters )
        {
            printf("[XML ERROR] <clusterid> too large for task (%d,%d)\n",
                   vspace_index, task_loc_index);
            exit(1);
        }
        task[task_index]->clusterid = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <clusterid> attribute for task (%d,%d)\n",
               vspace_index, task_loc_index);
        exit(1);
    }

    ////////// get proclocid attribute
    value = getIntValue(reader,"proclocid", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      proclocid = %x\n", value);
#endif
        if ( value >= cluster[task[task_index]->clusterid]->procs )
        {
            printf("[XML ERROR] <proclocid> too large for task (%d,%d)\n",
                   vspace_index, task_loc_index);
            exit(1);
        }
        task[task_index]->proclocid = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <locprocid> attribute for task (%d,%d)\n", 
                vspace_index, task_loc_index);
        exit(1);
    }

    ////////// get stackname attribute
    str = getStringValue(reader, "stackname" , &ok);
    if ( ok )
    {
        int index = getVobjLocId( vspace_index, str , vobj_loc_index);
        if ( index >= 0 ) 
        {
#if XML_PARSER_DEBUG
printf("      stackname = %s\n", str);
printf("      stackid   = %d\n", index);
#endif
            task[task_index]->vobjlocid = index;
        }
        else              
        {
            printf("[XML ERROR] illegal or missing <stackname> for task (%d,%d)\n", 
                    vspace_index, task_loc_index);
            exit(1);
        }
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <stackname> for task (%d,%d)\n", 
                vspace_index, task_loc_index);
        exit(1);
    }

    ////////// get startid  attribute
    value = getIntValue(reader,"startid", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      startid = %x\n", value);
#endif
        task[task_index]->startid = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <startid> attribute for task (%d,%d)\n", 
                vspace_index, task_loc_index);
        exit(1);
    }

    /////////// get usetty  attribute (optionnal : 0 if missing)
    value = getIntValue(reader,"usetty", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      usetty = %x\n", value);
#endif
        task[task_index]->use_tty = value;
    }  
    else
    {
        task[task_index]->use_tty = 0;
    }

    /////////// get usenic  attribute (optionnal : 0 if missing)
    value = getIntValue(reader,"usenic", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      usenic = %x\n", value);
#endif
        task[task_index]->use_nic = value;
    }  
    else
    {
        task[task_index]->use_nic = 0;
    }

    /////////// get usetimer attribute (optionnal : 0 if missing)
    value = getIntValue(reader,"usetimer", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      usetimer = %x\n", value);
#endif
        task[task_index]->use_timer = value;
    }  
    else
    {
        task[task_index]->use_timer = 0;
    }

    /////////// get usefbdma  attribute (optionnal : 0 if missing)
    value = getIntValue(reader,"usefbdma", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      usefbdma = %x\n", value);
#endif
        task[task_index]->use_fbdma = value;
    }  
    else
    {
        task[task_index]->use_fbdma = 0;
    }

    task_index++;
    task_loc_index++;
} // end taskNode()

//////////////////////////////////////////
void  vobjNode ( xmlTextReaderPtr reader )
//////////////////////////////////////////
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( vobj_index >= MAX_VOBJS )
    {
        printf("[XML ERROR] The number of vobjs is larger than %d\n", MAX_VSEGS);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("      vobj %d\n", vobj_loc_index);
#endif

    vobj[vobj_index] = (mapping_vobj_t*)malloc(sizeof(mapping_vobj_t));

    ///////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        name = %s\n", str);
#endif
        strncpy( vobj[vobj_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for vobj (%d,%d)\n", 
                vseg_index, vobj_loc_index);
        exit(1);
    }

    //////// get type attribute
    str = getStringValue(reader, "type", &ok);
#if XML_PARSER_DEBUG
printf("        type = %s\n", str);
#endif
    if (ok && (strcmp(str, "ELF") == 0))
    {
        vobj[vobj_index]->type = VOBJ_TYPE_ELF;

        //check that this vobj is the first in vseg
        if(vobj_count != 0) 
        {
            printf("[XML ERROR] an ELF vobj must be alone in a vseg (%d,%d)\n", 
                    vspace_index, vobj_loc_index);
            exit(1);
        }
    }
    else if (ok && (strcmp(str, "BLOB")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_BLOB;
    else if (ok && (strcmp(str, "PTAB")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_PTAB;
    else if (ok && (strcmp(str, "PERI")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_PERI;
    else if (ok && (strcmp(str, "MWMR")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_MWMR;
    else if (ok && (strcmp(str, "LOCK")    == 0)) vobj[vobj_index]->type = VOBJ_TYPE_LOCK;
    else if (ok && (strcmp(str, "BUFFER")  == 0)) vobj[vobj_index]->type = VOBJ_TYPE_BUFFER;
    else if (ok && (strcmp(str, "BARRIER") == 0)) vobj[vobj_index]->type = VOBJ_TYPE_BARRIER;
    else
    {
        printf("[XML ERROR] illegal or missing <type> attribute for vobj (%d,%d)\n", 
                vspace_index, vobj_loc_index);
        exit(1);
    }

    ////////// get length attribute 
    value = getIntValue(reader,"length", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        length = %d\n", value);
#endif
        vobj[vobj_index]->length = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <length> attribute for vobj (%d,%d)\n", 
                vspace_index, vobj_loc_index);
        exit(1);
    }

    ////////// get align attribute (optional : 0 if missing)
    value = getIntValue(reader,"align", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        align = %d\n", value);
#endif
        vobj[vobj_index]->align = value;
    }  
    else
    {
        vobj[vobj_index]->align = 0;
    }

    ////////// get binpath attribute (optional : '\0' if missing)
    str = getStringValue(reader, "binpath", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        binpath = %s\n", str);
#endif
        strncpy(vobj[vobj_index]->binpath, str, 63);
    }  
    else
    {
        vobj[vobj_index]->binpath[0] = '\0';
    }
    
    ////////// get init attribute (mandatory for mwmr and barrier)
    value = getIntValue(reader,"init", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        init  = %d\n", value);
#endif
        vobj[vobj_index]->init = value;
    }  
    else
    {
        if( (vobj[vobj_index]->type == VOBJ_TYPE_MWMR) || 
            (vobj[vobj_index]->type == VOBJ_TYPE_BARRIER) )
        {
            printf("[XML ERROR] illegal or missing <value> attribute for vobj (%d,%d). \
                                        All MWMR or BARRIER vobj must have a init value \n", 
                    vspace_index, vobj_loc_index);
            exit(1);
        }
        vobj[vobj_index]->init = 0;
    }

    vobj_index++;
    vobj_count++;
    vobj_loc_index++;
} // end vobjNode()

//////////////////////////////////////////
void  vsegNode ( xmlTextReaderPtr reader )
//////////////////////////////////////////
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

	vobj_count = 0;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( vseg_index >= MAX_VSEGS )
    {
        printf("[XML ERROR] The number of vsegs is larger than %d\n", MAX_VSEGS);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("    vseg %d\n", vseg_loc_index);
#endif

    vseg[vseg_index] = (mapping_vseg_t*)malloc(sizeof(mapping_vseg_t));
    
    ////////// set vobj_offset attributes
    vseg[vseg_index]->vobj_offset = vobj_index;
#if XML_PARSER_DEBUG
printf("      vobj_offset = %d\n", vobj_index);
#endif

    ///////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      name = %s\n", str);
#endif
        strncpy( vseg[vseg_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for vseg (%d,%d)\n", 
                vspace_index, vseg_loc_index);
        exit(1);
    }

    ////////// get ident attribute (optional : 0 if missing)
    value = getIntValue(reader,"ident", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      ident = %d\n", value);
#endif
        vseg[vseg_index]->ident = value;
    }  
    else
    {
        vseg[vseg_index]->ident = 0;
    }

    /////////// get vbase attribute
    value = getIntValue(reader,"vbase", &ok);
    if ( ok ) 
    {
#if XML_PARSER_DEBUG
printf("      vbase = 0x%x\n", value);
#endif
        vseg[vseg_index]->vbase = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <vbase> attribute for vseg (%d,%d)\n", 
                vspace_index, vseg_loc_index);
        exit(1);
    }

    ////////// get clusterid and psegname attributes
    value = getIntValue(reader,"clusterid", &ok);
    if ( ok == 0 )
    {
        printf("[XML ERROR] illegal or missing <clusterid> for vseg %d\n", 
                 vseg_loc_index);
        exit(1);
    }
    str = getStringValue(reader,"psegname", &ok);
    if ( ok == 0 )
    {
        printf("[XML ERROR] illegal or missing <psegname> for vseg %d\n", 
                 vseg_loc_index);
        exit(1);
    }

    /////////// set psegid field
    int index = getPsegId( value, str );
    if ( index >= 0 ) 
    {
#if XML_PARSER_DEBUG
printf("      clusterid = %d\n", value);
printf("      psegname  = %s\n", str);
printf("      psegid    = %d\n", index);
#endif
        vseg[vseg_index]->psegid = index;
    }
    else              
    {
        printf("[XML ERROR] pseg not found for vseg %d / clusterid = %d / psegname = %s\n", 
                   vseg_loc_index, value, str );
        exit(1);
    }  

    //////// get mode attribute
    str = getStringValue(reader,"mode", &ok);
#if XML_PARSER_DEBUG
printf("      mode = %s\n", str);
#endif
    if      (ok && (strcmp(str, "CXWU") == 0)) vseg[vseg_index]->mode = 0xF;
    else if (ok && (strcmp(str, "CXW_") == 0)) vseg[vseg_index]->mode = 0xE;
    else if (ok && (strcmp(str, "CX_U") == 0)) vseg[vseg_index]->mode = 0xD;
    else if (ok && (strcmp(str, "CX__") == 0)) vseg[vseg_index]->mode = 0xC;
    else if (ok && (strcmp(str, "C_WU") == 0)) vseg[vseg_index]->mode = 0xB;
    else if (ok && (strcmp(str, "C_W_") == 0)) vseg[vseg_index]->mode = 0xA;
    else if (ok && (strcmp(str, "C__U") == 0)) vseg[vseg_index]->mode = 0x9;
    else if (ok && (strcmp(str, "C___") == 0)) vseg[vseg_index]->mode = 0x8;
    else if (ok && (strcmp(str, "_XWU") == 0)) vseg[vseg_index]->mode = 0x7;
    else if (ok && (strcmp(str, "_XW_") == 0)) vseg[vseg_index]->mode = 0x6;
    else if (ok && (strcmp(str, "_X_U") == 0)) vseg[vseg_index]->mode = 0x5;
    else if (ok && (strcmp(str, "_X__") == 0)) vseg[vseg_index]->mode = 0x4;
    else if (ok && (strcmp(str, "__WU") == 0)) vseg[vseg_index]->mode = 0x3;
    else if (ok && (strcmp(str, "__W_") == 0)) vseg[vseg_index]->mode = 0x2;
    else if (ok && (strcmp(str, "___U") == 0)) vseg[vseg_index]->mode = 0x1;
    else if (ok && (strcmp(str, "____") == 0)) vseg[vseg_index]->mode = 0x0;
    else
    {
        printf("[XML ERROR] illegal or missing <mode> attribute for vseg (%d,%d)\n", 
                vspace_index, vseg_loc_index);
        exit(1);
    }
    
    ////////// get vobjs in vseg
    int status = xmlTextReaderRead(reader);
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "vobj")     == 0 ) vobjNode(reader);
        else if ( strcmp(tag, "#text"  )  == 0 ) { }
        else if ( strcmp(tag, "#comment") == 0 ) { }
        else if ( strcmp(tag, "vseg")     == 0 ) 
        {
            vseg[vseg_index]->vobjs = vobj_count;
            vseg_index++;
            vseg_loc_index++;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end vsegNode()

//////////////////////////////////////////
void vspaceNode( xmlTextReaderPtr reader )
//////////////////////////////////////////
{
    char*    		str;
    unsigned int 	ok;

    vobj_loc_index = 0;
	vseg_loc_index = 0;
	task_loc_index = 0;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    // checking source file consistency
    if ( vspace_index >= header->vspaces )
    {
        printf("[XML ERROR] The vspace index is too large : %d\n", 
                 vspace_index);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("\n  vspace %d\n", vspace_index);
#endif

    vspace[vspace_index] = (mapping_vspace_t*)malloc(sizeof(mapping_vspace_t));

    ////////// get name attribute
    str = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  name = %s\n", str);
#endif
        strncpy(vspace[vspace_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute for vspace %d\n", 
                 vspace_index);
        exit(1);
    }

    ////////// set vseg_offset and task_offset attributes
    vspace[vspace_index]->vseg_offset = vseg_index;
    vspace[vspace_index]->vobj_offset = vobj_index;
    vspace[vspace_index]->task_offset = task_index;
    
#if XML_PARSER_DEBUG
printf("  vseg_offset = %d\n", vseg_index);
printf("  vobj_offset = %d\n", vobj_index);
printf("  task_offset = %d\n", task_index);
#endif

    ////////// get startname attribute
    str = getStringValue(reader, "startname", &ok);
    if ( ok )
    {
        //used after parsing the vobjs
    }
    else
    {
        printf("[XML ERROR] illegal or missing <startname> attribute for vspace %s\n", 
                 vspace[vspace_index]->name);
        exit(1);
    }

    int status = xmlTextReaderRead(reader);
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "vseg")     == 0 ) vsegNode(reader);
        else if ( strcmp(tag, "task")     == 0 ) taskNode(reader);
        else if ( strcmp(tag, "#text")    == 0 ) { }
        else if ( strcmp(tag, "#comment") == 0 ) { }
        else if ( strcmp(tag, "vspace")   == 0 ) 
        {
            vspace[vspace_index]->vobjs = vobj_loc_index; 
            vspace[vspace_index]->tasks = task_loc_index ;
            vspace[vspace_index]->vsegs = vseg_loc_index ;

            // get index of the vobj containing the start vector
            int index =  getVobjLocId( vspace_index, str , vobj_loc_index );
            if(index == -1)
            {
                printf("[XML ERROR] vobj containing start vector not found in vspace %s\n",
                        vspace[vspace_index]->name);
                exit(-1);
            }
            else
            {
                vspace[vspace_index]->start_offset = index;
#if XML_PARSER_DEBUG
printf("      startname = %s\n", str);
printf("      startid   = %d\n", index);
printf("  end vspace %d\n\n", vspace_index);
#endif
            }

            // checking startid values for all tasks in vspace
            int task_id;
            int task_min = vspace[vspace_index]->task_offset;
            int task_max = task_min + vspace[vspace_index]->tasks; 
            for ( task_id = task_min ; task_id < task_max ; task_id++ )
            {
                if ( task[task_id]->startid >= vspace[vspace_index]->tasks )
                {
                    printf("[XML ERROR] <startid> too large for task (%d,%d)\n", 
                           vspace_index, task_id );
                    exit(1);
                }
            }

            vspace_index++;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end vspaceNode()

///////////////////////////////////////////
void cpPortNode ( xmlTextReaderPtr reader )
///////////////////////////////////////////
{
    char*           str;
    unsigned int    ok;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( cp_port_index >= MAX_CP_PORTS )
    {
        printf("[XML ERROR] The number of ports (for coprocs) is larger than %d\n", MAX_CP_PORTS);
    }

#if XML_PARSER_DEBUG
printf("\n  port %d\n", cp_port_index);
#endif

    cp_port[cp_port_index] = (mapping_cp_port_t*)malloc(sizeof(mapping_cp_port_t));
    cp_port_vobj_ref[cp_port_index] = (vobj_ref_t*)malloc(sizeof(vobj_ref_t));


    
    ///////// get direction attribute
    str = getStringValue( reader, "direction", &ok );
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      direction = %s\n", str);
#endif
        if      ( strcmp(str, "TO_COPROC")   ==  0 ) cp_port[cp_port_index]->direction = PORT_TO_COPROC;
        else if ( strcmp(str, "FROM_COPROC") ==  0 ) cp_port[cp_port_index]->direction = PORT_FROM_COPROC;
        else
        {
            printf("[XML ERROR] illegal <direction> for cp_port %d in cluster %d\n",
                   cp_port_index, cluster_index);
            exit(1);
        }
    }  
    else
    {
        printf("[XML ERROR] missing <direction> for cp_port %d in cluster %d\n",
               cp_port_index, cluster_index);
        exit(1);
    }
    
    /////////// get vspacename attribute 
    str = getStringValue( reader, "vspacename", &ok );
#if XML_PARSER_DEBUG
printf("      vspacename = %s\n", str);
#endif
    if ( ok ) 
    {
        strncpy(cp_port_vobj_ref[cp_port_index]->vspace_name, str, 31);
    }
    else
    {
        printf("[XML ERROR] missing <vspacename> for cp_port %d in cluster %d\n",
               cp_port_index, cluster_index);
        exit(1);
    }

    /////////// get vobjname attribute 
    str = getStringValue( reader, "vobjname", &ok );
#if XML_PARSER_DEBUG
printf("      vobjname = %s\n", str);
#endif
    if ( ok ) 
    {
        strncpy(cp_port_vobj_ref[cp_port_index]->vobj_name, str, 31);
    }
    else
    {
        printf("[XML ERROR] missing <vobjname> for cp_port %d in cluster %d\n",
               cp_port_index, cluster_index);
        exit(1);
    }
    
    cp_port_index++;
    cp_port_loc_index++;

} // end cpPortNode()

///////////////////////////////////////////
void periphNode ( xmlTextReaderPtr reader )
///////////////////////////////////////////
{
    char*           str;
    unsigned int    value;
    unsigned int 	ok;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( periph_index >= MAX_PERIPHS )
    {
        printf("[XML ERROR] The number of periphs is larger than %d\n", MAX_PERIPHS);
    }

#if XML_PARSER_DEBUG
printf("\n  periph %d\n", periph_index);
#endif

    periph[periph_index] = (mapping_periph_t*)malloc(sizeof(mapping_periph_t));


    ///////// get channels attribute (optionnal : 1 if missing)
    value = getIntValue( reader, "channels", &ok );
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("      channels = %d\n", value);
#endif
        periph[periph_index]->channels = value;
    }
    else
    {
        periph[periph_index]->channels = 1;
    }

    /////////// get psegname attribute 
    str = getStringValue(reader,"psegname", &ok);
    if ( ok == 0 )
    {
        printf("[XML ERROR] illegal or missing <psegname> for coproc %d in cluster %d\n", 
                 coproc_index, cluster_index);
        exit(1);
    }

    /////////// set psegid attribute
    int index = getPsegId( cluster_index, str );
    if ( index >= 0 ) 
    {
#if XML_PARSER_DEBUG
printf("      clusterid = %d\n", cluster_index);
printf("      psegname  = %s\n", str);
printf("      psegid    = %d\n", index);
#endif
        periph[periph_index]->psegid = index;
        assert(pseg[index]->type == PSEG_TYPE_PERI && 
        "peripheral psegname attribute must refer to a pseg of type PERI" );
    }
    else              
    {
        printf("[XML ERROR] pseg not found for periph %d / clusterid = %d / psegname = %s\n", 
                   periph_loc_index, cluster_index, str );
        exit(1);
    }  


    /////////// get type attribute 
    str = getStringValue( reader, "type", &ok );
    if ( ok ) 
    {
#if XML_PARSER_DEBUG
printf("      type     = %s\n", str);
#endif
        unsigned int error = 0;
  
        // The TTY, IOC, NIC, FBF and IOB peripherals cannot be replicated
        // one per architecture
        if      ( strcmp( str, "IOC" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_IOC;
            if ( header->ioc_clusterid == 0xFFFFFFFF) header->ioc_clusterid = cluster_index;
            else  error = 1;

            ioc_base_offset = pseg[ periph[periph_index]->psegid ]->base;
            nb_ioc_channel = periph[periph_index]->channels;
        } 
        else if ( strcmp( str, "TTY" ) == 0 )
        {
            periph[periph_index]->type = PERIPH_TYPE_TTY;
            if ( header->tty_clusterid == 0xFFFFFFFF) header->tty_clusterid = cluster_index;
            else  error = 1;

            tty_base_offset = pseg[ periph[periph_index]->psegid ]->base;
            nb_tty_channel = periph[periph_index]->channels;
        }
        else if ( strcmp( str, "FBF" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_FBF;
            if ( header->fbf_clusterid == 0xFFFFFFFF) header->fbf_clusterid = cluster_index;
            else  error = 1;

            fbf_base_offset = pseg[ periph[periph_index]->psegid ]->base;
        }
        else if ( strcmp( str, "NIC" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_NIC;
            if ( header->nic_clusterid == 0xFFFFFFFF) header->nic_clusterid = cluster_index;
            else  error = 1;

            nic_base_offset = pseg[ periph[periph_index]->psegid ]->base;
            nb_nic_channel = periph[periph_index]->channels;
        }
        else if ( strcmp( str, "IOB" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_IOB;
            iob_base_offset = pseg[ periph[periph_index]->psegid ]->base;

            if(io_mmu_active) error = 1;
            io_mmu_active = 1;
        }
        // The TIM, ICU, XICU, DMA and IOB peripherals can be replicated in several clusters
        // one per cluster
        else if ( strcmp( str, "TIM" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_TIM;
            if(found_timer) error = 1;
            found_timer = 1;

            if(tim_base_offset == 0xFFFFFFFF)
                tim_base_offset = pseg[ periph[periph_index]->psegid ]->base;

            if(nb_timer_channel_max < periph[periph_index]->channels)
                nb_timer_channel_max = periph[periph_index]->channels;
        }
        else if ( strcmp( str, "ICU" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_ICU;
            if(found_icu) error = 1;
            found_icu = 1;

            if(icu_base_offset == 0xFFFFFFFF)
                icu_base_offset = pseg[ periph[periph_index]->psegid ]->base;
        }
        else if ( strcmp( str, "XICU" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_XICU;
            if(found_xicu) error = 1;
            found_xicu = 1;

            //'icu' since we can't have both xicu and icu in an arch
            if(icu_base_offset == 0xFFFFFFFF)
                icu_base_offset = pseg[ periph[periph_index]->psegid ]->base;

            if(nb_timer_channel_max == 0)
                nb_timer_channel_max = 32;
        }
        else if ( strcmp( str, "DMA" ) == 0 ) 
        {
            periph[periph_index]->type = PERIPH_TYPE_DMA;
            if(found_dma) error = 1;
            found_dma = 1;

            if(dma_base_offset == 0xFFFFFFFF)
                dma_base_offset = pseg[ periph[periph_index]->psegid ]->base;
            if(nb_dma_channel_max < periph[periph_index]->channels)
                nb_dma_channel_max = periph[periph_index]->channels;
        }
        else
        {
            printf("[XML ERROR] illegal <type> for peripheral %d in cluster %d\n",
                   periph_loc_index, cluster_index);
            exit(1);
        }

        if(error)
        {
            printf("[XML ERROR] illegal <type> for peripheral %d in cluster %d\n",
                   periph_loc_index, cluster_index);
            exit(1);
        }
    }
    else
    {
        printf("[XML ERROR] missing <type> for peripheral  %d in cluster %d\n",
               periph_loc_index, cluster_index);
        exit(1);
    }

    
    periph_index++;
    periph_loc_index++;
    cluster[cluster_index]->periphs++;

} // end periphNode

/////////////////////////////////////////
void coprocNode ( xmlTextReaderPtr reader )
/////////////////////////////////////////
{
    char*           str;
    unsigned int 	ok;

    cp_port_loc_index = 0;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( coproc_index >= MAX_COPROCS )
    {
        printf("[XML ERROR] The number of coprocs is larger than %d\n", MAX_COPROCS);
    }

#if XML_PARSER_DEBUG
printf("\n  coproc %d\n", coproc_index);
#endif

    coproc[coproc_index] = (mapping_coproc_t*)malloc(sizeof(mapping_coproc_t));

    /////////// get name attribute 
    str = getStringValue( reader, "name", &ok );
    if ( ok ) 
    {
#if XML_PARSER_DEBUG
printf("      name = %s\n", str);
#endif
        strncpy(coproc[coproc_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> for coproc %d in cluster %d\n",
               coproc_index, cluster_index);
        exit(1);
    }

    /////////// get psegname attribute 
    str = getStringValue(reader,"psegname", &ok);
    if ( ok == 0 )
    {
        printf("[XML ERROR] illegal or missing <psegname> for coproc %d in cluster %d\n", 
                 coproc_index, cluster_index);
        exit(1);
    }

    /////////// set psegid attribute
    int index = getPsegId( cluster_index, str );
    if ( index >= 0 ) 
    {
#if XML_PARSER_DEBUG
printf("      clusterid = %d\n", cluster_index);
printf("      psegname  = %s\n", str);
printf("      psegid    = %d\n", index);
#endif
        coproc[coproc_index]->psegid = index;
        assert(pseg[index]->type == PSEG_TYPE_PERI && "coproc psegname attribute must refer to a pseg of type PERI" );
    }
    else              
    {
        printf("[XML ERROR] pseg not found for coproc %d / clusterid = %d / psegname = %s\n", 
                   coproc_index, cluster_index, str );
        exit(1);
    }  

    ////////// set port_offset 
    coproc[coproc_index]->port_offset = cp_port_index;

#if XML_PARSER_DEBUG
printf("      port_offset = %d\n", cp_port_index);
#endif

    int status = xmlTextReaderRead(reader);
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "port")     == 0 ) cpPortNode(reader);
        else if ( strcmp(tag, "#text")    == 0 ) { }
        else if ( strcmp(tag, "#comment") == 0 ) { }
        else if ( strcmp(tag, "coproc")   == 0 ) 
        {
            coproc[coproc_index]->ports = cp_port_loc_index;
            cluster[cluster_index]->coprocs++;
            coproc_loc_index++;
            coproc_index++;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end coprocNode()

///////////////////////////////////////
void irqNode( xmlTextReaderPtr reader )
///////////////////////////////////////
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( irq_index >= MAX_IRQS )
    {
        printf("[XML ERROR] The number of irqs is larger than %d\n", MAX_IRQS);
    }

#if XML_PARSER_DEBUG
printf("     irq %d\n", irq_loc_index);
#endif

    irq[irq_index] = (mapping_irq_t*)malloc(sizeof(mapping_irq_t));

    ///////// get type attribute
    str = getStringValue(reader,"type", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        type    = %s\n", str);
#endif
        if      ( strcmp(str, "HARD") == 0 ) irq[irq_index]->type = 0;
        else if ( strcmp(str, "SOFT") == 0 ) irq[irq_index]->type = 1;
        else
        {
            printf("[XML ERROR] undefined IRQ  <type> for processor %d in cluster %d\n",
                   cluster_index, proc_loc_index );
            exit(1);
        }
    }  
    else
    {
        printf("[XML ERROR] missing IRQ <type> for processor %d in cluster %d\n",
               cluster_index, proc_loc_index );
        exit(1);
    }

    ///////// get icuid attribute
    value = getIntValue(reader, "icuid", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        icuid   = %d\n", value);
#endif
        irq[irq_index]->icuid = value;
        if ( value >= 32 )
        {
            printf("[XML ERROR] IRQ <icuid> too large for processor %d in cluster %d\n",
                   cluster_index, proc_loc_index );
            exit(1);
        }
    }
    else
    {
        printf("[XML ERROR] missing IRQ <icuid> for processor %d in cluster %d\n",
               cluster_index, proc_loc_index );
        exit(1);
    }

    ///////// get isr attribute
    str = getStringValue(reader,"isr", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        isr     = %s\n", str);
#endif
        if      ( strcmp(str, "ISR_SWITCH" ) == 0 ) irq[irq_index]->isr = ISR_SWITCH;
        else if ( strcmp(str, "ISR_IOC"    ) == 0 ) irq[irq_index]->isr = ISR_IOC;
        else if ( strcmp(str, "ISR_DMA"    ) == 0 ) irq[irq_index]->isr = ISR_DMA;
        else if ( strcmp(str, "ISR_TTY"    ) == 0 ) irq[irq_index]->isr = ISR_TTY;
        else if ( strcmp(str, "ISR_TIMER"  ) == 0 ) irq[irq_index]->isr = ISR_TIMER;
        else
        {
            printf("[XML ERROR] illegal IRQ <isr> for processor %d in cluster %d\n",
                   cluster_index, proc_loc_index );
            exit(1);
        }
#if XML_PARSER_DEBUG
printf("        isrnum  = %d\n", irq[irq_index]->isr);
#endif
    }  
    else
    {
        printf("[XML ERROR] missing IRQ <isr> for processor %d in cluster %d\n",
               cluster_index, proc_loc_index );
        exit(1);
    }

    ///////// get channel attribute (optionnal : 0 if missing)
    value = getIntValue(reader, "channel", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("        channel = %d\n", value);
#endif
        irq[irq_index]->channel = value;
    }
    else
    {
        irq[irq_index]->channel = 0;
    }

    irq_index++;
    irq_loc_index++;

} // end irqNode

/////////////////////////////////////////
void procNode ( xmlTextReaderPtr reader )
/////////////////////////////////////////
{
    unsigned int 	ok;
    unsigned int    value;

    irq_loc_index = 0;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( proc_index >= MAX_PROCS )
    {
        printf("[XML ERROR] The number of procs is larger than %d\n", MAX_PROCS);
    }

#if XML_PARSER_DEBUG
printf("\n  proc %d\n", proc_index);
#endif

    proc[proc_index] = (mapping_proc_t*)malloc(sizeof(mapping_proc_t));


    /////////// get index attribute (optional)
    value = getIntValue(reader,"index",&ok);
    if ( ok && (value != proc_loc_index) )
    {
            printf("[XML ERROR] wrong proc index / expected value is %d", 
                proc_loc_index);
            exit(1);
    }

    ////////// set irq_offset attribute
    proc[proc_index]->irq_offset = irq_index;

#if XML_PARSER_DEBUG
printf("    irq_offset = %d\n", irq_index);
#endif

    int status = xmlTextReaderRead(reader);
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "irq")      == 0 ) irqNode(reader);
        else if ( strcmp(tag, "#text")    == 0 ) { }
        else if ( strcmp(tag, "#comment") == 0 ) { }
        else if ( strcmp(tag, "proc")     == 0 ) 
        {
            proc[proc_index]->irqs = irq_loc_index;
            cluster[cluster_index]->procs++;
            proc_loc_index++;
            proc_index++;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end procNode()


//////////////////////////////////////////
void  psegNode ( xmlTextReaderPtr reader )
//////////////////////////////////////////
{
    unsigned int 	ok;
    unsigned int 	value;
    char*    		str;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    if ( pseg_index >= MAX_PSEGS )
    {
        printf("[XML ERROR] The number of psegs is larger than %d\n", MAX_PSEGS);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("    pseg %d\n", pseg_index);
#endif

    pseg[pseg_index] = (mapping_pseg_t*)malloc(sizeof(mapping_pseg_t));

    /////// get name attribute
    str = getStringValue( reader, "name", &ok );
#if XML_PARSER_DEBUG
printf("      name = %s\n", str);
#endif
    if ( ok ) 
    {
        strncpy(pseg[pseg_index]->name, str, 31);
    }
    else
    {
        printf("[XML ERROR] illegal or missing <name> for pseg %d in cluster %d\n",
               pseg_index, cluster_index);
        exit(1);
    }

    //////// get type attribute
    str = getStringValue(reader, "type", &ok);
#if XML_PARSER_DEBUG
printf("      type = %s\n", str);
#endif
    if      (ok && (strcmp(str, "RAM" ) == 0)) pseg[pseg_index]->type = PSEG_TYPE_RAM;
    else if (ok && (strcmp(str, "ROM" ) == 0)) pseg[pseg_index]->type = PSEG_TYPE_ROM;
    else if (ok && (strcmp(str, "PERI") == 0)) pseg[pseg_index]->type = PSEG_TYPE_PERI;
    else
    {
        printf("[XML ERROR] illegal or missing <type> for pseg %s in cluster %d\n",
        pseg[pseg_index]->name, cluster_index);
        exit(1);
    }

    //////// get base attribute
    value = getIntValue( reader, "base", &ok );
#if XML_PARSER_DEBUG
printf("      base = 0x%x\n", value);
#endif
    if ( ok ) 
    {
        pseg[pseg_index]->base = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <base> for pseg %s in cluster %d\n",
        pseg[pseg_index]->name, cluster_index);
        exit(1);
    }

    //////// get length attribute
    value = getIntValue( reader, "length", &ok );
#if XML_PARSER_DEBUG
printf("      length = 0x%x\n", value);
#endif
    if ( ok )
    {
        pseg[pseg_index]->length = value;
    }  
    else
    {
        printf("[XML ERROR] illegal or missing <length> for pseg %s in cluster %d\n",
        pseg[pseg_index]->name, cluster_index);
        exit(1);
    }

    //////// set cluster attribute
    pseg[pseg_index]->cluster = cluster_index;

    pseg_index++;
    cluster[cluster_index]->psegs++;
} // end psegNode()

/////////////////////////////////////////////
void  clusterNode ( xmlTextReaderPtr reader )
/////////////////////////////////////////////
{
    unsigned int ok;
    unsigned int value;

    cluster[cluster_index] = (mapping_cluster_t*)malloc(sizeof(mapping_cluster_t));
    
    //initialise all variables
    //they will be incremented by *Node() functions
    //FIXME: calloc?
    cluster[cluster_index]->psegs = 0;
    cluster[cluster_index]->procs = 0;
    cluster[cluster_index]->coprocs = 0;
    cluster[cluster_index]->periphs = 0;


    //initialise global variables
    //TODO: delete those three
    proc_loc_index = 0;
    coproc_loc_index = 0;
    periph_loc_index = 0;

    // for replicated periph
    found_timer = 0;
    found_icu = 0;
    found_xicu = 0;
    found_dma = 0;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

    // checking source file consistency
    if ( cluster_index >= header->clusters )
    {
        printf("[XML ERROR] The cluster index is too large : %d\n", cluster_index);
        exit(1);
    }

#if XML_PARSER_DEBUG
printf("  cluster %d\n", cluster_index);
#endif


    /////////// check cluster index attribute (optional)
    value = getIntValue(reader,"index",&ok);
    if ( ok && (value != cluster_index) )
    {
            printf("[XML ERROR] wrong cluster index / expected value is %d", 
                cluster_index);
            exit(1);
    }

    ////////// set offsets
    cluster[cluster_index]->pseg_offset   = pseg_index;
    cluster[cluster_index]->proc_offset   = proc_index;
    cluster[cluster_index]->coproc_offset = coproc_index;
    cluster[cluster_index]->periph_offset = periph_index;

#if XML_PARSER_DEBUG
printf("    pseg_offset   = %d\n", pseg_index);
printf("    proc_offset   = %d\n", proc_index);
printf("    coproc_offset = %d\n", coproc_index);
printf("    periph_offset = %d\n", coproc_index);
#endif

    ////////// get psegs, procs, coprocs and periphs
    int status = xmlTextReaderRead(reader);

    while ( status == 1 )
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "pseg")     == 0 ) psegNode(reader);
        else if ( strcmp(tag, "proc")     == 0 ) procNode(reader);
        else if ( strcmp(tag, "coproc")   == 0 ) coprocNode(reader);
        else if ( strcmp(tag, "periph")   == 0 ) periphNode(reader);
        else if ( strcmp(tag, "#text")    == 0 ) { }
        else if ( strcmp(tag, "#comment") == 0 ) { }
        else if ( strcmp(tag, "cluster")  == 0 ) 
        {

            if(use_xicu == 0xFFFFFFFF)
                use_xicu = found_xicu;

            ////////////////// peripherals checks ////////////////////
            if( (found_timer  && use_xicu) || (!found_timer  && !use_xicu) )
            {
                printf("[XML ERROR] illegal or missing timer peripheral in cluster %d\n", cluster_index);
                exit(1);
            }

            if( (found_icu  && use_xicu) || (!found_icu  && !use_xicu) )
            {
                printf("[XML ERROR] illegal or missing icu peripheral in cluster %d\n", cluster_index);
                exit(1);
            }

            if( !found_xicu && use_xicu)
            {
                printf("[XML ERROR] illegal or missing dma peripheral in cluster %d\n", cluster_index);
                exit(1);
            }

            if(!found_dma)
            {
                printf("[XML ERROR] illegal or missing dma peripheral in cluster %d\n", cluster_index);
                exit(1);
            }

               
            if(nb_proc_max < cluster[cluster_index]->procs)
                nb_proc_max = cluster[cluster_index]->procs;

#if XML_PARSER_DEBUG
printf("    psegs   = %d\n", cluster[cluster_index]->psegs);
printf("    procs   = %d\n", cluster[cluster_index]->procs);
printf("    coprocs = %d\n", cluster[cluster_index]->coprocs);
printf("    periphs = %d\n", cluster[cluster_index]->periphs);
printf("    end cluster %d\n", cluster_index);
#endif
            cluster_index++;
            return;
        }
        status = xmlTextReaderRead(reader);
    }
} // end clusterNode()

//////////////////////////////////////////////
void clusterSetNode( xmlTextReaderPtr reader )
//////////////////////////////////////////////
{
    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("\n  clusters set\n");
#endif

    int status = xmlTextReaderRead(reader);
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "cluster")    == 0 ) clusterNode(reader);
        else if ( strcmp(tag, "#text")      == 0 ) { }
        else if ( strcmp(tag, "#comment")   == 0 ) { }
        else if ( strcmp(tag, "clusterset") == 0 ) 
        {
            // checking source file consistency
            if ( cluster_index != header->clusters )
            {
                printf("[XML ERROR] Wrong number of clusters\n");
                exit(1);
            }

            if(header->tty_clusterid == 0xFFFFFFFF)
            {
                printf("[XML ERROR] illegal or missing tty peripheral");
                exit(1);
            }

#if XML_PARSER_DEBUG
printf("  end cluster set\n\n");
#endif
            header->psegs    = pseg_index;
            header->procs    = proc_index;
            header->irqs     = irq_index;
            header->coprocs  = coproc_index;
            header->cp_ports = cp_port_index;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in clusterset node : %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead(reader);
    }
} // end clusterSetNode()

/////////////////////////////////////////////
void globalSetNode( xmlTextReaderPtr reader )
/////////////////////////////////////////////
{
    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("  globals set\n");
#endif

    int status = xmlTextReaderRead(reader);
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "vseg")      == 0 ) vsegNode(reader);
        else if ( strcmp(tag, "#text"  )   == 0 ) { }
        else if ( strcmp(tag, "#comment")  == 0 ) { }
        else if ( strcmp(tag, "globalset") == 0 ) 
        {
#if XML_PARSER_DEBUG
printf("  end global set\n\n");
#endif
            header->globals = vseg_index;
            vseg_loc_index = 0;
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in globalset node : %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end globalSetNode()

/////////////////////////////////////////////
void vspaceSetNode( xmlTextReaderPtr reader )
/////////////////////////////////////////////
{
    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("\n  vspaces set\n");
#endif

    int status = xmlTextReaderRead ( reader );
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "vspace"   ) == 0 ) vspaceNode(reader);
        else if ( strcmp(tag, "#text"    ) == 0 ) { }
        else if ( strcmp(tag, "#comment" ) == 0 ) { }
        else if ( strcmp(tag, "vspaceset") == 0 ) 
        {
            // checking source file consistency
            if ( vspace_index != header->vspaces )
            {
                printf("[XML ERROR] Wrong number of vspaces\n");
                exit(1);
            }
            else
            {
                header->vsegs = vseg_index;
                header->vobjs = vobj_index;
                header->tasks = task_index;
                return;
            }
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in vspaceset node : %s",tag);
            exit(1);
        }
        status = xmlTextReaderRead ( reader );
    }
} // end globalSetNode()

//////////////////////////////////////////
void headerNode(xmlTextReaderPtr reader )
//////////////////////////////////////////
{
    char*       	name;
    unsigned int 	value;
    unsigned int 	ok;

    if ( xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT ) return;

#if XML_PARSER_DEBUG
printf("mapping_info\n");
#endif

    header = (mapping_header_t*)malloc(sizeof(mapping_header_t));

    ////////// get name attribute
    name = getStringValue(reader, "name", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  name = %s\n", name);
#endif
	strncpy( header->name, name, 31);
    }      
    else
    {
        printf("[XML ERROR] illegal or missing <name> attribute in header\n");
        exit(1);
    }

    /////////// get cluster_x attribute
    cluster_x = getIntValue(reader, "cluster_x", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  cluster_x = %d\n", cluster_x);
#endif
        header->cluster_x = cluster_x;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <cluster_x> attribute in header\n");
        exit(1);
    }

    /////////// get cluster_y attribute
    cluster_y = getIntValue(reader, "cluster_y", &ok);
    if ( ok )
    {
#if XML_PARSER_DEBUG
printf("  cluster_y = %d\n", cluster_y);
#endif
        header->cluster_y = cluster_y;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <cluster_y> attribute in header\n");
        exit(1);
    }

    //check the number of cluster
    value = cluster_x*cluster_y;
    if ( value >= MAX_CLUSTERS )
    {
        printf("[XML ERROR] The number of clusters is larger than %d\n", MAX_CLUSTERS);
        exit(1);
    }
	
    header->clusters  = value;

#if XML_PARSER_DEBUG
printf("  clusters = %d\n", value);
#endif

    ///////// get vspaces attribute
    value = getIntValue(reader, "vspaces", &ok);
    if ( ok )
    {
        if ( value >= MAX_VSPACES )
        {
            printf("[XML ERROR] The number of vspaces is larger than %d\n", MAX_VSPACES);
            exit(1);
        }
#if XML_PARSER_DEBUG
printf("  vspaces = %d\n", value);
#endif
	header->vspaces  = value;
    }
    else
    {
        printf("[XML ERROR] illegal or missing <vspaces> attribute in mapping\n");
        exit(1);
    }

    //////// initialise non replicated peripherals cluster_id
    header->tty_clusterid = 0xFFFFFFFF;
    header->nic_clusterid = 0xFFFFFFFF;
    header->ioc_clusterid = 0xFFFFFFFF;
    header->fbf_clusterid = 0xFFFFFFFF;

    ///////// set signature
    header->signature = IN_MAPPING_SIGNATURE;

    int status = xmlTextReaderRead(reader);
    while ( status == 1 ) 
    {
        const char* tag = (const char*)xmlTextReaderConstName(reader);

        if      ( strcmp(tag, "clusterset")   == 0 ) clusterSetNode(reader);
        else if ( strcmp(tag, "globalset")    == 0 ) globalSetNode(reader);
        else if ( strcmp(tag, "vspaceset")    == 0 ) vspaceSetNode(reader);
        else if ( strcmp(tag, "#text")        == 0 ) { }
        else if ( strcmp(tag, "#comment")     == 0 ) { }
        else if ( strcmp(tag, "mapping_info") == 0 ) 
        {
#if XML_PARSER_DEBUG
printf("end mapping_info\n");
#endif
            return;
        }
        else 
        {
            printf("[XML ERROR] Unknown tag in header node : %s\n",tag);
            exit(1);
        }
        status = xmlTextReaderRead(reader);
    }
} // end headerNode()
    
///////////////////////////////////////
void BuildTable( int fdout,
               const char* type,
               unsigned int nb_elem,
               unsigned int elem_size,
               char** table)        
////////////////////////////////////////
{
    unsigned int i;
    // write element
    for ( i = 0 ; i < nb_elem ; i++ )
    {
        if(elem_size != write(fdout, table[i], elem_size))
        {
            printf("function %s: %s(%d) write  error \n", __FUNCTION__, type, i);
            exit(1);
        }

#if XML_PARSER_DEBUG
printf("Building binary: writing %s %d\n", type, i);
#endif
    }
}

int open_file(const char* file_path)
{

    //open file
    int fdout = open( file_path, (O_CREAT | O_RDWR), (S_IWUSR | S_IRUSR) );
    if ( fdout < 0)
    {
        perror("open");
        exit(1);
    }

    //reinitialise the file
    if( ftruncate(fdout, 0) )
    {
        perror("truncate");
        exit(1);
    }

//#if XML_PARSER_DEBUG
    printf("%s\n",file_path);
//#endif

    return fdout;
}


///////////////////////////
void  buildBin( const char* file_path )
///////////////////////////
{
    unsigned int    length;

    int fdout = open_file(file_path);

    // write header to binary file
    length = write(fdout, (char*)header, sizeof(mapping_header_t));
    if ( length != sizeof(mapping_header_t) )
    {
        printf("write header error : length = %d \n", length);
        exit(1);
    }

    // write clusters
    BuildTable(fdout, "cluster", cluster_index, sizeof(mapping_cluster_t), (char**) cluster);
    // write psegs
    BuildTable(fdout, "pseg", pseg_index, sizeof(mapping_pseg_t), (char**) pseg);
    // write vspaces
    BuildTable(fdout, "vspace", vspace_index, sizeof(mapping_vspace_t), (char**) vspace);
    // write vsegs
    BuildTable(fdout, "vseg", vseg_index, sizeof(mapping_vseg_t), (char**) vseg);
    // write vobjs
    BuildTable(fdout, "vobj", vobj_index, sizeof(mapping_vobj_t), (char**) vobj);
    // write tasks array
    BuildTable(fdout, "task", task_index, sizeof(mapping_task_t), (char**) task);
    //building procs array
    BuildTable(fdout, "proc", proc_index, sizeof(mapping_proc_t), (char**) proc);
    //building irqs array
    BuildTable(fdout, "irq", irq_index, sizeof(mapping_irq_t), (char**)irq);
    //building coprocs array
    BuildTable(fdout, "coproc", coproc_index, sizeof(mapping_coproc_t), (char**)coproc);
    //building cp_ports array
    BuildTable(fdout, "cp_port", cp_port_index, sizeof(mapping_cp_port_t),(char**) cp_port);
    //building periphs array
    BuildTable(fdout, "periph", periph_index, sizeof(mapping_periph_t), (char**)periph);
    
    close(fdout);

} // end buildBin()

///////////////////////////////////////////////////////////////////////
// this function set the value the vobj_id fiels of all cp_ports
///////////////////////////////////////////////////////////////////////
void prepareBuild()
{
    unsigned int i;
    //asign for all cp_ports the correct vspaceid and vobjid
    for(i=0; i< cp_port_index; i++)
    {
        int vspace_id = getVspaceId( cp_port_vobj_ref[i]->vspace_name );
        if ( vspace_id < 0 ) 
        {
            printf("[XML ERROR] illegal  <vspacename> for cp_port %d,\n", 
                    i);
            exit(1);
        }
        cp_port[i]->vspaceid = vspace_id;
        
        int vobj_id = getVobjLocId( vspace_id, 
                                    cp_port_vobj_ref[i]->vobj_name, 
                                    vspace[vspace_id]->vobjs );
        if ( vobj_id >= 0 ) 
        {

#if XML_PARSER_DEBUG
printf("\ncp_port = %d\n", i);
printf("      vspace_name  = %s\n", cp_port_vobj_ref[i]->vspace_name);
printf("      vobj_name    = %s\n", cp_port_vobj_ref[i]->vobj_name);
printf("      vobj_index   = %d\n", vobj_id);
#endif
            cp_port[i]->vobjlocid = vobj_id;

            assert((vobj[ vspace[vspace_id]->vobj_offset + vobj_id]->type == VOBJ_TYPE_MWMR)
            && "coproc ports has to refere to a vobj of type MWMR");
        }
        else              
        {
            printf("[XML ERROR] illegal  <vobjname> for cp_port %d,\n", 
                    i);
            exit(1);
        }
    }
}

//////////////////////////////////////////
void file_write(int fdout, char* towrite)
{
    unsigned int size = strlen(towrite);
    if(size != write(fdout, towrite, size))
    {
        printf("file_write error");
        exit(1);
    }
}

//////////////////////////////////////////////////
void def_int_write(int fdout, char* def, int num)
{
    char  buf[64];
    sprintf(buf, "#define\t %s  %d\n", def, num);
    file_write(fdout, buf);
}

/////////////////////////////////////////////////
void def_hex_write(int fdout, char* def, int num)
{
    char  buf[64];
    sprintf(buf, "#define\t %s  0x%x\n", def, num);
    file_write(fdout, buf);
}

///////////////////////////////////////
void  genHd( const char* file_path )
///////////////////////////////////////
{
    int fdout = open_file(file_path);

    char* prol = " /* Generated from the mapping_info file */\n\n#ifndef _HD_CONFIG_H\n#define _HD_CONFIG_H\n\n";
    file_write(fdout, prol);

    def_int_write(fdout, "CLUSTER_X"       , cluster_x);
    def_int_write(fdout, "CLUSTER_Y"       , cluster_y);
    def_int_write(fdout, "NB_CLUSTERS"     , cluster_index);
    def_hex_write(fdout, "CLUSTER_SIZE"    , ( ((unsigned long)1) << 32)/cluster_index);
    def_int_write(fdout, "NB_PROCS_MAX"    , nb_proc_max);
    def_int_write(fdout, "NB_TIMERS_MAX"   , nb_timer_channel_max);
    def_int_write(fdout, "NB_DMAS_MAX"     , nb_dma_channel_max);
    def_int_write(fdout, "NB_TTYS"         , nb_tty_channel);
    def_int_write(fdout, "NB_IOCS"         , nb_ioc_channel);
    def_int_write(fdout, "NB_NICS"         , nb_nic_channel);
    
    file_write(fdout, "\n");
    def_int_write(fdout, "USE_XICU"         , use_xicu);
    def_int_write(fdout, "IOMMU_ACTIVE "    , io_mmu_active);

    char* epil = "\n#endif //_HD_CONFIG_H";
    file_write(fdout, epil);

    close(fdout);
}

////////////////////////////////////////////////////////
void ld_write(int fdout, char* seg, unsigned int addr)
{
    char  buf[64];
    sprintf(buf, "%s = 0x%x;\n", seg, addr);
    file_write(fdout, buf);

}

///////////////////////////////////////
void  genLd( const char* file_path )
///////////////////////////////////////
{
    int fdout = open_file(file_path);

    char* prol = "/* Generated from the mapping_info file */\n\n";
    file_write(fdout, prol);

    //boot
    ld_write(fdout, "seg_boot_code_base"           , boot_code_base);
    ld_write(fdout, "seg_boot_stack_base"          , boot_stack_base);
    ld_write(fdout, "seg_mapping_base"        , boot_mapping_base );

    //kernel
    ld_write(fdout, "\nseg_kernel_code_base"         , kernel_code_base);
    ld_write(fdout, "seg_kernel_data_base"         , kernel_data_base);
    ld_write(fdout, "seg_kernel_uncdata_base"      , kernel_uncdata_base);
    ld_write(fdout, "seg_kernel_init_base"         , kernel_init_base);

    //peripherals
    ld_write(fdout, "\nseg_fbf_base" ,  fbf_base_offset);
    ld_write(fdout, "seg_icu_base" ,  icu_base_offset);
    ld_write(fdout, "seg_ioc_base" ,  ioc_base_offset);
    ld_write(fdout, "seg_nic_base" ,  nic_base_offset);
    ld_write(fdout, "seg_tty_base" ,  tty_base_offset);
    ld_write(fdout, "seg_dma_base" ,  dma_base_offset);
    ld_write(fdout, "seg_tim_base" ,  tim_base_offset);
    ld_write(fdout, "seg_gcd_base" ,  gcd_base_offset);
    ld_write(fdout, "seg_iob_base" ,  iob_base_offset);

    close(fdout);
}

char* buildPath(const char* path, const char*name)
{
    char *res = calloc(strlen(path) + strlen(name) + 1, 1);
    strcat(res, path);
    strcat(res, "/");
    strcat(res, name);
    return res; 
}

/////////////////////////////////////
int  main ( int argc, char* argv[] )
/////////////////////////////////////
{
    if ( argc < 3 ) 
    {
        printf("Usage: xml2bin <input_file_path> <output_path>\n");
        return 1;
    }

    struct stat dir_st;
    if(stat( argv[2], &dir_st ))
    {
        perror("bad path");
        exit(1);
    }

    if((dir_st.st_mode & S_IFDIR) == 0)
    {
        printf("path is not a dir: %s", argv[2] );
        exit(1);
    }

    
    char *map_path = buildPath(argv[2], "map.bin"); 
    char *ld_path = buildPath(argv[2],  "giet_vsegs.ld"); 
    char *hd_path = buildPath(argv[2],  "hard_config.h"); 

    LIBXML_TEST_VERSION;

    int status;
    xmlTextReaderPtr reader = xmlReaderForFile ( argv[1], NULL, 0 );

    if ( reader != NULL ) 
    {
        status = xmlTextReaderRead ( reader );
        while ( status == 1 ) 
        {
            const char* tag = (const char*)xmlTextReaderConstName(reader);

            if ( strcmp(tag,"mapping_info") == 0 )
            { 
                headerNode( reader );
                prepareBuild();
                buildBin( map_path );
                genHd(hd_path);
                genLd(ld_path);
            }
            else
            {
                printf("[XML ERROR] Wrong file type: \"%s\"\n", argv[1]);
                return 1;
            }
            status = xmlTextReaderRead ( reader );
        }
        xmlFreeTextReader ( reader );

        if ( status != 0 ) 
        {
            printf("[XML ERROR] Wrong Syntax in \"%s\" file\n", argv[1]);
            return 1;
        }
    }
    return 0;
} // end main()
