/////////////////////////////////////////////////////////////////////////////////////////
// File     : ctx_handler.c
// Date     : 01/04/2012
// Authors  : alain greiner & joel porquet
// Copyright (c) UPMC-LIP6
/////////////////////////////////////////////////////////////////////////////////////////
// The ctx_handler.h and ctx_handler.c files are part of the GIET-VM nano-kernel.
// This code is used to support context switch when several tasks are executing
// in time multiplexing on a single processor.
// The tasks are statically allocated to a processor in the boot phase, and
// there is one private scheduler per processor. Each sheduler occupies 4K bytes, 
// and contains up to 14 task contexts (task_id is from 0 to 13).
// The task context [14] is reserved for the "idle" task that does nothing, and
// is launched by the scheduler when there is no other runable task.
/////////////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <drivers.h>
#include <common.h>
#include <ctx_handler.h>
#include <mapping_info.h>
#include <sys_handler.h>

/////////////////////////////////////////////////////////////////////////////////////////
// A task context is an array of 64 words = 256 bytes. 
// It contains copies of processor registers (when the task is preempted):
// - GPR[i], generally stored in slot (i). $0, *26 & $27 are not saved.
// - HI & LO registers
// - CP0 registers: EPC, SR, CR, BVAR 
// - CP2 registers : PTPR 
// It contains some general informations associated to the task:
// - TTY	: terminal global index
// - FBDMA	: DMA channel global index
// - NIC	: NIC channel global index
// - TIMER  : Timer global index
// - PTAB   : page table virtual base address 
// - LTID	: Task local index (in scheduler)
// - VSID   : Virtual space index
// - RUN	: Task state (0 => sleeping / 1 => runable ) 
//
// ctx[0]<- ***|ctx[8] <- $8 |ctx[16]<- $16|ctx[24]<- $24|ctx[32]<- EPC  |ctx[40]<- TTY
// ctx[1]<- $1 |ctx[9] <- $9 |ctx[17]<- $17|ctx[25]<- $25|ctx[33]<- CR   |ctx[41]<- DMA
// ctx[2]<- $2 |ctx[10]<- $10|ctx[18]<- $18|ctx[26]<- LO |ctx[34]<- SR   |ctx[42]<- NIC
// ctx[3]<- $3 |ctx[11]<- $11|ctx[19]<- $19|ctx[27]<- HI |ctx[35]<- BVAR |ctx[43]<- TIMER
// ctx[4]<- $4 |ctx[12]<- $12|ctx[20]<- $20|ctx[28]<- $28|ctx[36]<- ***  |ctx[44]<- PTAB
// ctx[5]<- $5 |ctx[13]<- $13|ctx[21]<- $21|ctx[29]<- SP |ctx[37]<- ***  |ctx[45]<- LTID
// ctx[6]<- $6 |ctx[14]<- $14|ctx[22]<- $22|ctx[30]<- $30|ctx[38]<- ***  |ctx[46]<- VSID
// ctx[7]<- $7 |ctx[15]<- $15|ctx[23]<- $23|ctx[31]<- RA |ctx[39]<- PTPR |ctx[47]<- RUN
//////////////////////////////////////////////////////////////////////////////////////////

extern void _task_switch(unsigned int*, unsigned int*);

/////////////////////////////////////////////////////////////////////////////////
//	_ctx_switch()
// This function performs a context switch between the running task
// and  another task, using a round-robin sheduling policy between all
// tasks allocated to a given processor (static allocation).
// It selects the next runable task to resume execution. 
// If the only runable task is the current task, return without context switch.
// If there is no runable task, the scheduler switch to the default "idle" task.
//
// Implementation notes:
// - As we only have the scheduler physical address (in CP0_SCHED register),
//   this function must use specific assess functions to access the scheduler.
// - All the context switch procedure is executed with interrupts masked.
// - The return address contained in $31 is saved in the current task context
//   (in the ctx[31] slot), and the function actually returns to the address
//   contained in the ctx[31] slot of the next task context.
/////////////////////////////////////////////////////////////////////////////////
void _ctx_switch()
{
    // get scheduler physical address
    static_scheduler_t*	psched = (static_scheduler_t*)_get_sched();

    // get number of tasks allocated to scheduler
    unsigned int	tasks = _get_tasks_number();

    // get current task index
    unsigned int 	curr_task_id = _get_current_task_id();

    // select the next task using a round-robin policy
    unsigned int 	next_task_id;
    unsigned int	tid;
    unsigned int	found = 0;

    for ( tid = curr_task_id + 1 ; 
          tid < curr_task_id + 1 + tasks ;
          tid++ )
    {
        next_task_id = tid % tasks;

        // test if the task is runable
        if ( _get_context_slot( next_task_id, CTX_RUN_ID ) )	
        {
            found = 1;
           	break;
        }
    }

    // launch "idle" task if no runable task
    if ( found == 0 )
    {
        next_task_id = IDLE_TASK_INDEX;
    }

    // no switch if no change
    if ( curr_task_id != next_task_id )
    {
        unsigned int*	curr_ctx_paddr = &(psched->context[curr_task_id][0]);
        unsigned int*	next_ctx_paddr = &(psched->context[next_task_id][0]);

        _set_current_task_id( next_task_id );
        _task_switch( curr_ctx_paddr, next_ctx_paddr );

#if GIET_DEBUG_SWITCH
_get_lock( &_tty_put_lock );
_puts( "\n[GIET DEBUG] Context switch for processor ");
_putd( _procid() );
_puts( " at cycle ");
_putd( _proctime() );
_puts("\n");
_puts( " - tasks        = ");
_putd( tasks );
_puts("\n");
_puts( " - curr_task_id = ");
_putd( curr_task_id );
_puts("\n");
_puts( " - next_task_id = ");
_putd( next_task_id );
_puts("\n");
_release_lock( &_tty_put_lock );
#endif

    }
} //end _ctx_switch()

/////////////////////////////////////////////////////////////////////////////////////
// This function is executed as the"idle" task when no other task can be executed 
/////////////////////////////////////////////////////////////////////////////////////
void _ctx_idle()
{
    unsigned int delay = 1000000;

    while(1)
    {
        asm volatile("move  $3,   %0		\n"
                     "loop:					\n"
                     "addi	$3, $3, -1		\n"
                     "bnez  $3,	loop		\n"
                     "nop					\n"
                     :
                     : "r"(delay)
                     : "$3" ); 

        _get_lock( &_tty_put_lock );
        _puts( "\n[GIET WARNING] Processor ");
        _putd( _procid() );
        _puts( " still idle at cycle ");
        _putd( _proctime() );
        _puts("\n");
        _release_lock( &_tty_put_lock );
   
    }
} // end ctx_idle()

/////////////////////////////////////////////////////////////////////////////////
// The address of this functionis used to initialise the return address
// in the "idle" task context.
/////////////////////////////////////////////////////////////////////////////////
void _ctx_eret()
{
    asm volatile("eret");
}


