///////////////////////////////////////////////////////////////////////////////////
// File     : exc_handler.c
// Date     : 01/04/2012
// Author   : alain greiner and joel porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The exc_handler.c and exc_handler.h files are part of the GIET nano-kernel.
// They contains the exception handler code.
///////////////////////////////////////////////////////////////////////////////////

#include <exc_handler.h>
#include <ctx_handler.h>
#include <sys_handler.h>
#include <drivers.h>
#include <common.h>

///////////////////////////////////////////////////////////////////////////////////
// Prototypes of exception handlers.
///////////////////////////////////////////////////////////////////////////////////

static void _cause_ukn();
static void _cause_adel();
static void _cause_ades();
static void _cause_ibe();
static void _cause_dbe();
static void _cause_bp();
static void _cause_ri();
static void _cause_cpu();
static void _cause_ovf();

extern void _int_handler();
extern void _sys_handler();

///////////////////////////////////////////////////////////////////////////////////
// Initialize the exception vector indexed by the CR XCODE field
///////////////////////////////////////////////////////////////////////////////////
const _exc_func_t _cause_vector[16] = {
    &_int_handler,  /* 0000 : external interrupt */
    &_cause_ukn,    /* 0001 : undefined exception */
    &_cause_ukn,    /* 0010 : undefined exception */
    &_cause_ukn,    /* 0011 : undefined exception */
    &_cause_adel,   /* 0100 : illegal address read exception */
    &_cause_ades,   /* 0101 : illegal address write exception */
    &_cause_ibe,    /* 0110 : instruction bus error exception */
    &_cause_dbe,    /* 0111 : data bus error exception */
    &_sys_handler,  /* 1000 : system call */
    &_cause_bp,     /* 1001 : breakpoint exception */
    &_cause_ri,     /* 1010 : illegal codop exception */
    &_cause_cpu,    /* 1011 : illegal coprocessor access */
    &_cause_ovf,    /* 1100 : arithmetic overflow exception */
    &_cause_ukn,    /* 1101 : undefined exception */
    &_cause_ukn,    /* 1110 : undefined exception */
    &_cause_ukn,    /* 1111 : undefined exception */
};

static const char * exc_type[] = {
    "strange unknown cause",
    "illegal read address",
    "illegal write address",
    "inst bus error",
    "data bus error",
    "breakpoint",
    "reserved instruction",
    "illegal coproc access",
    "arithmetic overflow",
};


static void _display_cause(unsigned int type) {
    _get_lock(&_tty_put_lock);
    _puts("\n[GIET] Exception for task ");
    _putd(_get_current_task_id());
    _puts(" on processor ");
    _putd(_procid());
    _puts(" at cycle ");
    _putd(_proctime());
    _puts("\n - type      : ");
    _puts((char *) exc_type[type]);
    _puts("\n - EPC       : ");
    _putx(_get_epc());
    _puts("\n - BVAR      : ");
    _putx(_get_bvar());
    _puts("\n");
    _puts("...Task desactivated\n");
    _release_lock(&_tty_put_lock);

    // goes to sleeping state
    unsigned int task_id = _get_current_task_id();
    _set_context_slot( task_id, CTX_RUN_ID, 0);

    // deschedule
    _ctx_switch();
}

static void _cause_ukn()  { _display_cause(0); }
static void _cause_adel() { _display_cause(1); }
static void _cause_ades() { _display_cause(2); }
static void _cause_ibe()  { _display_cause(3); }
static void _cause_dbe()  { _display_cause(4); }
static void _cause_bp()   { _display_cause(5); }
static void _cause_ri()   { _display_cause(6); }
static void _cause_cpu()  { _display_cause(7); }
static void _cause_ovf()  { _display_cause(8); }

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

