//////////////////////////////////////////////////////////////////////////////////
// File     : barrier.c      
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// These barrier.c and barrier.h files are part of the GIET nano-kernel.
// This user-level library provides a synchronisation service between several
// tasks sharing the same address space in a parallel multi-tasks application.
// Neither the barrier_init(), nor the barrier_wait() function require a syscall.
// The barrier itself must have been allocated in a shared data segment.
///////////////////////////////////////////////////////////////////////////////////

#include "barrier.h"

///////////////////////////////////////////////////////////////////////////////////
//     barrier_init()
// This function makes a cooperative initialisation of the barrier: 
// several tasks try to initialize the barrier, but the initialisation 
// is done by only one task, using LL/SC instructions.
///////////////////////////////////////////////////////////////////////////////////
void barrier_init( giet_barrier_t * barrier, unsigned int value) {
    unsigned int * pinit  = (unsigned int *) &barrier->init;
    unsigned int * pcount = (unsigned int *) &barrier->count;

    // parallel initialisation using atomic instructions LL/SC 
    // inputs : pinit, pcount, value
    // no output
    asm volatile ("_barrier_init_test:                  \n"
            "ll   $2,     0(%0)                   \n" /* read initial value */
            "bnez $2,     _barrier_init_done      \n"
            "move $3,     %2                      \n"
            "sc   $3,     0(%0)                   \n" /* write initial value */
            "beqz $3,     _barrier_init_test      \n"
            "move $3,     %2                      \n"
            "sw   $3,     0(%1)                   \n" /* write count */
            "_barrier_init_done:                  \n"
            :
            : "r"(pinit), "r"(pcount), "r"(value)
            : "$2", "$3");
}


///////////////////////////////////////////////////////////////////////////////////
//    barrier_wait()
// This blocking function uses LL/SC to decrement the barrier's counter.
// Then, it uses a busy_waiting mechanism if it is not the last.
// (because the GIET does not support dynamic task scheduling/descheduling)
///////////////////////////////////////////////////////////////////////////////////
void barrier_wait(giet_barrier_t * barrier) {
    unsigned int * pcount  = (unsigned int *) &barrier->count;
    unsigned int maxcount = barrier->init;
    unsigned int count;

    // parallel decrement barrier counter using atomic instructions LL/SC
    // - input : pointer on the barrier counter
    // - output : counter value
    asm volatile ("_barrier_decrement:          \n"
            "ll   %0, 0(%1)               \n"
            "addi $3, %0,     -1          \n"
            "sc   $3, 0(%1)               \n"
            "beqz $3, _barrier_decrement  \n"
            : "=&r"(count)
            : "r"(pcount)
            : "$2", "$3");

    // the last task re-initializes the barrier counter to the max value,
    // waking up all other waiting tasks

    if (count == 1) {
        // last task
        *pcount = maxcount;
    }
    else {
        // other tasks busy-wait
        while (*pcount != maxcount) asm volatile ("nop");
    }
}

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

