////////////////////////////////////////////////////////////////////////////
// File     : mapping_info.h
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////
// The MAPPING_INFO data structure can be used with the GIET.
// It contains the mapping directive for one or several virtual spaces.
// Ech virtual space contains a variable number of virtual segments
// and a variable number of tasks. The number of virtual space can be one.
//
// The mapping table data structure is organised as the concatenation of
// a fixed size header, and 6 variable size arrays:
//
// - mapping_cluster_t  cluster[]  
// - mapping_pseg_t     pseg[]       
// - mapping_vspace_t   vspace[]  
// - mapping_vseg_t     vseg[]     
// - mapping_vseg_t     vobj[]    
// - mapping_task_t     task[]  
// - mapping_irq_t      irq[irqs]   
// - mapping_coproc_t   coproc[]
// - mapping_cp_port_t  cp_port[] 
// - mapping_periph_t   periph[]
//
// It is intended to be stored in the boot ROM at address MAPPING_BOOT_BASE. 
////////////////////////////////////////////////////////////////////////////

#ifndef _MAPPING_INFO_H_
#define _MAPPING_INFO_H_

#define MAPPING_HEADER_SIZE   sizeof(mapping_header_t)
#define MAPPING_CLUSTER_SIZE  sizeof(mapping_cluster_t)
#define MAPPING_VSPACE_SIZE   sizeof(mapping_vspace_t)
#define MAPPING_VSEG_SIZE     sizeof(mapping_vseg_t)
#define MAPPING_VOBJ_SIZE     sizeof(mapping_vobj_t)
#define MAPPING_PSEG_SIZE     sizeof(mapping_pseg_t)
#define MAPPING_TASK_SIZE     sizeof(mapping_task_t)
#define MAPPING_PROC_SIZE     sizeof(mapping_proc_t)
#define MAPPING_IRQ_SIZE      sizeof(mapping_irq_t)
#define MAPPING_COPROC_SIZE   sizeof(mapping_coproc_t)
#define MAPPING_CP_PORT_SIZE  sizeof(mapping_cp_port_t)

#define C_MODE_MASK  0b1000 // cacheable
#define X_MODE_MASK  0b0100 // executable
#define W_MODE_MASK  0b0010 // writable
#define U_MODE_MASK  0b0001 // user access

#define IN_MAPPING_SIGNATURE    0xDEADBEEF
#define OUT_MAPPING_SIGNATURE   0xBABEF00D

enum vobjType {
    VOBJ_TYPE_ELF      = 0, // loadable code/data object of elf files
    VOBJ_TYPE_BLOB     = 1, // loadable blob object
    VOBJ_TYPE_PTAB     = 2, // page table 
    VOBJ_TYPE_PERI     = 3, // hardware component
    VOBJ_TYPE_MWMR     = 4, // MWMR channel
    VOBJ_TYPE_LOCK     = 5, // Lock
    VOBJ_TYPE_BUFFER   = 6, // Any "no initialization needed" objects (stacks...)
    VOBJ_TYPE_BARRIER  = 7, // Barrier
    VOBJ_TYPE_CONST    = 8, // Constant
    VOBJ_TYPE_MEMSPACE = 9, // Memspace; different from buffer because we add infos at the beginning
};


enum psegType {
    PSEG_TYPE_RAM  = 0,
    PSEG_TYPE_ROM  = 1,
    PSEG_TYPE_PERI = 2,
};


enum periphType {
    PERIPH_TYPE_ICU  = 0,
    PERIPH_TYPE_TIM  = 1,
    PERIPH_TYPE_XICU = 2,
    PERIPH_TYPE_DMA  = 3,
    PERIPH_TYPE_IOC  = 4,
    PERIPH_TYPE_TTY  = 5,
    PERIPH_TYPE_FBF  = 6,
    PERIPH_TYPE_NIC  = 7,
    PERIPH_TYPE_IOB  = 8,
};


enum mwmrPortDirection {
    PORT_TO_COPROC   = 0, // status register
    PORT_FROM_COPROC = 1, // config register
};


///////////////////////////////

typedef struct mapping_header_s {
    unsigned int signature;      // must contain MAPPING_SIGNATURE
    unsigned int clusters;       // number of clusters
    unsigned int cluster_x;      // number of cluster on the abcsisse axe
    unsigned int cluster_y;      // number of cluster on the ordinate axe
    unsigned int globals;        // number of vsegs mapped in all vspaces
    unsigned int vspaces;        // number of virtual spaces

    unsigned int tty_clusterid;  // index of cluster containing TTY controler
    unsigned int ioc_clusterid;  // index of cluster containing IOC controler
    unsigned int nic_clusterid;  // index of cluster containing NIC controler
    unsigned int fbf_clusterid;  // index of cluster containing FBF controler

    unsigned int psegs;          // total number of physical segments (for all clusters)
    unsigned int vsegs;          // total number of virtual segments (for all vspaces)
    unsigned int vobjs;          // total number of virtual objects (for all vspaces)
    unsigned int tasks;          // total number of tasks (for all vspaces)
    unsigned int procs;          // total number of procs (for all clusters)
    unsigned int irqs;           // total number of irqs (for all processors)
    unsigned int coprocs;        // total number of coprocs (for all clusters)
    unsigned int cp_ports;       // total number of cp_ports (for all coprocs)
    unsigned int periphs;        // total number of peripherals (for all clusters)

    char name[32];    // mapping name
} mapping_header_t;


////////////////////////////////
typedef struct mapping_cluster_s {
    unsigned int psegs;          // number of psegs in cluster
    unsigned int pseg_offset;    // index of first pseg in pseg set

    unsigned int procs;          // number of processors in cluster
    unsigned int proc_offset;    // index of first proc in proc set

    unsigned int coprocs;        // number of coprocessors in cluster
    unsigned int coproc_offset;  // index of first coproc in coproc set

    unsigned int periphs;        // number of peripherals in cluster
    unsigned int periph_offset;  // index of first coproc in coproc set
} mapping_cluster_t;


/////////////////////////////
typedef struct mapping_pseg_s  {
    char name[32];               // pseg name (unique in a cluster)
    unsigned int base;           // base address in physical space
    unsigned int length;         // size (bytes)
    unsigned int type;           // RAM / ROM / PERI
    unsigned int cluster;        // index of cluster containing pseg
    unsigned int next_base;      // first free page base address
} mapping_pseg_t;


///////////////////////////////
typedef struct mapping_vspace_s {
    char name[32];               // virtual space name
    unsigned int start_offset;   // offset of the vobj containing the start vector
    unsigned int vsegs;          // number of vsegs in vspace
    unsigned int vobjs;          // number of vobjs in vspace
    unsigned int tasks;          // number of tasks in vspace
    unsigned int vseg_offset;    // index of first vseg in vspace 
    unsigned int vobj_offset;    // index of first vobjs in vspace
    unsigned int task_offset;    // index of first task in vspace
} mapping_vspace_t;


/////////////////////////////
typedef struct mapping_vseg_s {
    char name[32];               // vseg name (unique in vspace)
    unsigned int vbase;          // base address in virtual space (hexa)
    unsigned int pbase;          // base address in physical space (hexa)
    unsigned int length;         // size (bytes)
    unsigned int psegid;         // physical segment global index
    unsigned int mode;           // C-X-W-U flags
    unsigned int ident;          // identity mapping if non zero
    unsigned int vobjs;          // number of vobjs in vseg
    unsigned int vobj_offset;    // index of first vobjs in vseg
} mapping_vseg_t;


/////////////////////////////
typedef struct mapping_task_s {
    char name[32];               // task name (unique in vspace)
    unsigned int clusterid;      // physical cluster index
    unsigned int proclocid;      // processor local index (inside cluster)
    unsigned int vobjlocid;      // stack vobj index in vspace
    unsigned int startid;        // index in start_vector 
    unsigned int use_tty;        // TTY terminal required (global)
    unsigned int use_nic;        // NIC channel required (global)
    unsigned int use_timer;      // user timer required (local)
    unsigned int use_fbdma;      // DMA channel to frame buffer required (local)
} mapping_task_t;


/////////////////////////////
typedef struct mapping_vobj_s {
    char name[32];               // vobj name (unique in a vspace)
    char binpath[64];            // path for the binary code ("*.elf")
    unsigned int type;           // type of vobj
    unsigned int length;         // size (bytes)
    unsigned int align;          // required alignement (logarithm of 2)
    unsigned int vaddr;          // virtual base addresse of the vobj
    unsigned int paddr;          // physical base addresse of the vobj
    unsigned int init;           // init value (used by barrier or mwmr channel)
} mapping_vobj_t;


/////////////////////////////
typedef struct mapping_proc_s {
    unsigned int irqs;           // number of IRQs allocated to processor
    unsigned int irq_offset;     // index of first IRQ allocated to processor
} mapping_proc_t;


/////////////////////////////
typedef struct mapping_irq_s {
    unsigned int type;           // 0 => HW_IRQ  / 1 => SW_IRQ
    unsigned int icuid;          // IRQ Index for the ICU component
    unsigned int isr;            // Interrupt Service Routine Index 
    unsigned int channel;        // Channel Index (for multi-cannels peripherals)
} mapping_irq_t;


///////////////////////////////
typedef struct mapping_coproc_s {
    char name[32];               // coprocessor name
    unsigned int psegid;         // global pseg index 
    unsigned int ports;          // number of MWMR ports used by coprocessor
    unsigned int port_offset;    // index of first MWMR port used by coprocessor
} mapping_coproc_t;


////////////////////////////////
typedef struct mapping_cp_port_s {
    unsigned int direction;      // TO_COPROC == 0 / FROM_COPROC == 1
    unsigned int vspaceid;       // index of the vspace containing the MWMR channel
    unsigned int vobjlocid;      // local index of the vobj containing the MWMR channel
} mapping_cp_port_t;


///////////////////////////////
typedef struct mapping_periph_s {
    unsigned int type;           // IOC / TTY / TIM / DMA / FBF / NIC / IOB
    unsigned int psegid;         // pseg index in cluster
    unsigned int channels;       // number of channels
} mapping_periph_t;


#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

