/* -*- c++ -*-
 *
 * GIET_VM_LGPL_HEADER_BEGIN
 * 
 * This file is part of SoCLib, GNU LGPLv2.1.
 * 
 * SoCLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 of the License.
 * 
 * SoCLib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with SoCLib; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 * 
 * GIET_VM_LGPL_HEADER_END
 *
 * Copyright (c) UPMC, Lip6, SoC
 *         Mohamed Lamine Karaoui <Mohamed.Karaoui@lip6.fr>, 2012
 */

#include <algorithm>
#include <string.h>
#include <cassert>
#include <cstring>
#include <stdexcept>

#include <iostream>
#include <sstream>
#include <iomanip>

#include "pseg.h"
#include "exception.h"


/*
 * VSeg
 */

//////////////////////////////////////
const std::string & VSeg::name() const
{
	return m_name;
}

//////////////////////////////////////
const std::string & VSeg::file() const
{
	return m_file;
}

///////////////////////////
uintptr_t VSeg::vma() const
{
	return m_vma;
}

/////////////////////////
paddr_t VSeg::lma() const
{
	return m_lma;
}

///////////////////////////
size_t VSeg::length() const
{
	return m_length;
}

/////////////////////////
size_t VSeg::type() const
{
	return m_type;
}

/////////////////////////////////////////
void VSeg::print( std::ostream &o ) const
{
	o << std::hex << std::noshowbase 
      << "<Virtual segment from(vaddr): 0x" 
      << std::setw (8) << std::setfill('0') 
      << m_vma << ", to(paddr) 0x"
      << std::setw (16) << std::setfill('0') 
      << m_lma << ", size: 0x"
      << std::setw (8) << std::setfill('0') 
      << m_length  << ",ident: " 
      << (m_ident ? "yes" : "no") << ", in(file): "
      << m_file << ", name: " << m_name << ">";
}

/////////////
VSeg::~VSeg()
{
}

/////////////////////////////////////////
VSeg & VSeg::operator=( const VSeg &ref )
{
    if ( &ref == this )
        return *this;

    m_name = ref.m_name,
    m_file = ref.m_file;
    m_vma = ref.m_vma;
    m_lma = ref.m_lma;
    m_length = ref.m_length;
    m_ident = ref.m_ident;
	return *this;
}

////////////
VSeg::VSeg()
    : m_name("No Name"),
      m_file("Empty section"),
      m_vma(0),
      m_length(0),
      m_loadable(false),
      m_ident(0)
{
}

////////////////////////////////////
VSeg::VSeg(std::string&  binaryName, 
           std::string&  name, 
           uintptr_t     vma, 
           size_t        length, 
           bool          loadable, 
           bool          ident)
    : m_name(name),
      m_file(binaryName),
      m_vma(vma),
      m_length(length),
      m_loadable(loadable),
      m_ident(ident)
{
}

/////////////////////////////
VSeg::VSeg( const VSeg &ref )
    : m_name("To be copied"),
      m_file("Empty"),
      m_vma(0),
      m_length(0),
      m_loadable(false),
      m_ident(0)
{
    (*this) = ref;
}


/*
 * PSeg
 */

/////////////////////////
paddr_t PSeg::lma() const
{
	return m_lma;
}

///////////////////////////
paddr_t PSeg::limit() const
{
	return m_limit;
}

/////////////////////////////
paddr_t PSeg::length() const
{
	return m_length;
}

/////////////////////////
size_t PSeg::type() const
{
	return m_type;
}

/////////////////////////////
paddr_t PSeg::nextLma() const
{
	return m_nextLma;
}

//////////////////////////////////////
const std::string & PSeg::name() const
{
	return m_name;
}

//////////////////////// initialisation used[][] ??? (AG)
void PSeg::check() const
{

    if(this->m_type == PSEG_TYPE_PERI)
        return;

    std::vector<VSeg>::const_iterator it;
    size_t    size = m_vsegs.size();
    paddr_t   used[size][2];          // lma, lma+length
    size_t    i,j,error=0;
    
    for(it = m_vsegs.begin(), i= 0 ; it < m_vsegs.end() ; it++, i++)
    {
        paddr_t it_limit = (*it).lma() + (*it).length();
        for(j=0; j< i; j++)
        {
            if( used[j][0] == (*it).lma() ) //not the same lma ,
            {
                error = 1;
                std::cout << "ok \n";
            }
            if( used[j][1] == it_limit )  // and not the same limit
            {
                error = 2;
            }
            if( (used[j][0] < (*it).lma()) and ((*it).lma() < used[j][1]) ) // lma  within
            {
                error = 3;
            }
            if(  ((used[j][0] < it_limit) and (it_limit < used[j][1] )) ) // limit no within
            {
                error = 4;
                std::cout << "base: " << std::hex << (*it).lma() << std::endl;
                std::cout << "limit: " << std::hex << it_limit << std::endl;
                std::cout << "used[j][0]: " << std::hex << used[j][0] << std::endl;
                std::cout << "used[j][1]: " << std::hex << used[j][1] << std::endl;
            }
            if(error)
            {
                std::ostringstream err;
                err << " Error" << error << " ,ovelapping Buffers:" << std::endl 
                    << *it << std::endl << m_vsegs[j] << std::endl; 
                throw exception::RunTimeError( err.str().c_str() );
            }

        }
        used[i][0] = (*it).lma();
        used[i][1] = it_limit;
    }
}

//////////////////////////////////////
void PSeg::setName(std::string& name )
{
    m_name = name;
}

/////////////////////////////////////////////////////////
paddr_t PSeg::align( paddr_t toAlign, unsigned alignPow2)
{
    return ((toAlign + (1 << alignPow2) - 1 ) >> alignPow2) << alignPow2; 
}

//////////////////////////////////////////
paddr_t PSeg::pageAlign( paddr_t toAlign )
{
    size_t pgs = pageSize();
    size_t pageSizePow2 = __builtin_ctz(pgs);
    
    return align(toAlign, pageSizePow2); 
}

////////////////////////////////
void PSeg::setLma( paddr_t lma )
{
    m_lma = lma;
    
    m_nextLma = pageAlign(lma);

    m_pageLimit = pageAlign(m_lma+m_length); 

    m_limit = (m_lma + m_length);
}

/////////////////////////////////////
void PSeg::setLength( paddr_t length )
{
    m_length = length;

    m_pageLimit = pageAlign(m_lma+m_length); 

    m_limit = (m_lma + m_length);
}

////////////////////////////
void PSeg::add( VSeg& vseg )
{
    vseg.m_lma = m_nextLma;
//    incNextLma(vseg.length());   //for the next vseg
    m_vsegs.push_back(vseg); 
}

/////////////////////////////////
void PSeg::addIdent( VSeg& vseg )
{
    vseg.m_lma = vseg.m_vma;
    m_vsegs.push_back(vseg); 
}

/////////////////////////////////////////
void PSeg::setNextLma( paddr_t nextLma)
{
    m_nextLma = nextLma;
    confNextLma();
}

//////////////////////////////////
void PSeg::incNextLma( size_t inc)
{
    m_nextLma += inc;
    confNextLma();
}

////////////////////////
void PSeg::confNextLma()
{
    if(m_nextLma > m_limit)
    {
        std::cerr << "Erreur pseg overflow... nextLma: "
                  << std::hex << m_nextLma << ", limit: " 
                  << m_limit << std::endl;
        exit(1); 
    }

    m_nextLma = pageAlign( m_nextLma );

    if(m_nextLma > m_pageLimit)
    {
        std::cerr << "Erreur pseg page overflow... nextLma: "
                  << std::hex << m_nextLma << ", limit: " 
                  << m_pageLimit << std::endl;
        exit(1); 
    }
}

/////////////////////////////////
void PSeg::setPageSize(size_t pg)
{
    if( pg == 0)
    {
        std::cerr << "PageSize must be positive" << std::endl;
        return;
    }
    pageSize() = pg;
}

////////////////////////
size_t& PSeg::pageSize()
{
    static size_t m_pageSize;
    return m_pageSize;
}

/////////////////////////////////////////
PSeg & PSeg::operator=( const PSeg &ref )
{
    if ( &ref == this )
        return *this;

    m_name = ref.m_name;
    m_length = ref.m_length;
    m_limit = ref.m_limit;
    m_pageLimit = ref.m_pageLimit;
    m_lma = ref.m_lma;
    m_nextLma = ref.m_nextLma;
    m_vsegs = ref.m_vsegs;
    m_type = ref.m_type;

	return *this;
}

//////////////////////////////////////////
void PSeg::print( std::ostream &o ) const
{
	o << "<Physical segment "
	  << std::showbase << m_name
	  << ", from: " << std::hex 
      << m_lma << " to " << m_limit 
      << ", size : "  << m_length 
      << ", filled to: "  << m_nextLma 
      << ", type : "  << m_type 
      << ", containing: "<< std::endl;
        std::vector<VSeg>::const_iterator it;
        for(it = m_vsegs.begin(); it < m_vsegs.end(); it++)
    o << " " << *it << std::endl;

    o << ">";
}

////////////////////////////////////
PSeg::PSeg( const std::string &name,
            paddr_t           lma,
            paddr_t           length,
            size_t            type)
{
    m_name = name;
    m_length = length;
    m_type = type;

    setLma(lma);
}

////////////////////////////////////
PSeg::PSeg( const std::string &name):
      m_lma(0),
      m_length(0),
      m_nextLma(0),
      m_limit(0)
{
    m_name = name;
}

////////////////////////
PSeg::PSeg( paddr_t lma):
      m_name("No name"),
      m_lma(0),
      m_length(0),
      m_nextLma(0),
      m_limit(0)
{
    setLma(lma);
}

////////////
PSeg::PSeg()
    :
      m_name("Empty section"),
      m_lma(0),
      m_length(0),
      m_nextLma(0),
      m_limit(0)
{
}

/////////////////////////////
PSeg::PSeg( const PSeg &ref )
    : m_name("To be copied"),
      m_lma(0),
      m_length(0),
      m_nextLma(0),
      m_limit(0)
{
    (*this) = ref;
}

PSeg::~PSeg()
{
}



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

