///////////////////////////////////////////////////////////////////////////////////
// File     : drivers.c
// Date     : 23/05/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The drivers.c and drivers.h files are part ot the GIET-VM nano kernel.
// They contains the drivers for the peripherals available in the SoCLib library:
// - vci_multi_tty
// - vci_multi_timer
// - vci_multi_dma
// - vci_multi_icu
// - vci_xicu & vci_multi_icu
// - vci_gcd
// - vci_frame_buffer
// - vci_block_device
//
// For the peripherals replicated in each cluster (ICU, TIMER, DMA),
// the corresponding (virtual) base addresses must be completed by an offset 
// depending on the cluster index.
//
// The following global parameter must be defined in the giet_config.h file:
// - GIET_CLUSTER_INCREMENT
//
// The following global parameters must be defined in the hard_config.h file:
// - NB_CLUSTERS   
// - NB_PROCS_MAX  
// - NB_TIM_CHANNELS    
// - NB_DMA_CHANNELS     
// - NB_TTY_CHANNELS_MAX    
//
// The following virtual base addresses must be defined in the giet_vsegs.ld file:
// - seg_icu_base
// - seg_tim_base
// - seg_dma_base
// - seg_tty_base
// - seg_gcd_base
// - seg_fbf_base
// - seg_ioc_base
// - seg_nic_base
// - seg_cma_base
// - seg_iob_base
// - seg_mmc_base
//
///////////////////////////////////////////////////////////////////////////////////

#include <vm_handler.h>
#include <sys_handler.h>
#include <giet_config.h>
#include <drivers.h>
#include <common.h>
#include <hwr_mapping.h>
#include <mips32_registers.h>
#include <ctx_handler.h>

#if !defined(NB_CLUSTERS) 
# error: You must define NB_CLUSTERS in the hard_config.h file
#endif

#if (NB_CLUSTERS > 256)
# error: NB_CLUSTERS cannot be larger than 256!
#endif

#if !defined(NB_PROCS_MAX) 
# error: You must define NB_PROCS_MAX in the hard_config.h file
#endif

#if (NB_PROCS_MAX > 8)
# error: NB_PROCS_MAX cannot be larger than 8!
#endif

#if !defined(GIET_CLUSTER_INCREMENT) 
# error: You must define GIET_CLUSTER_INCREMENT in the giet_config.h file
#endif

#if !defined(GIET_USE_IOMMU) 
# error: You must define GIET_USE_IOMMU in the giet_config.h file
#endif

#if !defined(NB_TTY_CHANNELS)
# error: You must define NB_TTY_CHANNELS in the hard_config.h file
#endif

#if (NB_TTY_CHANNELS < 1)
# error: NB_TTY_CHANNELS cannot be smaller than 1!
#endif

#if !defined(NB_DMA_CHANNELS)
# error: You must define NB_DMA_CHANNELS in the hard_config.h file 
#endif

#if (NB_DMA_CHANNELS > 8)
# error: NB_DMA_CHANNELS cannot be smaller than 8!
#endif

#if !defined(NB_TIM_CHANNELS)
#define NB_TIM_CHANNELS 0
#endif

#if ( (NB_TIM_CHANNELS + NB_PROC_MAX) > 32 )
# error: NB_TIM_CHANNELS + NB_PROCS_MAX cannot be larger than 32
#endif

#if !defined(NB_IOC_CHANNELS)
# error: You must define NB_IOC_CHANNELS in the hard_config.h file 
#endif

#if ( NB_IOC_CHANNELS > 8 )
# error: NB_IOC_CHANNELS cannot be larger than 8
#endif

#if !defined(NB_NIC_CHANNELS)
# error: You must define NB_NIC_CHANNELS in the hard_config.h file 
#endif

#if ( NB_NIC_CHANNELS > 8 )
# error: NB_NIC_CHANNELS cannot be larger than 8
#endif

#if !defined(NB_CMA_CHANNELS)
# error: You must define NB_CMA_CHANNELS in the hard_config.h file 
#endif

#if ( NB_CMA_CHANNELS > 8 )
# error: NB_CMA_CHANNELS cannot be larger than 8
#endif

#if !defined( USE_XICU )
# error: You must define USE_XICU in the hard_config.h file
#endif

#if !defined( USE_IOB )
# error: You must define USE_IOB in the hard_config.h file
#endif


#define in_unckdata __attribute__((section (".unckdata")))

//////////////////////////////////////////////////////////////////////////////
//     Timers driver
//////////////////////////////////////////////////////////////////////////////
// This peripheral is replicated in all clusters.
// The timers can be implemented in a vci_timer component or in a vci_xicu 
// component (depending on the USE_XICU parameter).
// There is one timer (or xicu) component per cluster.
// There is two types of timers: 
// - "system" timers : one per processor, used for context switch.
//   local_id in [0, NB_PROCS_MAX-1],
// - "user" timers : requested by the task in the mapping_info data structure.
//   For each user timer, the timer_id is stored in the context of the task.
// The global index is cluster_id * (NB_PROCS_MAX+NB_TIM_CHANNELS) + local_id
//////////////////////////////////////////////////////////////////////////////
// The (virtual) base address of the associated segment is:
//
//       timer_address = seg_icu_base + cluster_id * GIET_CLUSTER_INCREMENT
//
// - cluster id is an explicit argument of all access functions
// - seg_icu_base must be defined in the giet_vsegs.ld file
// - GIET_CLUSTER_INCREMENT must be defined in the giet_config.h file
////////////////////////////////////////////////////////////////////////////////

// User Timer signaling variables 

#if (NB_TIM_CHANNELS > 0)
in_unckdata volatile unsigned char _user_timer_event[NB_CLUSTERS * NB_TIM_CHANNELS] 
                            = { [0 ... ((NB_CLUSTERS * NB_TIM_CHANNELS) - 1)] = 0 };
#endif

//////////////////////////////////////////////////////////////////////////////
//     _timer_start()
// This function activates a timer in the vci_timer (or vci_xicu) component
// by writing in the proper register the period value.
// It can be used by both the kernel to initialise a "system" timer,
// or by a task (through a system call) to configure an "user" timer.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////
unsigned int _timer_start( unsigned int cluster_id, 
                           unsigned int local_id, 
                           unsigned int period) 
{
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS)  return 1;
    if (local_id >= NB_TIM_CHANNELS)  return 2;

#if USE_XICU
    unsigned int * timer_address = (unsigned int *) ((char *) &seg_icu_base + 
                                                     (cluster_id * GIET_CLUSTER_INCREMENT));

    timer_address[XICU_REG(XICU_PTI_PER, local_id)] = period;
#else
    unsigned int* timer_address = (unsigned int *) ((char *) &seg_tim_base + 
                                                    (cluster_id * GIET_CLUSTER_INCREMENT));

    timer_address[local_id * TIMER_SPAN + TIMER_PERIOD] = period;
    timer_address[local_id * TIMER_SPAN + TIMER_MODE] = 0x3;
#endif
    return 0;
}

//////////////////////////////////////////////////////////////////////////////
//     _timer_stop()
// This function desactivates a timer in the vci_timer (or vci_xicu) component
// by writing in the proper register.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////
unsigned int _timer_stop( unsigned int cluster_id, 
                          unsigned int local_id) 
{
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS)  return 1;
    if (local_id >= NB_TIM_CHANNELS)  return 2;

#if USE_XICU
    unsigned int * timer_address = (unsigned int *) ((char *) &seg_icu_base + 
                                                     (cluster_id * GIET_CLUSTER_INCREMENT));

    timer_address[XICU_REG(XICU_PTI_PER, local_id)] = 0;
#else
    unsigned int* timer_address = (unsigned int *) ((char *) &seg_tim_base + 
                                                    (cluster_id * GIET_CLUSTER_INCREMENT));

    timer_address[local_id * TIMER_SPAN + TIMER_MODE] = 0;
#endif
    return 0;
}


//////////////////////////////////////////////////////////////////////////////
//     _timer_reset_irq()
// This function acknowlegge a timer interrupt in the vci_timer (or vci_xicu) 
// component by reading/writing in the proper register.
// It can be used by both the isr_switch() for a "system" timer, 
// or by the _isr_timer() for an "user" timer.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////
unsigned int _timer_reset_irq( unsigned int cluster_id, 
                               unsigned int local_id ) 
{
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS)  return 1;
    if (local_id >= NB_TIM_CHANNELS)  return 2;

#if USE_XICU
    unsigned int * timer_address = (unsigned int *) ((char *) &seg_icu_base +
                                                     (cluster_id * GIET_CLUSTER_INCREMENT));

    unsigned int bloup = timer_address[XICU_REG(XICU_PTI_ACK, local_id)];
    bloup++; // to avoid a warning 
#else
    unsigned int * timer_address = (unsigned int *) ((char *) &seg_tim_base + 
            (cluster_id * GIET_CLUSTER_INCREMENT));

    timer_address[local_id * TIMER_SPAN + TIMER_RESETIRQ] = 0;
#endif
    return 0;
}



///////////////////////////////////////////////////////////////////////
// _timer_reset_irq_cpt()
///////////////////////////////////////////////////////////////////////
// This function resets the period at the end of which
// an interrupt is sent. To do so, we re-write the period
// ini the proper register, what causes the count to restart.
// The period value is read from the same (TIMER_PERIOD) register,
// this is why in appearance we do nothing useful (read a value
// from a register and write this value in the same register)
// This function is called during a context switch (user or preemptive)
///////////////////////////////////////////////////////////////////////
unsigned int _timer_reset_irq_cpt(unsigned int cluster_id, unsigned int local_id) {
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS) {
        return 1;
    }
    if (local_id >= NB_TIM_CHANNELS) {
        return 2;
    }

#if USE_XICU
    unsigned int * timer_address = (unsigned int *) ((char *) &seg_icu_base + (cluster_id * GIET_CLUSTER_INCREMENT));
    unsigned int timer_period = timer_address[XICU_REG(XICU_PTI_PER, local_id)];

    // we write 0 first because if the timer is currently running, the corresponding timer counter is not reset
    timer_address[XICU_REG(XICU_PTI_PER, local_id)] = 0;
    timer_address[XICU_REG(XICU_PTI_PER, local_id)] = timer_period;
#else
    // We suppose that the TIMER_MODE register value is 0x3
    unsigned int * timer_address = (unsigned int *) ((char *) &seg_tim_base + (cluster_id * GIET_CLUSTER_INCREMENT));
    unsigned int timer_period = timer_address[local_id * TIMER_SPAN + TIMER_PERIOD];

    timer_address[local_id * TIMER_SPAN + TIMER_PERIOD] = timer_period;
#endif

    return 0;
}


/////////////////////////////////////////////////////////////////////////////////
//     VciMultiTty driver
/////////////////////////////////////////////////////////////////////////////////
// There is only one multi_tty controler in the architecture.
// The total number of TTYs is defined by the configuration parameter NB_TTY_CHANNELS.
// The "system" terminal is TTY[0].
// The "user" TTYs are allocated to applications by the GIET in the boot phase,
// as defined in the mapping_info data structure. The corresponding tty_id must 
// be stored in the context of the task by the boot code.
// The TTY address is : seg_tty_base + tty_id*TTY_SPAN
/////////////////////////////////////////////////////////////////////////////////

// TTY variables
in_unckdata volatile unsigned char _tty_get_buf[NB_TTY_CHANNELS];
in_unckdata volatile unsigned char _tty_get_full[NB_TTY_CHANNELS] 
                                     = { [0 ... NB_TTY_CHANNELS - 1] = 0 };
in_unckdata unsigned int _tty_put_lock = 0;  // protect kernel TTY[0]

////////////////////////////////////////////////////////////////////////////////
//      _tty_error()
////////////////////////////////////////////////////////////////////////////////
void _tty_error(unsigned int tty_id, unsigned int task_id) 
{
    unsigned int proc_id = _procid();

    _get_lock(&_tty_put_lock);
    if (tty_id == 0xFFFFFFFF) _puts("\n[GIET ERROR] no TTY assigned to the task ");
    else                      _puts("\n[GIET ERROR] TTY index too large for task ");
    _putd(task_id);
    _puts(" on processor ");
    _putd(proc_id);
    _puts("\n");
    _release_lock(&_tty_put_lock);
}


/////////////////////////////////////////////////////////////////////////////////
//      _tty_write()
// Write one or several characters directly from a fixed-length user buffer to
// the TTY_WRITE register of the TTY controler.
// It doesn't use the TTY_PUT_IRQ interrupt and the associated kernel buffer.
// This is a non blocking call: it tests the TTY_STATUS register, and stops
// the transfer as soon as the TTY_STATUS[WRITE] bit is set. 
// The function returns  the number of characters that have been written.
/////////////////////////////////////////////////////////////////////////////////
unsigned int _tty_write(const char * buffer, 
                        unsigned int length) 
{
    unsigned int nwritten;
    unsigned int tty_id = _get_context_slot(CTX_TTY_ID);
    unsigned int* tty_address = (unsigned int *) &seg_tty_base;

    for (nwritten = 0; nwritten < length; nwritten++) 
    {
        // check tty's status 
        if ((tty_address[tty_id * TTY_SPAN + TTY_STATUS] & 0x2) == 0x2) break;
        tty_address[tty_id * TTY_SPAN + TTY_WRITE] = (unsigned int) buffer[nwritten];
    }
    return nwritten;
}

//////////////////////////////////////////////////////////////////////////////
//      _tty_read()
// This non-blocking function uses the TTY_GET_IRQ[tty_id] interrupt and 
// the associated kernel buffer, that has been written by the ISR.
// It get the TTY terminal index from the context of the current task.
// It fetches one single character from the _tty_get_buf[tty_id] kernel
// buffer, writes this character to the user buffer, and resets the
// _tty_get_full[tty_id] buffer.
// The length argument is not used.
// Returns 0 if the kernel buffer is empty, 1 if the buffer is full.
//////////////////////////////////////////////////////////////////////////////
unsigned int _tty_read(char * buffer, 
                       unsigned int length) 
{
    unsigned int tty_id = _get_context_slot(CTX_TTY_ID);

    if (_tty_get_full[tty_id] == 0) 
    {
        return 0;
    }
    else 
    {
        *buffer = _tty_get_buf[tty_id];
        _tty_get_full[tty_id] = 0;
        return 1;
    }
}

////////////////////////////////////////////////////////////////////////////////
//     _tty_get_char()
// This function is used by the _isr_tty to read a character in the TTY
// terminal defined by the tty_id argument. The character is stored
// in requested buffer, and the IRQ is acknowledged.
// Returns 0 if success, 1 if tty_id too large. 
////////////////////////////////////////////////////////////////////////////////
unsigned int _tty_get_char(unsigned int tty_id, 
                           unsigned char * buffer) 
{
    // checking argument
    if (tty_id >= NB_TTY_CHANNELS) { return 1; }

    // compute terminal base address 
    unsigned int * tty_address = (unsigned int *) &seg_tty_base; 

    *buffer = (unsigned char) tty_address[tty_id * TTY_SPAN + TTY_READ];
    return 0;
}


////////////////////////////////////////////////////////////////////////////////
//     VciMultiIcu or VciXicu driver
////////////////////////////////////////////////////////////////////////////////
// This hardware component is replicated in all clusters.
// There is one vci_multi_icu (or vci_xicu) component per cluster, 
// and the number of independant ICUs is equal to NB_PROCS_MAX, 
// because there is one private interrupt controler per processor.
////////////////////////////////////////////////////////////////////////////////
// The (virtual) base address of the associated segment is:
//
//       icu_address = seg_icu_base + cluster_id * GIET_CLUSTER_INCREMENT
//
// - cluster id is an explicit argument of all access functions
// - seg_icu_base must be defined in the giet_vsegs.ld file
// - GIET_CLUSTER_INCREMENT must be defined in the giet_config.h file
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//     _icu_set_mask()
// This function can be used with both the vci_xicu & vci_multi_icu components.
// It set the mask register for the ICU channel identified by the cluster index 
// and the processor index: all '1' bits are set / all '0' bits are not modified.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _icu_set_mask( unsigned int cluster_id,
                            unsigned int proc_id,
                            unsigned int value,
                            unsigned int is_timer) 
{
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS) return 1; 
    if (proc_id >= NB_PROCS_MAX)   return 1; 

    unsigned int * icu_address = (unsigned int *) ((char *) &seg_icu_base + 
                                                   (cluster_id * GIET_CLUSTER_INCREMENT));
#if USE_XICU
    if (is_timer) 
    {
        icu_address[XICU_REG(XICU_MSK_PTI_ENABLE, proc_id)] = value;
    }
    else 
    {
        icu_address[XICU_REG(XICU_MSK_HWI_ENABLE, proc_id)] = value;
    }
#else
    icu_address[proc_id * ICU_SPAN + ICU_MASK_SET] = value; 
#endif
    return 0;
}


////////////////////////////////////////////////////////////////////////////////
//     _icu_get_index()
// This function can be used with both the vci_xicu & vci_multi_icu components.
// It returns the index of the highest priority (smaller index) active HWI.
// The ICU channel is identified by the cluster index and the processor index.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _icu_get_index( unsigned int cluster_id, 
                             unsigned int proc_id, 
                             unsigned int * buffer) 
{
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS)  return 1;
    if (proc_id >= NB_PROCS_MAX)    return 1;

    unsigned int * icu_address = (unsigned int *) ((char *) &seg_icu_base + 
                                                   (cluster_id * GIET_CLUSTER_INCREMENT));
#if USE_XICU
    unsigned int prio = icu_address[XICU_REG(XICU_PRIO, proc_id)];
    unsigned int pti_ok = (prio & 0x00000001);
    unsigned int hwi_ok = (prio & 0x00000002);
    unsigned int swi_ok = (prio & 0x00000004);
    unsigned int pti_id = (prio & 0x00001F00) >> 8;
    unsigned int hwi_id = (prio & 0x001F0000) >> 16;
    unsigned int swi_id = (prio & 0x1F000000) >> 24;
    if      (pti_ok) { *buffer = pti_id; }
    else if (hwi_ok) { *buffer = hwi_id; }
    else if (swi_ok) { *buffer = swi_id; }
    else             { *buffer = 32; }
#else
    *buffer = icu_address[proc_id * ICU_SPAN + ICU_IT_VECTOR]; 
#endif
    return 0;
}


////////////////////////////////////////////////////////////////////////////////
//     VciGcd driver
////////////////////////////////////////////////////////////////////////////////
// The Greater Dommon Divider is a -very- simple hardware coprocessor
// performing the computation of the GCD of two 32 bits integers.
// It has no DMA capability.
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//     _gcd_write()
// Write a 32-bit word in a memory mapped register of the GCD coprocessor.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _gcd_write( unsigned int register_index, 
                         unsigned int value) 
{
    // parameters checking
    if (register_index >= GCD_END)  return 1; 

    unsigned int * gcd_address = (unsigned int *) &seg_gcd_base;

    gcd_address[register_index] = value; // write word
    return 0;
}


////////////////////////////////////////////////////////////////////////////////
//     _gcd_read()
// Read a 32-bit word in a memory mapped register of the GCD coprocessor.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _gcd_read( unsigned int register_index, 
                        unsigned int * buffer ) 
{
    // parameters checking 
    if (register_index >= GCD_END)  return 1;

    unsigned int * gcd_address = (unsigned int *) &seg_gcd_base;

    *buffer = gcd_address[register_index]; // read word
    return 0;
}

////////////////////////////////////////////////////////////////////////////////
// VciBlockDevice driver
////////////////////////////////////////////////////////////////////////////////
// The VciBlockDevice is a single channel external storage contrôler.
//
// The IOMMU can be activated or not:
// 
// 1) When the IOMMU is used, a fixed size 2Mbytes vseg is allocated to 
// the IOC peripheral, in the I/O virtual space, and the user buffer is
// dynamically remapped in the IOMMU page table. The corresponding entry 
// in the IOMMU PT1 is defined by the kernel _ioc_iommu_ix1 variable.
// The number of pages to be unmapped is stored in the _ioc_npages variable.
// The number of PT2 entries is dynamically computed and stored in the
// kernel _ioc_iommu_npages variable. It cannot be larger than 512.
// The user buffer is unmapped by the _ioc_completed() function when 
// the transfer is completed.
//
// 2/ If the IOMMU is not used, we check that  the user buffer is mapped to a
// contiguous physical buffer (this is generally true because the user space
// page tables are statically constructed to use contiguous physical memory).
//
// Finally, the memory buffer must fulfill the following conditions:
// - The user buffer must be word aligned, 
// - The user buffer must be mapped in user address space, 
// - The user buffer must be writable in case of (to_mem) access,
// - The total number of physical pages occupied by the user buffer cannot
//   be larger than 512 pages if the IOMMU is activated,
// - All physical pages occupied by the user buffer must be contiguous
//   if the IOMMU is not activated.
// An error code is returned if these conditions are not verified.
//
// As the IOC component can be used by several programs running in parallel,
// the _ioc_lock variable guaranties exclusive access to the device.  The
// _ioc_read() and _ioc_write() functions use atomic LL/SC to get the lock.
// and set _ioc_lock to a non zero value.  The _ioc_write() and _ioc_read()
// functions are blocking, polling the _ioc_lock variable until the device is
// available.
// When the tranfer is completed, the ISR routine activated by the IOC IRQ
// set the _ioc_done variable to a non-zero value. Possible address errors
// detected by the IOC peripheral are reported by the ISR in the _ioc_status
// variable.
// The _ioc_completed() function is polling the _ioc_done variable, waiting for
// transfer completion. When the completion is signaled, the _ioc_completed()
// function reset the _ioc_done variable to zero, and releases the _ioc_lock
// variable.
//
// In a multi-processing environment, this polling policy should be replaced by
// a descheduling policy for the requesting process.
///////////////////////////////////////////////////////////////////////////////

// IOC global variables
in_unckdata volatile unsigned int _ioc_status= 0;
in_unckdata volatile unsigned int _ioc_done = 0;
in_unckdata unsigned int _ioc_lock = 0;
in_unckdata unsigned int _ioc_iommu_ix1 = 0;
in_unckdata unsigned int _ioc_iommu_npages; 

///////////////////////////////////////////////////////////////////////////////
//      _ioc_access()
// This function transfer data between a memory buffer and the block device.
// The buffer lentgth is (count*block_size) bytes.
// Arguments are:
// - to_mem     : from external storage to memory when non 0
// - lba        : first block index on the external storage.
// - user_vaddr : virtual base address of the memory buffer.
// - count      : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _ioc_access( unsigned int to_mem,
                          unsigned int lba,
                          unsigned int user_vaddr,
                          unsigned int count) 
{
    unsigned int user_vpn_min;     // first virtuel page index in user space
    unsigned int user_vpn_max;     // last virtual page index in user space
    unsigned int vpn;              // current virtual page index in user space
    unsigned int ppn;              // physical page number
    unsigned int flags;            // page protection flags
    unsigned int ix2;              // page index in IOMMU PT1 page table
    unsigned int ppn_first;        // first physical page number for user buffer
    unsigned int buf_xaddr = 0;    // user buffer virtual address in IO space (if IOMMU)
    paddr_t      buf_paddr = 0;    // user buffer physical address (if no IOMMU),

    // check buffer alignment
    if ((unsigned int) user_vaddr & 0x3)
    {
        _get_lock(&_tty_put_lock);
        _puts("[GIET ERROR] in _ioc_access() : user buffer not word aligned\n");
        _release_lock(&_tty_put_lock);
        return 1; 
    }

    unsigned int * ioc_address = (unsigned int *) &seg_ioc_base ;

    unsigned int block_size = ioc_address[BLOCK_DEVICE_BLOCK_SIZE];
    unsigned int length = count * block_size;

    // get user space page table virtual address
    unsigned int user_pt_vbase = _get_context_slot(CTX_PTAB_ID);

    user_vpn_min = user_vaddr >> 12;
    user_vpn_max = (user_vaddr + length - 1) >> 12;

    // loop on all virtual pages covering the user buffer
    for (vpn = user_vpn_min, ix2 = 0 ; 
         vpn <= user_vpn_max ; 
         vpn++, ix2++ ) 
    {
        // get ppn and flags for each vpn
        unsigned int ko = _v2p_translate((page_table_t *) user_pt_vbase,
                                          vpn,
                                          &ppn,
                                          &flags);
        // check access rights
        if (ko)
        {
            _get_lock(&_tty_put_lock);
            _puts("[GIET ERROR] in _ioc_access() : user buffer unmapped\n");
            _release_lock(&_tty_put_lock);
            return 1; 
        }
        if ((flags & PTE_U) == 0) 
        {
            _get_lock(&_tty_put_lock);
            _puts("[GIET ERROR] in _ioc_access() : user buffer not in user space\n");
            _release_lock(&_tty_put_lock);
            return 1; 
        }
        if (((flags & PTE_W) == 0 ) && to_mem)
        {
            _get_lock(&_tty_put_lock);
            _puts("[GIET ERROR] in _ioc_access() : user buffer not writable\n");
            _release_lock(&_tty_put_lock);
            return 1; 
        }

        // save first ppn value
        if (ix2 == 0) ppn_first = ppn;

        if ( GIET_USE_IOMMU && USE_IOB ) // user buffer remapped in the I/0 space
        {
            // check buffer length < 2 Mbytes
            if (ix2 > 511) 
            {
                _get_lock(&_tty_put_lock);
                _puts("[GIET ERROR] in _ioc_access() : user buffer > 2 Mbytes\n");
                _release_lock(&_tty_put_lock);
                return 1; 
            }

            // map the physical page in IOMMU page table
            _iommu_add_pte2( _ioc_iommu_ix1,    // PT1 index
                             ix2,               // PT2 index
                             ppn,               // Physical page number    
                             flags);            // Protection flags

            // compute user buffer virtual adress in IO space
            buf_xaddr = (_ioc_iommu_ix1) << 21 | (user_vaddr & 0xFFF);
        }
        else            // No IOMMU
        {
            // check that physical pages are contiguous
            if ((ppn - ppn_first) != ix2) 
            {
                _get_lock(&_tty_put_lock);
                _puts("[GIET ERROR] in _ioc_access() : split physical user buffer\n");
                _release_lock(&_tty_put_lock);
                return 1; 
            }

            // compute user buffer physical adress
            buf_paddr = (((paddr_t)ppn_first) << 12) | (user_vaddr & 0xFFF);
        }
    } // end for vpn

    // register the number of pages to be unmapped
    _ioc_iommu_npages = (user_vpn_max - user_vpn_min) + 1;

    // invalidate local data cache in case of memory write
    if (to_mem) _dcache_buf_invalidate((void *) user_vaddr, length);

#if GIET_DEBUG_IOC_DRIVER
_get_lock(&_tty_put_lock);
_puts("\n[GIET DEBUG]  IOC_ACCESS at cycle ");
_putd( _proctime() );
_puts("\n - proc_id         = ");
_putd( _procid() );
_puts("\n - ioc_vbase       = ");
_putx( (unsigned int)ioc_address );
_puts("\n - psched_vbase    = ");
_putx( (unsigned int)_get_sched() );
_puts("\n - pt_vbase        = ");
_putx( user_pt_vbase );
_puts("\n - user_buf_vbase  = ");
_putx( user_vaddr );
_puts("\n - user_buf_length = ");
_putx( length );
_puts("\n - user_buf_paddr  = ");
_putl( buf_paddr );
_puts("\n - user_buf_xaddr  = ");
_putx( buf_xaddr );
_puts("\n");
_release_lock(&_tty_put_lock);
#endif

    // Invalidate L2 cache if IO Bridge is used
    if ( to_mem && USE_IOB ) _memc_inval( buf_paddr, length );
    
    // get the lock on ioc device 
    _get_lock(&_ioc_lock);

    // peripheral configuration  
    if ( GIET_USE_IOMMU && USE_IOB ) 
    {
        ioc_address[BLOCK_DEVICE_BUFFER] = buf_xaddr;
    }
    else
    {
        ioc_address[BLOCK_DEVICE_BUFFER]     = (unsigned int)buf_paddr;
        ioc_address[BLOCK_DEVICE_BUFFER_EXT] = (unsigned int)(buf_paddr>>32);
    }
    ioc_address[BLOCK_DEVICE_COUNT] = count;
    ioc_address[BLOCK_DEVICE_LBA] = lba;
    if (to_mem == 0) 
    {
        ioc_address[BLOCK_DEVICE_OP] = BLOCK_DEVICE_WRITE;
    }
    else 
    {
        ioc_address[BLOCK_DEVICE_OP] = BLOCK_DEVICE_READ;
    }
    return 0;
}

/////////////////////////////////////////////////////////////////////////////////
// _ioc_completed()
//
// This function checks completion of an I/O transfer and reports errors. 
// As it is a blocking call, the processor is stalled.
// If the virtual memory is activated, the pages mapped in the I/O virtual
// space are unmapped, and the IOB TLB is cleared.
// Returns 0 if success, > 0 if error.
/////////////////////////////////////////////////////////////////////////////////
unsigned int _ioc_completed() 
{
    unsigned int ret;
    unsigned int ix2;

    // busy waiting
    while (_ioc_done == 0) { asm volatile("nop"); }

#if GIET_DEBUG_IOC_DRIVER
_get_lock(&_tty_put_lock);
_puts("\n[GIET DEBUG]  IOC_COMPLETED at cycle ");
_putd( _proctime() );
_puts("\n - proc_id         = ");
_putd( _procid() );
_puts("\n");
_release_lock(&_tty_put_lock);
#endif

    // unmap the buffer from IOMMU page table if IOMMU is activated
    if ( GIET_USE_IOMMU && USE_IOB ) 
    {
        unsigned int * iob_address = (unsigned int *) &seg_iob_base;

        for (ix2 = 0; ix2 < _ioc_iommu_npages; ix2++) 
        {
            // unmap the page in IOMMU page table
            _iommu_inval_pte2(
                    _ioc_iommu_ix1, // PT1 index 
                    ix2 );          // PT2 index

            // clear IOMMU TLB
            iob_address[IOB_INVAL_PTE] = (_ioc_iommu_ix1 << 21) | (ix2 << 12); 
        }
    }

    // test IOC status 
    if ((_ioc_status != BLOCK_DEVICE_READ_SUCCESS)
            && (_ioc_status != BLOCK_DEVICE_WRITE_SUCCESS)) ret = 1; // error 
    else                                                    ret = 0; // success

    // reset synchronization variables
    _ioc_done = 0;
    asm volatile("sync");
    _ioc_lock = 0;

    return ret;
}


///////////////////////////////////////////////////////////////////////////////
//     _ioc_read()
// Transfer data from the block device to a memory buffer in user space. 
// - lba    : first block index on the block device
// - buffer : base address of the memory buffer (must be word aligned)
// - count  : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _ioc_read( unsigned int lba, 
                        void * buffer, 
                        unsigned int count) 
{
    return _ioc_access(
            1,        // read access
            lba,
            (unsigned int) buffer,
            count);
}


///////////////////////////////////////////////////////////////////////////////
//     _ioc_write()
// Transfer data from a memory buffer in user space to the block device. 
// - lba    : first block index on the block device
// - buffer : base address of the memory buffer (must be word aligned)
// - count  : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _ioc_write( unsigned int lba, 
                         const void * buffer, 
                         unsigned int count) 
{
    return _ioc_access(
            0, // write access
            lba,
            (unsigned int) buffer,
            count);
}


///////////////////////////////////////////////////////////////////////////////
//     _ioc_get_status()
// This function returns the transfert status, and acknowledge the IRQ.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _ioc_get_status(unsigned int * status) 
{
    // get IOC base address
    unsigned int * ioc_address = (unsigned int *) &seg_ioc_base;

    *status = ioc_address[BLOCK_DEVICE_STATUS]; // read status & reset IRQ 
    return 0;
}


///////////////////////////////////////////////////////////////////////////////
//     _ioc_get_block_size()
// This function returns the block_size with which the IOC has been configured.
///////////////////////////////////////////////////////////////////////////////
unsigned int _ioc_get_block_size() 
{
    // get IOC base address
    unsigned int * ioc_address = (unsigned int *) &seg_ioc_base;
    
    return  ioc_address[BLOCK_DEVICE_BLOCK_SIZE];
}


//////////////////////////////////////////////////////////////////////////////////
// VciMultiDma driver
//////////////////////////////////////////////////////////////////////////////////
// The DMA controllers are physically distributed in the clusters.
// There is  (NB_CLUSTERS * NB_DMA_CHANNELS) channels, indexed by a global index:
//        dma_id = cluster_id * NB_DMA_CHANNELS + loc_id
//
// As a DMA channel is a private ressource allocated to a task,
// there is no lock protecting exclusive access to the channel.
// The signalisation between the OS and the DMA uses the _dma_done[dma_id] 
// synchronisation variables  (set by the ISR, and reset by the OS).
// The transfer status is copied by the ISR in the _dma_status[dma_id] variables.
//////////////////////////////////////////////////////////////////////////////////
// The (virtual) base address of the associated segment is:
//
//       dma_address = seg_dma_base + cluster_id * GIET_CLUSTER_INCREMENT
//
// - seg_dma_base  must be defined in the giet_vsegs.ld file
// - GIET_CLUSTER_INCREMENT  must be defined in the giet_config.h file
////////////////////////////////////////////////////////////////////////////////

#if NB_DMA_CHANNELS > 0

// in_unckdata unsigned int            _dma_lock[NB_DMA_CHANNELS * NB_CLUSTERS] 
// = { [0 ... (NB_DMA_CHANNELS * NB_CLUSTERS) - 1] = 0 };

in_unckdata volatile unsigned int    _dma_done[NB_DMA_CHANNELS * NB_CLUSTERS] 
        = { [0 ... (NB_DMA_CHANNELS * NB_CLUSTERS) - 1] = 0 };
in_unckdata volatile unsigned int _dma_status[NB_DMA_CHANNELS * NB_CLUSTERS];
in_unckdata unsigned int _dma_iommu_ix1 = 1;
in_unckdata unsigned int _dma_iommu_npages[NB_DMA_CHANNELS * NB_CLUSTERS];
#endif

//////////////////////////////////////////////////////////////////////////////////
// _dma_reset_irq()
//////////////////////////////////////////////////////////////////////////////////
unsigned int _dma_reset_irq( unsigned int cluster_id, 
                             unsigned int channel_id) 
{
#if NB_DMA_CHANNELS > 0
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS)  return 1;
    if (channel_id >= NB_DMA_CHANNELS)  return 1; 

    // compute DMA base address
    unsigned int * dma_address = (unsigned int *) ((char *) &seg_dma_base + 
                                                   (cluster_id * GIET_CLUSTER_INCREMENT));

    dma_address[channel_id * DMA_SPAN + DMA_RESET] = 0;            
    return 0;
#else
    return -1;
#endif
}


//////////////////////////////////////////////////////////////////////////////////
// _dma_get_status()
//////////////////////////////////////////////////////////////////////////////////
unsigned int _dma_get_status( unsigned int cluster_id, 
                              unsigned int channel_id, 
                              unsigned int * status) 
{
#if NB_DMA_CHANNELS > 0
    // parameters checking 
    if (cluster_id >= NB_CLUSTERS)  return 1;
    if (channel_id >= NB_DMA_CHANNELS)  return 1;

    // compute DMA base address
    unsigned int * dma_address = (unsigned int *) ((char *) &seg_dma_base + 
                                                   (cluster_id * GIET_CLUSTER_INCREMENT));

    *status = dma_address[channel_id * DMA_SPAN + DMA_LEN];
    return 0;
#else
    return -1;
#endif
}


//////////////////////////////////////////////////////////////////////////////////
// _dma_transfer()
// Transfer data between a user buffer and a device buffer using DMA.
// Only one device type is supported: Frame Buffer (dev_type == 0)
// Arguments are:
// - dev_type     : device type.
// - to_user      : from  device buffer to user buffer when true.
// - offset       : offset (in bytes) in the device buffer.
// - user_vaddr   : virtual base address of the user buffer.
// - length       : number of bytes to be transfered.
// 
// The cluster_id and channel_id are obtained from task context (CTX_DMA_ID).
// The user buffer must be mapped in user address space and word-aligned.
// The user buffer length must be multiple of 4 bytes. 
// We compute the physical base addresses for both the device buffer
// and the user buffer before programming the DMA transfer.
// The GIET being fully static, we don't need to split the transfer in 4 Kbytes
// pages, because the user buffer is contiguous in physical space.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _dma_transfer( unsigned int dev_type,
                            unsigned int to_user,
                            unsigned int offset,
                            unsigned int user_vaddr,
                            unsigned int length ) 
{
#if NB_DMA_CHANNELS > 0
    unsigned int ko;           // unsuccessfull V2P translation
    unsigned int device_vbase; // device buffer vbase address
    unsigned int flags;        // protection flags
    unsigned int ppn;          // physical page number
    paddr_t      user_pbase;   // user buffer pbase address
    paddr_t      device_pbase; // frame buffer pbase address

    // check user buffer address and length alignment
    if ((user_vaddr & 0x3) || (length & 0x3)) 
    {
        _get_lock(&_tty_put_lock);
        _puts("\n[GIET ERROR] in _dma_transfer : user buffer not word aligned\n");
        _release_lock(&_tty_put_lock);
        return 1;
    }

    // get DMA channel and compute DMA vbase address
    unsigned int dma_id      = _get_context_slot(CTX_DMA_ID);
    if ( dma_id == 0xFFFFFFFF )
    {
        _get_lock(&_tty_put_lock);
        _puts("\n[GIET ERROR] in _dma_transfer : no DMA channel allocated\n");
        _release_lock(&_tty_put_lock);
        return 1;
    }
    unsigned int cluster_id  = dma_id / NB_DMA_CHANNELS;
    unsigned int channel_id  = dma_id % NB_DMA_CHANNELS;
    unsigned int * dma_vbase  = (unsigned int *) ((char *) &seg_dma_base + 
                                                (cluster_id * GIET_CLUSTER_INCREMENT));
    // get page table address
    unsigned int user_ptab = _get_context_slot(CTX_PTAB_ID);

    // get devic buffer virtual address, depending on peripheral type
    if (dev_type == 0) 
    {
        device_vbase = (unsigned int) &seg_fbf_base + offset;
    }
    else 
    {
        _get_lock(&_tty_put_lock);
        _puts("\n[GIET ERROR] in _dma_transfer : device type not supported\n");
        _release_lock(&_tty_put_lock);
        return 1;
    }

    // get device buffer physical address
    ko = _v2p_translate( (page_table_t*) user_ptab, 
                         (device_vbase >> 12), 
                         &ppn, 
                         &flags );
    if (ko) 
    {
        _get_lock(&_tty_put_lock);
        _puts("\n[GIET ERROR] in _dma_transfer : device buffer unmapped\n");
        _release_lock(&_tty_put_lock);
        return 1;
    }
    device_pbase = ((paddr_t)ppn << 12) | (device_vbase & 0x00000FFF);

    // Compute user buffer physical address
    ko = _v2p_translate( (page_table_t*) user_ptab, 
                         (user_vaddr >> 12), 
                         &ppn, 
                         &flags );
    if (ko) 
    {
        _get_lock(&_tty_put_lock);
        _puts("\n[GIET ERROR] in _dma_transfer() : user buffer unmapped\n");
        _release_lock(&_tty_put_lock);
        return 1;
    } 
    if ((flags & PTE_U) == 0) 
    {
        _get_lock(&_tty_put_lock);
        _puts("[GIET ERROR] in _dma_transfer() : user buffer not in user space\n");
        _release_lock(&_tty_put_lock);
        return 1; 
    }
    if (((flags & PTE_W) == 0 ) && to_user) 
    {
        _get_lock(&_tty_put_lock);
        _puts("\n[GIET ERROR] in _dma_transfer() : user buffer not writable\n");
        _release_lock(&_tty_put_lock);
        return 1;
    }
    user_pbase = (((paddr_t)ppn) << 12) | (user_vaddr & 0x00000FFF);

/*  This is a draft for IOMMU support

    // loop on all virtual pages covering the user buffer
    unsigned int user_vpn_min = user_vaddr >> 12;
    unsigned int user_vpn_max = (user_vaddr + length - 1) >> 12;
    unsigned int ix2          = 0;
    unsigned int ix1          = _dma_iommu_ix1 + dma_id;

    for ( vpn = user_vpn_min ; vpn <= user_vpn_max ; vpn++ )
    {
    // get ppn and flags for each vpn
    unsigned int ko = _v2p_translate( (page_table_t*)user_pt_vbase,
    vpn,
    &ppn,
    &flags );

    // check access rights
    if ( ko )                                 return 3;     // unmapped
    if ( (flags & PTE_U) == 0 )               return 4;     // not in user space
    if ( ( (flags & PTE_W) == 0 ) && to_user ) return 5;     // not writable

    // save first ppn value
    if ( ix2 == 0 ) ppn_first = ppn;

    if ( GIET_USE_IOMMU && USE_IOB )    // user buffer remapped in the I/0 space
    {
    // check buffer length < 2 Mbytes
    if ( ix2 > 511 ) return 2;

    // map the physical page in IOMMU page table
    _iommu_add_pte2( ix1,        // PT1 index
    ix2,        // PT2 index
    ppn,        // physical page number 
    flags );    // protection flags
    }
    else            // no IOMMU : check that physical pages are contiguous
    {
    if ( (ppn - ppn_first) != ix2 )       return 6;     // split physical buffer  
    }

    // increment page index
    ix2++;
    } // end for vpn

    // register the number of pages to be unmapped if iommu activated
    _dma_iommu_npages[dma_id] = (user_vpn_max - user_vpn_min) + 1;

*/

    // invalidate data cache in case of memory write
    if (to_user) _dcache_buf_invalidate((void *) user_vaddr, length);

// get the lock 
//  _get_lock(&_dma_lock[dma_id]);

#if GIET_DEBUG_DMA_DRIVER
_get_lock(&_tty_put_lock);
_puts("\n[GIET DEBUG] DMA TRANSFER at cycle ");
_putd( _proctime() );
_puts("\n - cluster_id       = ");
_putx( cluster_id );
_puts("\n - channel_id       = ");
_putx( channel_id );
_puts("\n - dma_vbase        = ");
_putx( (unsigned int)dma_vbase );
_puts("\n - device_buf_vbase = ");
_putx( device_vbase );
_puts("\n - device_buf_pbase = ");
_putl( device_pbase );
_puts("\n - user_buf_vbase   = ");
_putx( user_vaddr );
_puts("\n - user_buf_pbase   = ");
_putl( user_pbase );
_puts("\n");
_release_lock(&_tty_put_lock);
#endif

    // DMA configuration 
    if (to_user) 
    {
        dma_vbase[channel_id * DMA_SPAN + DMA_SRC]     = (unsigned int)(device_pbase);
        dma_vbase[channel_id * DMA_SPAN + DMA_SRC_EXT] = (unsigned int)(device_pbase>>32);
        dma_vbase[channel_id * DMA_SPAN + DMA_DST]     = (unsigned int)(user_pbase);
        dma_vbase[channel_id * DMA_SPAN + DMA_DST_EXT] = (unsigned int)(user_pbase>>32);
    }
    else 
    {
        dma_vbase[channel_id * DMA_SPAN + DMA_SRC]     = (unsigned int)(user_pbase);
        dma_vbase[channel_id * DMA_SPAN + DMA_SRC_EXT] = (unsigned int)(user_pbase>>32);
        dma_vbase[channel_id * DMA_SPAN + DMA_DST]     = (unsigned int)(device_pbase);
        dma_vbase[channel_id * DMA_SPAN + DMA_DST_EXT] = (unsigned int)(device_pbase>>32);
    }
    dma_vbase[channel_id * DMA_SPAN + DMA_LEN] = (unsigned int) length;

    return 0;

#else // NB_DMA_CHANNELS == 0
    _get_lock(&_tty_put_lock);
    _puts("\n[GIET ERROR] in _dma_transfer() : NB_DMA_CHANNELS == 0");
    _release_lock(&_tty_put_lock);
    return 1;
#endif

}  // end _dma_transfer()  


//////////////////////////////////////////////////////////////////////////////////
// _dma_completed()
// This function checks completion of a DMA transfer to or from a peripheral
// device (Frame Buffer or Multi-Nic).
// As it is a blocking call, the processor is busy waiting.
// Returns 0 if success, > 0 if error 
// (1 == read error / 2 == DMA idle error / 3 == write error)
//////////////////////////////////////////////////////////////////////////////////
unsigned int _dma_completed() 
{
#if NB_DMA_CHANNELS > 0
    unsigned int dma_id  = _get_context_slot(CTX_DMA_ID);
    unsigned int dma_ret;

    // busy waiting with a pseudo random delay between bus access
    while (_dma_done[dma_id] == 0) 
    {
        unsigned int delay = (( _proctime() ^ _procid() << 4) & 0x3F) + 1;
        asm volatile(
                "move  $3,   %0                 \n"
                "loop_nic_completed:            \n"
                "addi  $3,   $3, -1             \n"
                "bnez  $3,   loop_nic_completed \n"
                "nop                            \n"
                :
                : "r" (delay)
                : "$3"); 
    }

#if GIET_DEBUG_DMA_DRIVER
_get_lock(&_tty_put_lock);
_puts("\n[GIET DEBUG] DMA COMPLETED at cycle ");
_putd( _proctime() );
_puts("\n - cluster_id       = ");
_putx( dma_id/NB_DMA_CHANNELS );
_puts("\n - channel_id       = ");
_putx( dma_id%NB_DMA_CHANNELS );
_puts("\n");
_release_lock(&_tty_put_lock);
#endif

    // reset synchronization variables
    _dma_done[dma_id] = 0;
    dma_ret = _dma_status[dma_id];
    asm volatile("sync\n");

//    _dma_lock[dma_id] = 0;

    return dma_ret;

#else // NB_DMA_CHANNELS == 0
    return -1;
#endif

}  // end _dma_completed


//////////////////////////////////////////////////////////////////////////////////
//     VciFrameBuffer driver
//////////////////////////////////////////////////////////////////////////////////
// The vci_frame_buffer device can be accessed directly by software with memcpy(),
// or it can be accessed through a multi-channels DMA component:
//  
// The '_fb_sync_write' and '_fb_sync_read' functions use a memcpy strategy to
// implement the transfer between a data buffer (user space) and the frame
// buffer (kernel space). They are blocking until completion of the transfer.
//
// The '_fb_write()', '_fb_read()' and '_fb_completed()' functions use the 
// VciMultiDma components (distributed in the clusters) to transfer data 
// between the user buffer and the frame buffer. A DMA channel is
// allocated to each task requesting it in the mapping_info data structure.
//////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////
// _fb_sync_write()
// Transfer data from an memory buffer to the frame_buffer device using a memcpy. 
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
//////////////////////////////////////////////////////////////////////////////////

unsigned int _fb_sync_write(unsigned int offset, 
                            const void * buffer, 
                            unsigned int length) 
{
    unsigned char * fb_address = (unsigned char *) &seg_fbf_base + offset;
    memcpy((void *) fb_address, (void *) buffer, length);
    return 0;
}


//////////////////////////////////////////////////////////////////////////////////
// _fb_sync_read()
// Transfer data from the frame_buffer device to a memory buffer using a memcpy.
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _fb_sync_read( unsigned int   offset, 
                            const void*    buffer, 
                            unsigned int   length) 
{
    unsigned char* fb_address = (unsigned char *) &seg_fbf_base + offset;
    memcpy((void *) buffer, (void *) fb_address, length);
    return 0;
}


//////////////////////////////////////////////////////////////////////////////////
// _fb_write()
// Transfer data from a memory buffer to the frame_buffer device using  DMA.
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _fb_write( unsigned int   offset, 
                        const void*    buffer, 
                        unsigned int   length) 
{
    return _dma_transfer( 0,             // frame buffer
                          0,             // write 
                          offset,
                          (unsigned int) buffer,
                          length );
}


//////////////////////////////////////////////////////////////////////////////////
// _fb_read()
// Transfer data from the frame_buffer device to a memory buffer using  DMA.
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _fb_read( unsigned int   offset, 
                       const void*    buffer, 
                       unsigned int   length ) 
{
    return _dma_transfer( 0,    // frame buffer
                          1,    // read 
                          offset,
                          (unsigned int) buffer,
                          length );
}


//////////////////////////////////////////////////////////////////////////////////
// _fb_completed()
// This function checks completion of a DMA transfer to or fom the frame buffer.
// As it is a blocking call, the processor is busy waiting.
// Returns 0 if success, > 0 if error 
// (1 == read error / 2 == DMA idle error / 3 == write error)
//////////////////////////////////////////////////////////////////////////////////
unsigned int _fb_completed() 
{
    return _dma_completed();
}

//////////////////////////////////////////////////////////////////////////////////
//     VciMultiNic driver
//////////////////////////////////////////////////////////////////////////////////
// The VciMultiNic device can be accessed directly by software with memcpy(),
// or it can be accessed through a multi-channels DMA component:
//  
// The '_nic_sync_write' and '_nic_sync_read' functions use a memcpy strategy to
// implement the transfer between a data buffer (user space) and the NIC
// buffer (kernel space). They are blocking until completion of the transfer.
//
// The '_nic_write()', '_nic_read()' and '_nic_completed()' functions use the 
// VciMultiDma components (distributed in the clusters) to transfer data 
// between the user buffer and the NIC. A  NIDMA channel is allocated to each 
// task requesting it in the mapping_info data structure.
//////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////
// _nic_sync_write()
// Transfer data from an memory buffer to the NIC device using a memcpy.
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_sync_write( unsigned int   offset, 
                              const void*    buffer, 
                              unsigned int   length ) 
{
    unsigned char* nic_address = (unsigned char *) &seg_nic_base + offset;
    memcpy((void *) nic_address, (void *) buffer, length);
    return 0;
}


//////////////////////////////////////////////////////////////////////////////////
// _nic_sync_read()
// Transfer data from the NIC device to a memory buffer using a memcpy. 
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_sync_read(unsigned int offset, const void * buffer, unsigned int length) {
    unsigned char *nic_address = (unsigned char *) &seg_nic_base + offset;
    memcpy((void *) buffer, (void *) nic_address, length);
    return 0;
}


//////////////////////////////////////////////////////////////////////////////////
// _nic_write()
// Transfer data from a memory buffer to the NIC device using  DMA.
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_write(unsigned int offset, const void * buffer, unsigned int length) {
    return _dma_transfer(
            1,            // NIC
            0,            // write
            offset,
            (unsigned int) buffer,
            length );    
}


//////////////////////////////////////////////////////////////////////////////////
// _nic_read()
// Transfer data from the NIC device to a memory buffer using  DMA.
// - offset : offset (in bytes) in the frame buffer.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_read(unsigned int offset, const void * buffer, unsigned int length) {
    return _dma_transfer(
            1,            // NIC
            1,            // read
            offset,
            (unsigned int) buffer,
            length );    
}


//////////////////////////////////////////////////////////////////////////////////
// _nic_completed()
// This function checks completion of a DMA transfer to or fom a NIC channel.
// As it is a blocking call, the processor is busy waiting.
// Returns 0 if success, > 0 if error 
// (1 == read error / 2 == DMA idle error / 3 == write error)
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_completed() 
{
    return _dma_completed();
}

//////////////////////////////////////////////////////////////////////////////////
//     VciMemCache driver
//////////////////////////////////////////////////////////////////////////////////
// The VciMemCache device can be accessed through a configuration interface.
// as a set of uncached, memory mapped registers.
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// _memc_inval()
// This function invalidate all cache lines covering a memory buffer defined
// by the physical base address, and the length.
// The buffer address MSB are used to compute the cluster index.
///////////////////////////////////////////////////////////////////////////////////
void _memc_inval( paddr_t      buf_paddr,
                  unsigned int buf_length )
{
    unsigned int cluster_id    = (unsigned int)((buf_paddr>>32)/(256/NB_CLUSTERS));

    unsigned int * mmc_address = (unsigned int *) ((char *) &seg_mmc_base + 
                                                   (cluster_id * GIET_CLUSTER_INCREMENT));

    // get the lock protecting exclusive access to MEMC
    while ( mmc_address[MEMC_LOCK] ) { asm volatile("nop"); }

    // write inval arguments
    mmc_address[MEMC_ADDR_LO]    = (unsigned int)buf_paddr;
    mmc_address[MEMC_ADDR_HI]    = (unsigned int)(buf_paddr>>32);
    mmc_address[MEMC_BUF_LENGTH] = buf_length;
    mmc_address[MEMC_CMD_TYPE]   = MEMC_CMD_INVAL;

    // release the lock protecting MEMC
    mmc_address[MEMC_LOCK] = 0;
}
///////////////////////////////////////////////////////////////////////////////////
// _heap_info()
// This function returns the information associated to a heap (size and vaddr)
// It uses the global task index (CTX_GTID_ID, unique for each giet task) and the
// vspace index (CTX_VSID_ID) defined in the task context. 
///////////////////////////////////////////////////////////////////////////////////
unsigned int _heap_info( unsigned int* vaddr, 
                         unsigned int* size ) 
{
    mapping_header_t * header  = (mapping_header_t *) (&seg_mapping_base);
    mapping_task_t * tasks     = _get_task_base(header);
    mapping_vobj_t * vobjs     = _get_vobj_base(header);
    mapping_vspace_t * vspaces = _get_vspace_base(header);

    unsigned int taskid        = _get_context_slot(CTX_GTID_ID);
    unsigned int vspaceid      = _get_context_slot(CTX_VSID_ID);

    int heap_local_vobjid      = tasks[taskid].heap_vobjid;
    if (heap_local_vobjid != -1) 
    {
        unsigned int vobjheapid = heap_local_vobjid + vspaces[vspaceid].vobj_offset;
        *vaddr                  = vobjs[vobjheapid].vaddr;
        *size                   = vobjs[vobjheapid].length;
        return 0;
    }
    else 
    {
        *vaddr = 0;
        *size = 0;
        return 0;
    }
}

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

