//////////////////////////////////////////////////////////////////////////////////
// File     : stdio.h         
// Date     : 01/04/2010
// Author   : alain greiner & Joel Porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#ifndef _STDIO_H
#define _STDIO_H

// These define must be synchronised with 
// the _syscall_vector defined in file sys_handler.c

#define SYSCALL_PROCID            0x00
#define SYSCALL_PROCTIME          0x01
#define SYSCALL_TTY_WRITE         0x02
#define SYSCALL_TTY_READ          0x03
#define SYSCALL_TIMER_START       0x04
#define SYSCALL_TIMER_STOP        0x05
#define SYSCALL_FREE_06           0x06
#define SYSCALL_FREE_07           0x07
#define SYSCALL_HEAP_INFO         0x08
#define SYSCALL_LOCAL_TASK_ID     0x09
#define SYSCALL_GLOBAL_TASK_ID    0x0A
#define SYSCALL_FB_CMA_INIT       0x0B
#define SYSCALL_FB_CMA_WRITE      0x0C
#define SYSCALL_FB_CMA_STOP       0x0D
#define SYSCALL_EXIT              0x0E
#define SYSCALL_PROC_NUMBER       0x0F

#define SYSCALL_FB_SYNC_WRITE     0x10
#define SYSCALL_FB_SYNC_READ      0x11
#define SYSCALL_FREE_12           0x12
#define SYSCALL_FREE_13           0x13
#define SYSCALL_FREE_14           0x14
#define SYSCALL_FREE_15           0x15
#define SYSCALL_FREE_16           0x16
#define SYSCALL_FREE_17           0x17
#define SYSCALL_FREE_18           0x18
#define SYSCALL_CTX_SWITCH        0x19
#define SYSCALL_VOBJ_GET_VBASE    0x1A
#define SYSCALL_FREE_1B           0x1B
#define SYSCALL_NIC_CMA_START     0x1C
#define SYSCALL_NIC_CMA_STOP      0x1D
#define SYSCALL_NIC_SYNC_READ     0x1E
#define SYSCALL_NIC_SYNC_WRITE    0x1F

#define SYSCALL_FAT_OPEN          0x20
#define SYSCALL_FAT_READ          0x21
#define SYSCALL_FAT_WRITE         0x22
#define SYSCALL_FAT_LSEEK         0x23
#define SYSCALL_FAT_CLOSE         0x24

//////////////////////////////////////////////////////////////////////////////////
// This generic C function is used to implement all system calls.
// It writes the system call arguments in the proper registers,
// and tells GCC what has been modified by system call execution.
//////////////////////////////////////////////////////////////////////////////////
static inline int sys_call( int call_no,
                            int arg_0, 
                            int arg_1, 
                            int arg_2, 
                            int arg_3 ) 
{
    register int reg_no_and_output asm("v0") = call_no;
    register int reg_a0 asm("a0") = arg_0;
    register int reg_a1 asm("a1") = arg_1;
    register int reg_a2 asm("a2") = arg_2;
    register int reg_a3 asm("a3") = arg_3;

    asm volatile(
            "syscall"
            : "=r" (reg_no_and_output)  /* output argument */
            : "r" (reg_a0),             /* input arguments */
            "r" (reg_a1),
            "r" (reg_a2),
            "r" (reg_a3),
            "r" (reg_no_and_output)
            : "memory",
            /* These persistant registers will be saved on the stack by the
             * compiler only if they contain relevant data. */
            "at",
            "v1",
            "ra",
            "t0",
            "t1",
            "t2",
            "t3",
            "t4",
            "t5",
            "t6",
            "t7",
            "t8",
            "t9"
               );
    return reg_no_and_output;
}

//////////////////////////////////////////////////////////////////////////
// MIPS32 related system calls 
//////////////////////////////////////////////////////////////////////////

extern int giet_procid();

extern int giet_proctime();

//////////////////////////////////////////////////////////////////////////
// TTY device related system calls 
//////////////////////////////////////////////////////////////////////////

extern int giet_tty_putc(char byte);

extern int giet_tty_puts(char* buf);

extern int giet_tty_putw(unsigned int val);

extern int giet_tty_getc_no_irq(char* byte);

extern int giet_tty_getc(char* byte);

extern int giet_tty_gets(char* buf, unsigned int bufsize);

extern int giet_tty_getw(unsigned int* val);

extern int giet_tty_printf(char* format,...);

//////////////////////////////////////////////////////////////////////////
// TIMER device related system calls 
//////////////////////////////////////////////////////////////////////////

extern int giet_timer_start();

extern int giet_timer_stop();
 
//////////////////////////////////////////////////////////////////////////
// Frame buffer device related system calls 
//////////////////////////////////////////////////////////////////////////

extern int giet_fb_sync_read( unsigned int offset, 
                              void*        buffer, 
                              unsigned int length );

extern int giet_fb_sync_write(unsigned int offset, 
                              void*        buffer, 
                              unsigned int length);

extern int giet_fb_cma_init(  void*        buf0, 
                              void*        buf1,
                              unsigned int length);

extern int giet_fb_cma_write(unsigned int buf_id);

extern int giet_fb_cma_stop();

//////////////////////////////////////////////////////////////////////////
// Network controller related system calls 
//////////////////////////////////////////////////////////////////////////

extern int giet_nic_cma_start();

extern int giet_nic_cma_stop();

//////////////////////////////////////////////////////////////////////////
// FAT related system calls 
//////////////////////////////////////////////////////////////////////////

extern int giet_fat_open(  const char*  pathname,
                           unsigned int flags );

extern int giet_fat_read(  unsigned int fd,
                           void*        buffer,
                           unsigned int count,
                           unsigned int offset );

extern int giet_fat_write( unsigned int fd,
                           void*        buffer,
                           unsigned int count,
                           unsigned int offset );

extern int giet_fat_lseek( unsigned int fd,
                           unsigned int offset,
                           unsigned int whence );

extern int giet_fat_close( unsigned int fd );

//////////////////////////////////////////////////////////////////////////
// Miscelaneous system calls
//////////////////////////////////////////////////////////////////////////

extern int giet_vobj_get_vbase( char*         vspace_name, 
                                char*         vobj_name, 
                                unsigned int  vobj_type, 
                                unsigned int* vobj_vaddr);

extern int giet_procnumber();

extern void giet_exit();

extern int giet_context_switch();

extern int giet_proc_task_id();

extern int giet_heap_info( unsigned int* vaddr, 
                           unsigned int* size );

extern int giet_global_task_id();

extern void giet_assert( unsigned int, 
                         char* string );

extern int giet_rand();

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

