///////////////////////////////////////////////////////////////////////////////////
// File     : utils.c
// Date     : 18/10/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The utils.c and utils.h files are part of the GIET-VM nano-kernel.
// They define more or less the GIET-VM HAL (Hardware Abstraction Layer),
// and contains various utility functions, that can be used by both the 
// boot code and the kernel code.
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <mapping_info.h>
#include <utils.h>
#include <ctx_handler.h>
#include <tty_driver.h>
#include <stdarg.h>

// This global variable is allocated in the boot.c file or in kernel_init.c file
extern static_scheduler_t* _schedulers[NB_CLUSTERS * NB_PROCS_MAX];

///////////////////////////////////////////////////////////////////////////////////
// This function implements a pseudo-random delay.
// The val argument define approximately an exponentially increasing mean delay,
// and should not be larger than 32.
///////////////////////////////////////////////////////////////////////////////////
inline void _random_wait( unsigned int val )
{
    unsigned int mask  = (1<<(val&0x1F))-1;
    unsigned int delay = (_get_proctime() ^ (_get_procid()<<4)) & mask;
    asm volatile( "move  $3,   %0                 \n"
                  "loop_nic_completed:            \n"
                  "addi  $3,   $3, -1             \n"
                  "bnez  $3,   loop_nic_completed \n"
                  "nop                            \n"
                  :
                  : "r" (delay)
                  : "$3" ); 
}
///////////////////////////////////////////////////////////////////////////////////
// Copy a source memory buffer content to a dest memory buffer (size bytes)
// Code taken from MutekH.
///////////////////////////////////////////////////////////////////////////////////
inline void* _memcpy( void*        dest,     // dest buffer vbase
                      const void*  source,   // source buffer vbase
                      unsigned int size )    // bytes
{
    unsigned int*       dst = dest;
    const unsigned int* src = source;

    // word-by-word copy
    if (!((unsigned int) dst & 3) && !((unsigned int) src & 3)) 
    {
        while (size > 3) 
        {
            *dst++ = *src++;
            size -= 4;
        }
    }

    unsigned char * cdst = (unsigned char *) dst;
    unsigned char * csrc = (unsigned char *) src;

    /* byte-by-byte copy */
    while (size--) 
    {
        *cdst++ = *csrc++;
    }
    return dest;
}
//////////////////////////////////////////////////////////////////////////////////
// Fill a byte string with a byte value.
//////////////////////////////////////////////////////////////////////////////////
inline void * _memset( void*        dst, 
                       int          value, 
                       unsigned int count ) 
{
    char * a = (char *) dst;
    while (count--) 
    {
        *a++ = (char)value;
    }
    return dst;
}

//////////////////////////////////////////////////////////////////////////////////
// Processor suicide: infinite loop  
//////////////////////////////////////////////////////////////////////////////////
inline void _exit() 
{
    while (1) { asm volatile ("nop"); }
}
///////////////////////////////////////////////////////////////////////////////////
//         CP0 and CP2 registers access functions
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// Returns the value contained in CP0 SCHED register
// (virtual base address of the processor scheduler).
///////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_sched() 
{
    unsigned int ret;
    asm volatile( "mfc0    %0,   $22    \n" : "=r"(ret) );
    return ret;
}
///////////////////////////////////////////////////////////////////////////////////
// Returns PTPR register content.
///////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_mmu_ptpr() 
{
    unsigned int ret;
    asm volatile( "mfc2    %0,        $0" : "=r"(ret));
    return ret;
}
///////////////////////////////////////////////////////////////////////////////////
// Returns EPC register content.
///////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_epc() 
{
    unsigned int ret;
    asm volatile("mfc0    %0,        $14" : "=r"(ret));
    return ret;
}
///////////////////////////////////////////////////////////////////////////////////
// Returns BVAR register content.
///////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_bvar() 
{
    unsigned int ret;
    asm volatile( "mfc0    %0,        $8" : "=r"(ret));
    return ret;
}
///////////////////////////////////////////////////////////////////////////////////
// Returns CR register content.
///////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_cr() 
{
    unsigned int ret;
    asm volatile("mfc0    %0,        $13" : "=r"(ret));
    return ret;
}
///////////////////////////////////////////////////////////////////////////////////
// Returns SR register content
///////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_sr() 
{
    unsigned int ret;
    asm volatile( "mfc0    %0,        $12" : "=r"(ret));
    return ret;
}
//////////////////////////////////////////////////////////////////////////////
// This function set a new value for the CP0 status register.
//////////////////////////////////////////////////////////////////////////////
inline void _set_sr(unsigned int val) 
{
    asm volatile( "mtc0    %0,        $12" ::"r" (val));
}
//////////////////////////////////////////////////////////////////////////////////
// Returns processor index
//////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_procid() 
{
    unsigned int ret;
    asm volatile ("mfc0    %0,       $15, 1":"=r" (ret));
    return (ret & 0x3FF);
}
///////////////////////////////////////////////////////////////////////////////////
// Returns local time (32 bits value)
// boot_proctime() 
///////////////////////////////////////////////////////////////////////////////////
inline unsigned int _get_proctime() 
{
    unsigned int ret;
    asm volatile ("mfc0   %0,        $9":"=r" (ret));
    return ret;
}
///////////////////////////////////////////////////////////////////////////////////
// Returns index of the currently running task from the sheduler.
///////////////////////////////////////////////////////////////////////////////////
unsigned int _get_proc_task_id() 
{
    static_scheduler_t * psched = (static_scheduler_t *) _get_sched();
    return (unsigned int) (psched->current);
}

///////////////////////////////////////////////////////////////////////////////////
// Disables IRQs
///////////////////////////////////////////////////////////////////////////////////
inline void _it_disable() 
{
    asm volatile(
            "li      $3,        0xFFFFFFFE    \n"
            "mfc0    $4,        $12           \n"
            "and     $3,        $3, $4        \n"
            "mtc0    $3,        $12           \n"
            ::: "$3", "$4");
}
///////////////////////////////////////////////////////////////////////////////////
// Enables IRQs
///////////////////////////////////////////////////////////////////////////////////
inline void _it_enable() 
{
    asm volatile(
            "li      $3,        0x00000001    \n"
            "mfc0    $4,        $12           \n"
            "or      $3,        $3, $4        \n"
            "mtc0    $3,        $12           \n"
            ::: "$3", "$4");
}

//////////////////////////////////////////////////////////////////////////////
// This function set a new value for the MMU PTPR register.
//////////////////////////////////////////////////////////////////////////////
inline void _set_mmu_ptpr(unsigned int val) 
{
    asm volatile ("mtc2  %0, $0"::"r" (val));
}
//////////////////////////////////////////////////////////////////////////////
// This function set a new value for the MMU MODE register.
//////////////////////////////////////////////////////////////////////////////
inline void _set_mmu_mode(unsigned int val) 
{
    asm volatile ("mtc2  %0, $1"::"r" (val));
}
//////////////////////////////////////////////////////////////////////////////
// This function set a new value in CP0 SCHED register.
// (virtual base address of the processor scheduler).
//////////////////////////////////////////////////////////////////////////////
inline void _set_sched(unsigned int val) 
{
    asm volatile ("mtc0  %0, $22"::"r" (val));
}

////////////////////////////////////////////////////////////////////////////
//          Physical addressing related functions
////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////
// This function makes a physical read access to a 32 bits word in memory, 
// after a temporary DTLB de-activation and paddr extension.
////////////////////////////////////////////////////////////////////////////
inline unsigned int _physical_read( unsigned long long paddr ) 
{
    unsigned int value;
    unsigned int lsb = (unsigned int) paddr;
    unsigned int msb = (unsigned int) (paddr >> 32);

    asm volatile(
            "mfc2   $2,     $1                 \n"     /* $2 <= MMU_MODE   */
            "andi   $3,     $2,        0xb     \n"
            "mtc2   $3,     $1                 \n"     /* DTLB off         */    

            "mtc2   %2,     $24                \n"     /* PADDR_EXT <= msb */   
            "lw     %0,     0(%1)              \n"     /* value <= *paddr  */
            "mtc2   $0,     $24                \n"     /* PADDR_EXT <= 0   */   

            "mtc2   $2,     $1                 \n"     /* restore MMU_MODE */
            : "=r" (value)
            : "r" (lsb), "r" (msb)
            : "$2", "$3");
    return value;
}
////////////////////////////////////////////////////////////////////////////
// This function makes a physical write access to a 32 bits word in memory, 
// after a temporary DTLB de-activation and paddr extension.
////////////////////////////////////////////////////////////////////////////
inline void _physical_write( unsigned long long paddr, 
                      unsigned int       value ) 
{
    unsigned int lsb = (unsigned int)paddr;
    unsigned int msb = (unsigned int)(paddr >> 32);

    asm volatile(
            "mfc2   $2,     $1                 \n"     /* $2 <= MMU_MODE   */
            "andi   $3,     $2,        0xb     \n"
            "mtc2   $3,     $1                 \n"     /* DTLB off         */    

            "mtc2   %2,     $24                \n"     /* PADDR_EXT <= msb */   
            "sw     %0,     0(%1)              \n"     /* *paddr <= value  */
            "mtc2   $0,     $24                \n"     /* PADDR_EXT <= 0   */   

            "mtc2   $2,     $1                 \n"     /* restore MMU_MODE */
            :
            : "r" (value), "r" (lsb), "r" (msb)
            : "$2", "$3");
}

///////////////////////////////////////////////////////////////////////////////////
//     Locks access functions
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// Takes a lock with an ll/sc atomic access.
// A pseudo random delay is introduced before retry in case of miss
// (delay average value = 100 cycles)
///////////////////////////////////////////////////////////////////////////////////
inline void _get_lock(unsigned int * plock) 
{
    register unsigned int delay = ( _get_proctime() ^ _get_procid() << 4) & 0xFF;

    asm volatile (
            "_lock_llsc:             \n"
            "ll   $2,    0(%0)       \n" /* $2 <= _ioc_lock current value */
            "bnez $2,    _lock_delay \n" /* delay if _ioc_lock already taken */
            "li   $3,    1           \n" /* $3 <= argument for sc */
            "sc   $3,    0(%0)       \n" /* try to set _ioc_lock */
            "bnez $3,    _lock_ok    \n" /* exit if atomic */
            "_lock_delay:            \n"
            "move $4,    %1          \n" /* $4 <= delay */
            "_lock_loop:             \n"
            "addi $4,    $4,    -1   \n" /* $4 <= $4 - 1 */
            "beqz $4,    _lock_loop  \n" /* test end delay */
            "j           _lock_llsc  \n" /* retry */
            "_lock_ok:               \n"
            :
            :"r"(plock), "r"(delay)
            :"$2", "$3", "$4");
}
///////////////////////////////////////////////////////////////////////////////////
// Release a previouly taken lock.
///////////////////////////////////////////////////////////////////////////////////
inline void _release_lock(unsigned int * plock) 
{
    asm volatile ( "sync\n" ); // necessary because of the TSAR consistency model 
    *plock = 0;
}

///////////////////////////////////////////////////////////////////////////////////
// Display a string on TTY0 / used for system code debug and log.
// It does not use the TTY driver, but uses the seg_tty_base variable...
///////////////////////////////////////////////////////////////////////////////////
void _puts(char * buffer) 
{
    unsigned int n;
    for (n = 0; n < 1000; n++) 
    {
        if (buffer[n] == 0)  break; 
    }
    _tty_write( buffer, n, 0 );   // last argument is TTY channel
}

///////////////////////////////////////////////////////////////////////////////////
//           Access functions to system terminal TTY0
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// Display a 32 bits unsigned int as an hexadecimal string on TTY0 
///////////////////////////////////////////////////////////////////////////////////
void _putx(unsigned int val) 
{
    static const char HexaTab[] = "0123456789ABCDEF";
    char buf[11];
    unsigned int c;

    buf[0] = '0';
    buf[1] = 'x';
    buf[10] = 0;

    for (c = 0; c < 8; c++) 
    { 
        buf[9 - c] = HexaTab[val & 0xF];
        val = val >> 4;
    }
    _puts(buf);
}

///////////////////////////////////////////////////////////////////////////////////
// Display a 64 bits unsigned long as an hexadecimal string on TTY0 
///////////////////////////////////////////////////////////////////////////////////
void _putl(unsigned long long val)
{
    static const char HexaTab[] = "0123456789ABCDEF";
    char buf[19];
    unsigned int c;

    buf[0] = '0';
    buf[1] = 'x';
    buf[18] = 0;

    for (c = 0; c < 16; c++) 
    { 
        buf[17 - c] = HexaTab[(unsigned int)val & 0xF];
        val = val >> 4;
    }
    _puts(buf);
}

///////////////////////////////////////////////////////////////////////////////////
// Display a 32 bits unsigned int as a decimal string on TTY0
///////////////////////////////////////////////////////////////////////////////////
void _putd(unsigned int val) 
{
    static const char DecTab[] = "0123456789";
    char buf[11];
    unsigned int i;
    unsigned int first;

    buf[10] = 0;

    for (i = 0; i < 10; i++) {
        if ((val != 0) || (i == 0)) {
            buf[9 - i] = DecTab[val % 10];
            first = 9 - i;
        }
        else {
            break;
        }
        val /= 10;
    }
    _puts(&buf[first]);
}

///////////////////////////////////////////////////////////////////////////////////
// Compare two strings s1 & s2 (no more than n characters)
///////////////////////////////////////////////////////////////////////////////////
unsigned int _strncmp( const char * s1, 
                       const char * s2, 
                       unsigned int n ) 
{
    unsigned int i;
    for (i = 0; i < n; i++) 
    {
        if (s1[i] != s2[i])  return 1; 
        if (s1[i] == 0)      break;
    }
    return 0;
}

///////////////////////////////////////////////////////////////////////////////////
// Copy source string to dest string
///////////////////////////////////////////////////////////////////////////////////
char* _strcpy( char* dest, char* source )
{
    if (!dest || !source) return dest;

    while (*source)
        *(dest++) = *(source++);

    return dest;
}

///////////////////////////////////////////////////////////////////////////////////
// Invalidate all data cache lines corresponding to a memory 
// buffer (identified by an address and a size).
// TODO This should be replaced by a write to the CP2 MMU_DCACHE_INVAL
// register, to be more processor independant.
///////////////////////////////////////////////////////////////////////////////////
void _dcache_buf_invalidate( const void * buffer, 
                             unsigned int size) 
{
    unsigned int i;
    unsigned int tmp;
    unsigned int line_size;

    // compute data cache line size based on config register (bits 12:10)
    asm volatile(
                 "mfc0 %0, $16, 1" 
                 : "=r" (tmp) );
    tmp = ((tmp >> 10) & 0x7);
    line_size = 2 << tmp;

    // iterate on cache lines 
    for (i = 0; i < size; i += line_size) 
    {
        asm volatile(
                " cache %0, %1"
                : :"i" (0x11), "R" (*((unsigned char *) buffer + i)) );
    }
}

////////////////////////////////////////////////////////////////////////////////////
// This function returns the content of a context slot 
// for any task identified by the ltid argument (local task index),
// and the gpid argument (global processor index)
////////////////////////////////////////////////////////////////////////////////////
unsigned int _get_task_slot( unsigned int gpid,
                             unsigned int ltid,
                             unsigned int slot )
{
    static_scheduler_t* psched  = (static_scheduler_t*)_schedulers[gpid];
    return psched->context[ltid][slot];
}

////////////////////////////////////////////////////////////////////////////////////
// This function updates the content of a context slot 
// for any task identified by the ltid argument (local task index),
// and the gpid argument (global processor index)
////////////////////////////////////////////////////////////////////////////////////
void _set_task_slot( unsigned int gpid,
                     unsigned int ltid,
                     unsigned int slot,
                     unsigned int value )
{
    static_scheduler_t* psched  = (static_scheduler_t*)_schedulers[gpid];
    psched->context[ltid][slot] = value;
}

////////////////////////////////////////////////////////////////////////////////////
// This function returns the content of a context slot 
// for the running task (defined by the scheduler current field).
////////////////////////////////////////////////////////////////////////////////////
unsigned int _get_context_slot( unsigned int slot )
{
    static_scheduler_t* psched  = (static_scheduler_t*)_get_sched();
    unsigned int        task_id = psched->current;
    return psched->context[task_id][slot];
}

////////////////////////////////////////////////////////////////////////////////////
// This function updates the content of a context slot for the running task.
////////////////////////////////////////////////////////////////////////////////////
void _set_context_slot( unsigned int slot,
                       unsigned int value )
{
    static_scheduler_t* psched  = (static_scheduler_t*)_get_sched();
    unsigned int        task_id = psched->current;
    psched->context[task_id][slot] = value;
}

///////////////////////////////////////////////////////////////////////////////////
// This function returns the information associated to a heap (size and vaddr)
// It uses the global task index (CTX_GTID_ID, unique for each giet task) and the
// vspace index (CTX_VSID_ID) defined in the task context. 
///////////////////////////////////////////////////////////////////////////////////
unsigned int _heap_info( unsigned int* vaddr, 
                         unsigned int* size ) 
{
    mapping_header_t * header  = (mapping_header_t *) (&seg_boot_mapping_base);
    mapping_task_t * tasks     = _get_task_base(header);
    mapping_vobj_t * vobjs     = _get_vobj_base(header);
    mapping_vspace_t * vspaces = _get_vspace_base(header);

    unsigned int taskid        = _get_context_slot(CTX_GTID_ID);
    unsigned int vspaceid      = _get_context_slot(CTX_VSID_ID);

    int heap_local_vobjid      = tasks[taskid].heap_vobjid;
    if (heap_local_vobjid != -1) 
    {
        unsigned int vobjheapid = heap_local_vobjid + vspaces[vspaceid].vobj_offset;
        *vaddr                  = vobjs[vobjheapid].vaddr;
        *size                   = vobjs[vobjheapid].length;
        return 0;
    }
    else 
    {
        *vaddr = 0;
        *size = 0;
        return 0;
    }
}

/////////////////////////////////////////////////////////////////////////////
//      Access functions to mapping_info data structure
/////////////////////////////////////////////////////////////////////////////
inline mapping_cluster_t * _get_cluster_base(mapping_header_t * header) 
{
    return (mapping_cluster_t *) ((char *) header +
            MAPPING_HEADER_SIZE);
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_pseg_t * _get_pseg_base(mapping_header_t * header) 
{
    return (mapping_pseg_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters);
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_vspace_t * _get_vspace_base(mapping_header_t * header) 
{
    return (mapping_vspace_t *)  ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs);
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_vseg_t * _get_vseg_base(mapping_header_t * header)
{
    return (mapping_vseg_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces);
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_vobj_t * _get_vobj_base(mapping_header_t * header) 
{
    return (mapping_vobj_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VSEG_SIZE * header->vsegs );
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_task_t * _get_task_base(mapping_header_t * header) 
{
    return (mapping_task_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs);
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_proc_t *_get_proc_base(mapping_header_t * header) 
{
    return (mapping_proc_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_TASK_SIZE * header->tasks);
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_irq_t *_get_irq_base(mapping_header_t * header) 
{
    return (mapping_irq_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs);
}
/////////////////////////////////////////////////////////////////////////////
inline mapping_coproc_t *_get_coproc_base(mapping_header_t * header) 
{
    return (mapping_coproc_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs +
            MAPPING_IRQ_SIZE * header->irqs);
}
///////////////////////////////////////////////////////////////////////////////////
inline mapping_cp_port_t *_get_cp_port_base(mapping_header_t * header) 
{
    return (mapping_cp_port_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs +
            MAPPING_IRQ_SIZE * header->irqs +
            MAPPING_COPROC_SIZE * header->coprocs);
}
///////////////////////////////////////////////////////////////////////////////////
inline mapping_periph_t *_get_periph_base(mapping_header_t * header) 
{
    return (mapping_periph_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->clusters +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs +
            MAPPING_IRQ_SIZE * header->irqs +
            MAPPING_COPROC_SIZE * header->coprocs +
            MAPPING_CP_PORT_SIZE * header->cp_ports);
}

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

