///////////////////////////////////////////////////////////////////////////////////
// File     : xcu_driver.c
// Date     : 23/05/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The xcu_driver.c and xcu_driver.h files are part ot the GIET-VM nano-kernel.
// This driver supports the SoCLib vci_xicu, that is a vectorised interrupt
// controler supporting IPI (Inter Processor Interrupts) and integrated timers.
//
// It can exist several interrupt controller unit in the architecture 
// (one per cluster), and each one can contain several channels.
// The number of XICU channels is equal to NB_PROCS_MAX, because there is 
// one private XICU channel per processor in a cluster.
////////////////////////////////////////////////////////////////////////////////
// The virtual base address of the segment associated to the component is:
//
//      seg_xcu_base + cluster_xy * vseg_cluster_increment
//
// The seg_xcu_base and vseg_cluster_increment values must be defined 
// in giet_vsegs.ld file.
////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <xcu_driver.h>
#include <tty_driver.h>
#include <mapping_info.h>
#include <utils.h>

#if !defined(X_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(Y_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(X_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(Y_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(NB_PROCS_MAX) 
# error: You must define NB_PROCS_MAX in the hard_config.h file
#endif

#if !defined( USE_XICU )
# error: You must define USE_XICU in the hard_config.h file
#endif

////////////////////////////////////////////////////////////////////////////////
//     _xcu_set_mask()
// This function set the mask register for the XICU channel identified 
// by the cluster index and the processor index. 
// All '1' bits are set / all '0' bits are not modified.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _xcu_set_mask( unsigned int cluster_xy, 
                            unsigned int proc_id,
                            unsigned int value,
                            unsigned int irq_type ) 
{
    // parameters checking 
    unsigned int x = cluster_xy >> Y_WIDTH;
    unsigned int y = cluster_xy & ((1<<Y_WIDTH)-1);
    if (x >= X_SIZE)             return 1; 
    if (y >= Y_SIZE)             return 1; 
    if (proc_id >= NB_PROCS_MAX) return 1; 

#if USE_XICU
    unsigned int* xcu_address = (unsigned int *) ((unsigned int)&seg_xcu_base + 
                                (cluster_xy * (unsigned int)&vseg_cluster_increment));
    unsigned int func;
    if      (irq_type == IRQ_TYPE_PTI) func = XICU_MSK_PTI_ENABLE;
    else if (irq_type == IRQ_TYPE_SWI) func = XICU_MSK_WTI_ENABLE;
    else                               func = XICU_MSK_HWI_ENABLE;
    xcu_address[XICU_REG(func,proc_id)] = value;
    return 0;
#else
    _tty_get_lock( 0 );
    _puts("[GIET ERROR] _xcu_set_mask should not be used if USE_XICU is not set\n");
    _tty_release_lock( 0 );
    return 1;
#endif
}

////////////////////////////////////////////////////////////////////////////////
//     _xcu_get_index()
// This function returns the index of the highest priority (smaller index) 
// - active HWI (Hardware Interrupt), or
// - active PTI (Timer Interrupt), or
// - active SWI (Software Interrupt).
// The ICU channel is identified by the cluster index and the processor index.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _xcu_get_index( unsigned int cluster_xy, 
                             unsigned int proc_id, 
                             unsigned int * buffer) 
{
    // parameters checking 
    unsigned int x = cluster_xy >> Y_WIDTH;
    unsigned int y = cluster_xy & ((1<<Y_WIDTH)-1);
    if (x >= X_SIZE)             return 1; 
    if (y >= Y_SIZE)             return 1; 
    if (proc_id >= NB_PROCS_MAX) return 1; 

#if USE_XICU
    unsigned int* xcu_address = (unsigned int *) ((unsigned int)&seg_xcu_base + 
                                (cluster_xy * (unsigned int)&vseg_cluster_increment));

    unsigned int prio = xcu_address[XICU_REG(XICU_PRIO, proc_id)];
    unsigned int pti_ok = (prio & 0x00000001);
    unsigned int hwi_ok = (prio & 0x00000002);
    unsigned int swi_ok = (prio & 0x00000004);
    unsigned int pti_id = (prio & 0x00001F00) >> 8;
    unsigned int hwi_id = (prio & 0x001F0000) >> 16;
    unsigned int swi_id = (prio & 0x1F000000) >> 24;
    if      (pti_ok)  *buffer = pti_id;
    else if (hwi_ok)  *buffer = hwi_id;
    else if (swi_ok)  *buffer = swi_id; 
    else              *buffer = 32; 
    return 0;
#else
    _tty_get_lock( 0 );
    _puts("[GIET ERROR] _xcu_get_index should not be used if USE_XICU is not set\n");
    _tty_release_lock( 0 );
    return 1;
#endif
}

////////////////////////////////////////////////////////////////////////////////
//     _xcu_send_ipi()
// This function can be used only in an architecture using  XICU components.
// It writes the "wdata" value in the mailbox defined by the cluster index
// and the processor index.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _xcu_send_ipi( unsigned int cluster_xy,
                            unsigned int proc_id,
                            unsigned int wdata )
{ 
    // parameters checking 
    unsigned int x = cluster_xy >> Y_WIDTH;
    unsigned int y = cluster_xy & ((1<<Y_WIDTH)-1);
    if (x >= X_SIZE)             return 1; 
    if (y >= Y_SIZE)             return 1; 
    if (proc_id >= NB_PROCS_MAX) return 1; 

#if USE_XICU
    unsigned int* xcu_address = (unsigned int *) ((unsigned int)&seg_xcu_base + 
                                 (cluster_xy * (unsigned int)&vseg_cluster_increment));
    xcu_address[XICU_REG(XICU_WTI_REG, proc_id)] = wdata;
    return 0; 
#else
    _tty_get_lock( 0 );
    _puts("[GIET ERROR] _xcu_send_ipi should not be used if USE_XICU is not set\n");
    _tty_release_lock( 0 );
    return 1;
#endif
} 

////////////////////////////////////////////////////////////////////////////////
//    _xcu_timer_start()
// This function activates a timer contained in XICU by writing in the
// proper register the period value.
// Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////
unsigned int _xcu_timer_start( unsigned int cluster_xy,
                               unsigned int proc_id,
                               unsigned int period )
{
    // parameters checking 
    unsigned int x = cluster_xy >> Y_WIDTH;
    unsigned int y = cluster_xy & ((1<<Y_WIDTH)-1);
    if (x >= X_SIZE)             return 1; 
    if (y >= Y_SIZE)             return 1; 
    if (proc_id >= NB_PROCS_MAX) return 1; 

#if USE_XICU
    unsigned int* xcu_address = (unsigned int *) ((unsigned int)&seg_xcu_base + 
                                (cluster_xy * (unsigned int)&vseg_cluster_increment));
    xcu_address[XICU_REG(XICU_PTI_PER, proc_id)] = period;
    return 0;
#else
    _tty_get_lock( 0 );
    _puts("[GIET ERROR] _xcu_timer_start should not be used if USE_XICU is not set\n");
    _tty_release_lock( 0 );
    return 1;
#endif
}

//////////////////////////////////////////////////////////////////////////////
//     _xcu_timer_stop()
// This function desactivates a timer in XICU component
// by writing in the proper register.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////
unsigned int _xcu_timer_stop( unsigned int cluster_xy, 
                              unsigned int proc_id) 
{
    // parameters checking 
    unsigned int x = cluster_xy >> Y_WIDTH;
    unsigned int y = cluster_xy & ((1<<Y_WIDTH)-1);
    if (x >= X_SIZE)             return 1; 
    if (y >= Y_SIZE)             return 1; 
    if (proc_id >= NB_PROCS_MAX) return 1; 

#if USE_XICU
    unsigned int * xcu_address = (unsigned int *) ((unsigned int)&seg_xcu_base + 
                                 (cluster_xy * (unsigned int)&vseg_cluster_increment));
    xcu_address[XICU_REG(XICU_PTI_PER, proc_id)] = 0;
    return 0;
#else
    _tty_get_lock( 0 );
    _puts("[GIET ERROR] _xcu_timer_stop should not be used if USE_XICU is not set\n");
    _tty_release_lock( 0 );
    return 1;
#endif
}

//////////////////////////////////////////////////////////////////////////////
//     _xcu_timer_reset_irq()
// This function acknowlegge a timer interrupt in XICU 
// component by reading in the proper register.
// It can be used by both the isr_switch() for a "system" timer, 
// or by the _isr_timer() for an "user" timer.
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////
unsigned int _xcu_timer_reset_irq( unsigned int cluster_xy, 
                                   unsigned int proc_id ) 
{
    // parameters checking 
    unsigned int x = cluster_xy >> Y_WIDTH;
    unsigned int y = cluster_xy & ((1<<Y_WIDTH)-1);
    if (x >= X_SIZE)             return 1; 
    if (y >= Y_SIZE)             return 1; 
    if (proc_id >= NB_PROCS_MAX) return 1; 

#if USE_XICU
    unsigned int * xcu_address = (unsigned int *) ((unsigned int)&seg_xcu_base +
                                 (cluster_xy * (unsigned int)&vseg_cluster_increment));

    unsigned int bloup = xcu_address[XICU_REG(XICU_PTI_ACK, proc_id)];
    bloup++; // to avoid a warning 
    return 0;
#else
    _tty_get_lock( 0 );
    _puts("[GIET ERROR] _xcu_timer_reset_irq should not be used if USE_XICU is not set\n");
    _tty_release_lock( 0 );
    return 1;
#endif
}

//////////////////////////////////////////////////////////////////////////////
//     _xcu_timer_reset_cpt()
// This function resets a timer counter. To do so, we re-write the period
// in the proper register, what causes the count to restart.
// The period value is read from the same (TIMER_PERIOD) register,
// this is why in appearance we do nothing useful (read a value
// from a register and write this value in the same register)
// This function is called during a context switch (user or preemptive)
/////////////////////////////////////////////////////////////////////////////
unsigned int _xcu_timer_reset_cpt( unsigned int cluster_xy, 
                                   unsigned int proc_id ) 
{
    // parameters checking 
    unsigned int x = cluster_xy >> Y_WIDTH;
    unsigned int y = cluster_xy & ((1<<Y_WIDTH)-1);
    if (x >= X_SIZE)             return 1; 
    if (y >= Y_SIZE)             return 1; 
    if (proc_id >= NB_PROCS_MAX) return 1; 

#if USE_XICU
    unsigned int * xcu_address = (unsigned int *) ((unsigned int) &seg_xcu_base + 
                                 (cluster_xy * (unsigned int)&vseg_cluster_increment));

    unsigned int period = xcu_address[XICU_REG(XICU_PTI_PER, proc_id)];

    // we write 0 first because if the timer is currently running, 
    // the corresponding timer counter is not reset
    xcu_address[XICU_REG(XICU_PTI_PER, proc_id)] = 0;
    xcu_address[XICU_REG(XICU_PTI_PER, proc_id)] = period;
    return 0;
#else
    _tty_get_lock( 0 );
    _puts("[GIET ERROR] _xcu_timer_reset_irq should not be used if USE_XICU is not set\n");
    _tty_release_lock( 0 );
    return 1;
#endif
}


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

