/////////////////////////////////////////////////////////////////////////////////////////
// File     : ctx_handler.c
// Date     : 01/04/2012
// Authors  : alain greiner & joel porquet
// Copyright (c) UPMC-LIP6
/////////////////////////////////////////////////////////////////////////////////////////
// The ctx_handler.h and ctx_handler.c files are part of the GIET-VM nano-kernel.
// This code is used to support context switch when several tasks are executing
// in time multiplexing on a single processor.
// The tasks are statically allocated to a processor in the boot phase, and
// there is one private scheduler per processor. Each sheduler occupies 4K bytes, 
// and contains up to 14 task contexts (task_id is from 0 to 13).
// The task context [14] is reserved for the "idle" task that does nothing, and
// is launched by the scheduler when there is no other runable task.
/////////////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <tim_driver.h>
#include <xcu_driver.h>
#include <tty_driver.h>
#include <utils.h>
#include <ctx_handler.h>
#include <mapping_info.h>
#include <sys_handler.h>

/////////////////////////////////////////////////////////////////////////////////////////
// A task context is an array of 64 words = 256 bytes. 
// It contains copies of processor registers (when the task is preempted):
// - GPR[i], generally stored in slot (i). $0, $26 & $27 are not saved.
// - HI & LO registers
// - CP0 registers: EPC, SR, CR, BVAR 
// - CP2 registers : PTPR 
// It contains some general informations associated to the task:
// - TTY    : TTY channel global index
// - NIC    : NIC channel global index
// - CMA    : CMA channel global index
// - HBA    : HBA channel global index
// - DMA    : DMA channel local index
// - TIM    : TIM channel local index
// - PTAB   : page table virtual base address 
// - LTID   : Task local index (in scheduler)
// - VSID   : Virtual space index
// - RUN    : Task state (0 => sleeping / 1 => runnable ) 
// - TRDID  : Thread ID index (in vspace)
//
// ctx[0]<- ***|ctx[8] <- $8 |ctx[16]<- $16|ctx[24]<- $24|ctx[32]<- EPC  |ctx[40]<- TTY
// ctx[1]<- $1 |ctx[9] <- $9 |ctx[17]<- $17|ctx[25]<- $25|ctx[33]<- CR   |ctx[41]<- DMA
// ctx[2]<- $2 |ctx[10]<- $10|ctx[18]<- $18|ctx[26]<- LO |ctx[34]<- SR   |ctx[42]<- NIC
// ctx[3]<- $3 |ctx[11]<- $11|ctx[19]<- $19|ctx[27]<- HI |ctx[35]<- BVAR |ctx[43]<- TIM
// ctx[4]<- $4 |ctx[12]<- $12|ctx[20]<- $20|ctx[28]<- $28|ctx[36]<- PTAB |ctx[44]<- HBA
// ctx[5]<- $5 |ctx[13]<- $13|ctx[21]<- $21|ctx[29]<- SP |ctx[37]<- LTID |ctx[45]<- CMA 
// ctx[6]<- $6 |ctx[14]<- $14|ctx[22]<- $22|ctx[30]<- $30|ctx[38]<- VSID |ctx[46]<- GTID
// ctx[7]<- $7 |ctx[15]<- $15|ctx[23]<- $23|ctx[31]<- RA |ctx[39]<- PTPR |ctx[47]<- RUN
//
// ctx[48]<- TRDID
//////////////////////////////////////////////////////////////////////////////////////////

extern void _task_switch(unsigned int *, unsigned int *);

/////////////////////////////////////////////////////////////////////////////////
//    _ctx_switch()
// This function performs a context switch between the running task
// and  another task, using a round-robin sheduling policy between all
// tasks allocated to a given processor (static allocation).
// It selects the next runable task to resume execution. 
// If the only runable task is the current task, return without context switch.
// If there is no runable task, the scheduler switch to the default "idle" task.
//
// Implementation note
// The return address contained in $31 is saved in the current task context
// (in the ctx[31] slot), and the function actually returns to the address
// contained in the ctx[31] slot of the next task context.
/////////////////////////////////////////////////////////////////////////////////
void _ctx_switch() 
{
    // get scheduler address
    static_scheduler_t* psched = (static_scheduler_t*)_get_sched();

    // get number of tasks allocated to scheduler
    unsigned int tasks = psched->tasks;

    // get current task index
    unsigned int curr_task_id = psched->current;

    // select the next task using a round-robin policy
    unsigned int next_task_id;
    unsigned int tid;
    unsigned int found = 0;

    for (tid = curr_task_id + 1; tid < curr_task_id + 1 + tasks; tid++) 
    {
        next_task_id = tid % tasks;
        // test if the task is runable
        if ( psched->context[next_task_id][CTX_RUN_ID] ) 
        {
            found = 1;
            break;
        }
    }

    // launch "idle" task if no runable task
    if (found == 0) 
    {
        next_task_id = IDLE_TASK_INDEX;
    }

    // no switch if no change
    if (curr_task_id != next_task_id) 
    {
        unsigned int* curr_ctx_vaddr = &(psched->context[curr_task_id][0]);
        unsigned int* next_ctx_vaddr = &(psched->context[next_task_id][0]);
        unsigned int procid = _get_procid();
        unsigned int local_id = procid % NB_PROCS_MAX;
        unsigned int cluster_id = procid / NB_PROCS_MAX;

        // set current task index 
        psched->current = next_task_id;

        // reset timer counter
#if USE_XICU
        _xcu_timer_reset_cpt(cluster_id, local_id);
#else
        _timer_reset_cpt(cluster_id, local_id); 
#endif

        // makes context switch
        _task_switch(curr_ctx_vaddr, next_ctx_vaddr);

#if GIET_DEBUG_SWITCH
_tty_get_lock( 0 );
_puts("\n[GIET DEBUG] Context switch for processor ");
_putd(_get_procid());
_puts(" at cycle ");
_putd(_get_proctime());
_puts("\n");
_puts(" - tasks        = ");
_putd(tasks);
_puts("\n");
_puts(" - curr_task_id = ");
_putd( curr_task_id );
_puts("\n");
_puts(" - next_task_id = ");
_putd(next_task_id);
_puts("\n");
_tty_release_lock(  0 );
#endif

    }
} //end _ctx_switch()

/////////////////////////////////////////////////////////////////////////////////////
// This function is executed as the"idle" task when no other task can be executed 
/////////////////////////////////////////////////////////////////////////////////////
void _idle_task() 
{
    unsigned int count = GIET_IDLE_TASK_PERIOD;
    while(1)
    {
        asm volatile(
                "move   $3,   %0              \n"
                "_idle_task_loop:             \n"
                "addi   $3,   $3,   -1        \n"
                "bnez   $3,   _idle_task_loop \n"
                "nop                          \n"
                :
                : "r"(count)
                : "$3" ); 

#if GIET_IDLE_TASK_VERBOSITY == 1
        _tty_get_lock( 0 );
        _puts("\n[GIET WARNING] Processor ");
        _putd(_get_procid());
        _puts(" idle at cycle ");
        _putd(_get_proctime());
        _puts("\n");
        _tty_release_lock( 0 );
#endif

         count = GIET_IDLE_TASK_PERIOD;
    }
} // end ctx_idle()


/////////////////////////////////////////////////////////////////////////////////
// The address of this functionis used to initialise the return address
// in the "idle" task context.
/////////////////////////////////////////////////////////////////////////////////
void _ctx_eret() 
{
    asm volatile("eret");
}


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

