/**
 * \file    spi.c
 * \date    31 August 2012
 * \author  Cesar Fuguet <cesar.fuguet-tortolero@lip6.fr>
 */
#include <spi_driver.h>
#include <utils.h>

/**
 * \param   x: input value
 *
 * \return  byte-swapped value
 *
 * \brief   byte-swap a 32bit word
 */
static unsigned int bswap32(unsigned int x)
{
  unsigned int y;
  y =  (x & 0x000000ff) << 24;
  y |= (x & 0x0000ff00) <<  8;
  y |= (x & 0x00ff0000) >>  8;
  y |= (x & 0xff000000) >> 24;
  return y;
}

/**
 * \param   spi :   Initialized pointer to the SPI controller
 *
 * \brief   Wait until the SPI controller has finished a transfer
 *
 * Wait until the GO_BUSY bit of the SPI controller be deasserted
 */
static void _spi_wait_if_busy(struct spi_dev * spi)
{
    register int delay;

    while(SPI_IS_BUSY(spi))
    {
        for (delay = 0; delay < 100; delay++);
    }
}

/**
 * \param   spi : Initialized pointer to the SPI controller
 *
 * \return  void
 *
 * \brief   Init transfer of the tx registers to the selected slaves
 */
static void _spi_init_transfer(struct spi_dev * spi)
{
    unsigned int spi_ctrl = ioread32(&spi->ctrl);

    iowrite32(&spi->ctrl, spi_ctrl | SPI_CTRL_GO_BSY);
}

/**
 * \param   spi_freq    : Desired frequency for the generated clock from the SPI
 *                        controller
 * \param   sys_freq    : System clock frequency
 *
 * \brief   Calculated the value for the divider register in order to obtain the SPI
 *          desired clock frequency 
 */
static unsigned int _spi_calc_divider_value( unsigned int spi_freq ,
                                             unsigned int sys_freq )
{
    return ((sys_freq / (spi_freq * 2)) - 1);
}

void spi_put_tx(struct spi_dev * spi, unsigned char byte, int index)
{
    _spi_wait_if_busy(spi);
    {
        iowrite8(&spi->rx_tx[index % 4], byte);
        _spi_init_transfer(spi);
    }
    _spi_wait_if_busy(spi);
}

volatile unsigned char spi_get_rx(struct spi_dev * spi, int index)
{
    return ioread8(&spi->rx_tx[index % 4]);
}

unsigned int spi_get_data( struct spi_dev * spi,
                           paddr_t buffer      ,
                           unsigned int count  )
{
    unsigned int spi_ctrl0; // ctrl value before calling this function
    unsigned int spi_ctrl;
    int i;

    /*
     * Only reading of one block (512 bytes) are supported
     */
    if ( count != 512 ) return 1;

    _spi_wait_if_busy(spi);

    spi_ctrl0 = ioread32(&spi->ctrl);

#if 0
    /*
     * Read data using SPI DMA mechanism
     * Two restrictions:
     *   1. Can transfer only one block (512 bytes).
     *   2. The destination buffer must be aligned to SPI burst size (64 bytes)
     */
    if ((buffer & 0x3f) == 0)
    {
        _puts("spi_get_data(): Starting DMA transfer / count = ");
        _putx(count);
        _puts("\n");

        _puts("spi_get_data(): buffer = ");
        _putx(buffer);
        _puts("\n");

        spi->dma_base  = (buffer      ) & ((1 << 32) - 1); // 32 lsb
        spi->dma_baseh = (buffer >> 32) & ((1 << 8 ) - 1); // 8  msb
        spi->dma_count = count | SPI_DMA_COUNT_READ;

        while ( (spi->dma_count >> 1) );

        goto reset_ctrl;
    }
#endif

    /*
     * Read data without SPI DMA mechanism
     *
     * Switch to 128 bits words
     */

    spi_ctrl = (spi_ctrl0 & ~SPI_CTRL_CHAR_LEN_MASK) | 128;
    iowrite32(&spi->ctrl, spi_ctrl);

    /* 
     * Read data.
     * Four 32 bits words at each iteration (128 bits = 16 bytes)
     */
    for (i = 0; i < count/16; i++)
    {
        iowrite32(&spi->rx_tx[0], 0xffffffff);
        iowrite32(&spi->rx_tx[1], 0xffffffff);
        iowrite32(&spi->rx_tx[2], 0xffffffff);
        iowrite32(&spi->rx_tx[3], 0xffffffff);
        iowrite32(&spi->ctrl,  spi_ctrl | SPI_CTRL_GO_BSY);

        _spi_wait_if_busy(spi);

        _physical_write( buffer, bswap32(ioread32(&spi->rx_tx[3])) ); 
        buffer += 4;
        _physical_write( buffer, bswap32(ioread32(&spi->rx_tx[2])) ); 
        buffer += 4;
        _physical_write( buffer, bswap32(ioread32(&spi->rx_tx[1])) ); 
        buffer += 4;
        _physical_write( buffer, bswap32(ioread32(&spi->rx_tx[0])) ); 
        buffer += 4;
    }

//reset_ctrl:

    /* Switch back to original word size */
    iowrite32(&spi->ctrl, spi_ctrl0);

    return 0;
}

void spi_ss_assert(struct spi_dev * spi, int index)
{
    unsigned int spi_ss = ioread32(&spi->ss);

    iowrite32(&spi->ss, spi_ss | (1 << index));
}

void spi_ss_deassert(struct spi_dev * spi, int index)
{
    unsigned int spi_ss = ioread32(&spi->ss);

    iowrite32(&spi->ss, spi_ss & ~(1 << index));
}

void _spi_init ( struct spi_dev * spi,
                 int spi_freq        ,
                 int sys_freq        ,
                 int char_len        ,
                 int tx_edge         ,
                 int rx_edge         )
{
    unsigned int spi_ctrl = ioread32(&spi->ctrl);

    if      ( tx_edge == 0 ) spi_ctrl |=  SPI_CTRL_TXN_EN;
    else if ( tx_edge == 1 ) spi_ctrl &= ~SPI_CTRL_TXN_EN;
    if      ( rx_edge == 0 ) spi_ctrl |=  SPI_CTRL_RXN_EN;
    else if ( rx_edge == 1 ) spi_ctrl &= ~SPI_CTRL_RXN_EN;
    if      ( char_len > 0 ) spi_ctrl  = (spi_ctrl & ~SPI_CTRL_CHAR_LEN_MASK) |
                                         (char_len &  SPI_CTRL_CHAR_LEN_MASK);

    iowrite32(&spi->ctrl, spi_ctrl);

    if (spi_freq > 0 && sys_freq > 0)
        iowrite32(&spi->divider, _spi_calc_divider_value(spi_freq, sys_freq));
}

/*
 * vim: tabstop=4 : shiftwidth=4 : expandtab : softtabstop=4
 */
