/**
 * \file sdc_driver.h
 * \date 30 August 2012
 * \author Cesar fuguet <cesar.fuguet-tortolero@lip6.fr>
 *
 * This file defines the driver of a SD Card device using an SPI controller
 */

#ifndef SDC_DRIVER_H
#define SDC_DRIVER_H

#include <ioc_driver.h>
#include <spi_driver.h>
#include <mapping_info.h>

/**
 * \brief SD Card type definition
 */
struct sdcard_dev
{ 
    /** 
     * SPI controller pointer 
     */
    struct spi_dev * spi;

    /**
     * Block length of the SDCARD
     */
    unsigned int block_length;

    /**
     * Access pointer representing the offset in bytes used
     * to read or write in the SDCARD.
     *
     * \note this driver is for cards SDSD, therefore this offset
     *       must be multiple of the block length
     */ 
    unsigned int access_pointer;

    /**
     * Slave ID. This ID represents the number of the slave select signal
     * used in the hardware platform (SPI channel)
     */
    int    slave_id;

    /* is the card high capacity ? */
    int sdhc;
};

unsigned int _sdc_init( unsigned int channel );


unsigned int _sdc_read( unsigned int mode,
                        unsigned int lba,
                        paddr_t      buffer,
                        unsigned int count);


unsigned int _sdc_write( unsigned int mode,
                         unsigned int lba,
                         paddr_t      buffer,
                         unsigned int count);

unsigned int _sdc_get_block_size();

unsigned int _sdc_get_status( unsigned int channel ,
                              unsigned int* status );


/**
 * SD Card constants
 */

/** Number of retries after an unacknowledge command */
#define SDCARD_COMMAND_TIMEOUT      100

/** This command is a simple SD commmand */
#define SDCARD_CMD                  0

/** This is an application specific command */
#define SDCARD_ACMD                 1

/** The transmition is done in the negative edge of the clock */
#define SDCARD_TX_NEGEDGE           0

/** The transmition is done in the positive edge of the clock */
#define SDCARD_TX_POSEDGE           1

/** The reception is done in the negative edge of the clock */
#define SDCARD_RX_NEGEDGE           0

/** The reception is done in the positive edge of the clock */
#define SDCARD_RX_POSEDGE           1

/**
 * SD Card macros
 */

/** Check if the response is valid */
#define SDCARD_CHECK_R1_VALID(x)    (~x & SDCARD_R1_RSP_VALID) ? 1 : 0

/**
 * Check if there is an error in the response
 *
 * \note this macro must be used after verify that the response is
 *       valid
 */
#define SDCARD_CHECK_R1_ERROR(x)    ( x & 0x7E)                ? 1 : 0

/**
 * SD Card Response 1 (R1) format constants
 */
#define SDCARD_R1_IN_IDLE_STATE     ( 1 << 0 ) /**< \brief R1 bit 0 */
#define SDCARD_R1_ERASE_RESET       ( 1 << 1 ) /**< \brief R1 bit 1 */
#define SDCARD_R1_ILLEGAL_CMD       ( 1 << 2 ) /**< \brief R1 bit 2 */
#define SDCARD_R1_COM_CRC_ERR       ( 1 << 3 ) /**< \brief R1 bit 3 */
#define SDCARD_R1_ERASE_SEQ_ERR     ( 1 << 4 ) /**< \brief R1 bit 4 */
#define SDCARD_R1_ADDRESS_ERR       ( 1 << 5 ) /**< \brief R1 bit 5 */
#define SDCARD_R1_PARAMETER_ERR     ( 1 << 6 ) /**< \brief R1 bit 6 */
#define SDCARD_R1_RSP_VALID         ( 1 << 7 ) /**< \brief R1 bit 7 */

#endif

/*
 * vim: tabstop=4 : shiftwidth=4 : expandtab : softtabstop=4
 */
