////////////////////////////////////////////////////////////////////////////
// File     : mapping_info.h
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////
// The MAPPING_INFO data structure, used by the GIET_VM kernel contains:
// 
// 1) a description of a clusterized hardware architecture.
// The number of cluster is variable (can be one). The number of processors
// per cluster is variable (can be one). The number of peripherals per cluser
// and coprocessor per cluster is variable. The number of physical memory
// banks per cluster is variable.
//
// 2/ a description of the applications (called vspaces) to be - statically - 
// launched on the platform. The number of parallel tasks per application is
// variable (can be one). Multi-Writer/Multi-Reader communication channels
// betwwen tasks are supported. Each vspace contains a variable number 
// of virtual segments (called vsegs).
//
// 3/ the mapping directives: both tasks on processors, and software objects 
// (vobjs and vsegs) on the physical memory banks (called psegs).
// and a variable number of tasks. The number of virtual space can be one.
//
// The mapping_info data structure is organised as the concatenation of
// a fixed size header, and 11 variable size arrays:
//
// - mapping_cluster_t  cluster[]  
// - mapping_pseg_t     pseg[]       
// - mapping_vspace_t   vspace[]  
// - mapping_vseg_t     vseg[]     
// - mapping_vseg_t     vobj[]    
// - mapping_task_t     task[]  
// - mapping_proc_t     proc[]  
// - mapping_irq_t      irq[]   
// - mapping_coproc_t   coproc[]
// - mapping_cp_port_t  cp_port[] 
// - mapping_periph_t   periph[]
//
// It is intended to be stored in memory in the seg_boot_mapping segment. 
////////////////////////////////////////////////////////////////////////////

#ifndef _MAPPING_INFO_H_
#define _MAPPING_INFO_H_

#define MAPPING_HEADER_SIZE   sizeof(mapping_header_t)
#define MAPPING_CLUSTER_SIZE  sizeof(mapping_cluster_t)
#define MAPPING_VSPACE_SIZE   sizeof(mapping_vspace_t)
#define MAPPING_VSEG_SIZE     sizeof(mapping_vseg_t)
#define MAPPING_VOBJ_SIZE     sizeof(mapping_vobj_t)
#define MAPPING_PSEG_SIZE     sizeof(mapping_pseg_t)
#define MAPPING_TASK_SIZE     sizeof(mapping_task_t)
#define MAPPING_PROC_SIZE     sizeof(mapping_proc_t)
#define MAPPING_IRQ_SIZE      sizeof(mapping_irq_t)
#define MAPPING_COPROC_SIZE   sizeof(mapping_coproc_t)
#define MAPPING_CP_PORT_SIZE  sizeof(mapping_cp_port_t)

#define C_MODE_MASK  0b1000   // cacheable
#define X_MODE_MASK  0b0100   // executable
#define W_MODE_MASK  0b0010   // writable
#define U_MODE_MASK  0b0001   // user access

#define IN_MAPPING_SIGNATURE    0xDEADBEEF
#define OUT_MAPPING_SIGNATURE   0xBABEF00D

typedef unsigned long long  paddr_t;

enum vobjType 
{
    VOBJ_TYPE_ELF      = 0,  // loadable code/data object of elf files
    VOBJ_TYPE_BLOB     = 1,  // loadable blob object
    VOBJ_TYPE_PTAB     = 2,  // page table 
    VOBJ_TYPE_PERI     = 3,  // hardware component
    VOBJ_TYPE_MWMR     = 4,  // MWMR channel
    VOBJ_TYPE_LOCK     = 5,  // Lock
    VOBJ_TYPE_BUFFER   = 6,  // Any "no initialization" objects (stacks...)
    VOBJ_TYPE_BARRIER  = 7,  // Barrier
    VOBJ_TYPE_CONST    = 8,  // Constant
    VOBJ_TYPE_MEMSPACE = 9,  // Memspace (descriptor must be initialised)
    VOBJ_TYPE_SCHED    = 10, // Array of schedulers (one per cluster)
    VOBJ_TYPE_BOOT     = 11, // loader by the preloader
};

enum irqType
{
    IRQ_TYPE_HWI = 0,        // HARD in map.xml file
    IRQ_TYPE_SWI = 1,        // SOFT in map.xml file,
    IRQ_TYPE_PTI = 2,        // TIME in map.xml file,
};

enum psegType 
{
    PSEG_TYPE_RAM  = 0,
    PSEG_TYPE_ROM  = 1,      // deprecated => you should use PSEG_TYPE_PERI  
    PSEG_TYPE_PERI = 2,
};

// The enum definitions for psegType and periphType must be kept
// consistent with the definitions in the xml_driver.c file...

enum periphType 
{
    PERIPH_TYPE_CMA       = 0,
    PERIPH_TYPE_DMA       = 1,
    PERIPH_TYPE_FBF       = 2,
    PERIPH_TYPE_HBA       = 3,
    PERIPH_TYPE_ICU       = 4,
    PERIPH_TYPE_IOB       = 5,
    PERIPH_TYPE_IOC       = 6,
    PERIPH_TYPE_MMC       = 7,
    PERIPH_TYPE_MWR       = 8,
    PERIPH_TYPE_NIC       = 9,
    PERIPH_TYPE_ROM       = 10,
    PERIPH_TYPE_SIM       = 11,
    PERIPH_TYPE_TIM       = 12,
    PERIPH_TYPE_TTY       = 13,
    PERIPH_TYPE_XCU       = 14,

    PERIPH_TYPE_MAX_VALUE = 15,
};

enum periphSubtype 
{
    PERIPH_SUBTYPE_BDV       = 0,
    PERIPH_SUBTYPE_HBA       = 1,
    PERIPH_SUBTYPE_SPI       = 2,

    PERIPH_SUBTYPE_MAX_VALUE = 3,
};

enum mwmrPortDirection 
{
    PORT_TO_COPROC   = 0, // status register
    PORT_FROM_COPROC = 1, // config register
};


////////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_header_s 
{
    unsigned int signature;          // must contain MAPPING_SIGNATURE
    unsigned int x_size;             // actual number of clusters in a row
    unsigned int y_size;             // actual number of clusters in a column
    unsigned int x_width;            // number of bits to encode x coordinate 
    unsigned int y_width;            // number of bits to encode y coordinate 
    unsigned int globals;            // number of vsegs mapped in all vspaces
    unsigned int vspaces;            // number of virtual spaces
    unsigned int increment;          // vseg cluster increment for replicated periphs

    unsigned int cma_cluster;        // index of cluster containing CMA controler
    unsigned int cma_cluster_bis;    // index of cluster containing second CMA controler

    unsigned int fbf_cluster;        // index of cluster containing FBF controler
    unsigned int fbf_cluster_bis;    // index of cluster containing second FBF controler

    unsigned int hba_cluster;        // index of cluster containing HBA controler
    unsigned int hba_cluster_bis;    // index of cluster containing second HBA controler

    unsigned int iob_cluster;        // index of cluster containing IOB controler
    unsigned int iob_cluster_bis;    // index of cluster containing second IOB controler

    unsigned int ioc_cluster;        // index of cluster containing IOC controler
    unsigned int ioc_cluster_bis;    // index of cluster containing second IOC controler

    unsigned int nic_cluster;        // index of cluster containing NIC controler
    unsigned int nic_cluster_bis;    // index of cluster containing second NIC controler

    unsigned int rom_cluster;        // index of cluster containing ROM controler
    unsigned int rom_cluster_bis;    // index of cluster containing second ROM controler

    unsigned int sim_cluster;        // index of cluster containing SIM controler
    unsigned int sim_cluster_bis;    // index of cluster containing second SIM controler

    unsigned int tty_cluster;        // index of cluster containing TTY controler
    unsigned int tty_cluster_bis;    // index of cluster containing second TTY controler

    unsigned int psegs;              // total number of physical segments (for all clusters)
    unsigned int vsegs;              // total number of virtual segments (for all vspaces)
    unsigned int vobjs;              // total number of virtual objects (for all vspaces)
    unsigned int tasks;              // total number of tasks (for all vspaces)
    unsigned int procs;              // total number of procs (for all clusters)
    unsigned int irqs;               // total number of irqs (for all processors)
    unsigned int coprocs;            // total number of coprocs (for all clusters)
    unsigned int cp_ports;           // total number of cp_ports (for all coprocs)
    unsigned int periphs;            // total number of peripherals (for all clusters)

    char name[32];                   // mapping name
} mapping_header_t;


/////////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_cluster_s 
{
    unsigned int    x;               // x coordinate
    unsigned int    y;               // y coordinate

    unsigned int    psegs;           // number of psegs in cluster
    unsigned int    pseg_offset;     // index of first pseg in pseg set

    unsigned int    procs;           // number of processors in cluster
    unsigned int    proc_offset;     // index of first proc in proc set
 
    unsigned int    coprocs;         // number of coprocessors in cluster
    unsigned int    coproc_offset;   // index of first coproc in coproc set

    unsigned int    periphs;         // number of peripherals in cluster
    unsigned int    periph_offset;   // index of first coproc in coproc set
} mapping_cluster_t;


////////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_vspace_s 
{
    char            name[32];        // virtual space name
    unsigned int    start_offset;    // offset of the vobj containing start vector
    unsigned int    vsegs;           // number of vsegs in vspace
    unsigned int    vobjs;           // number of vobjs in vspace
    unsigned int    tasks;           // number of tasks in vspace
    unsigned int    vseg_offset;     // index of first vseg in vspace 
    unsigned int    vobj_offset;     // index of first vobjs in vspace
    unsigned int    task_offset;     // index of first task in vspace
} mapping_vspace_t;


//////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_vseg_s 
{
    char            name[32];        // vseg name (unique in vspace)
    unsigned int    vbase;           // base address in virtual space
    paddr_t         pbase;           // base address in physical space
    unsigned int    length;          // size (bytes)
    unsigned int    psegid;          // physical segment global index
    unsigned int    mode;            // C-X-W-U flags
    unsigned int    vobjs;           // number of vobjs in vseg
    unsigned int    vobj_offset;     // index of first vobj in vseg
    unsigned int    next_vseg;       // linked list of vsegs mapped on pseg
    char            mapped;          // mapped if non zero
    char            ident;           // identity mapping if non zero
    char            rsvd0;           // reserved 
    char            rsvd1;           // reserved
} mapping_vseg_t;


//////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_pseg_s  
{
    char            name[32];        // pseg name (unique in a cluster)
    paddr_t         base;            // base address in physical space
    paddr_t         length;          // size (bytes)
    unsigned int    type;            // RAM / PERI
    unsigned int    clusterid;       // linear index in array of clusters
    unsigned int    next_vseg;       // linked list of vsegs mapped on pseg
} mapping_pseg_t;


//////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_task_s 
{
    char            name[32];        // task name (unique in vspace)
    unsigned int    clusterid;       // linear index in array of clusters
    unsigned int    proclocid;       // processor local index (inside cluster)
    unsigned int    trdid;           // thread index in vspace
    unsigned int    stack_vobjid;    // stack vobj index in vspace
    unsigned int    heap_vobjid;     // heap vobj index in vspace
    unsigned int    startid;         // index in start_vector 
    unsigned int    use_tty;         // TTY channel required (global)
    unsigned int    use_nic;         // NIC channel required (global)
    unsigned int    use_cma;         // CMA channel required (global)
    unsigned int    use_hba;         // IOC channel required (global)
    unsigned int    use_tim;         // user timer required (local)
} mapping_task_t;


//////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_vobj_s 
{
    char            name[32];        // vobj name (unique in a vspace)
    char            binpath[64];     // path for the binary code ("*.elf")
    unsigned int    type;            // type of vobj
    unsigned int    length;          // size (bytes)
    unsigned int    align;           // required alignement (logarithm of 2)
    unsigned int    vaddr;           // virtual base addresse of the vobj
    paddr_t         paddr;           // physical base addresse of the vobj
    unsigned int    init;            // init value (used by barrier or mwmr channel)
} mapping_vobj_t;


//////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_proc_s 
{
    unsigned int    irqs;            // number of IRQs allocated to processor
    unsigned int    irq_offset;      // index of first IRQ allocated to processor
} mapping_proc_t;


/////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_irq_s 
{
    unsigned int    type;            // HWI / SWI / PTI
    unsigned int    icuid;           // IRQ Index for the ICU component
    unsigned int    isr;             // ISR Index (defined in irq_handler.h)
    unsigned int    channel;         // Channel Index (for multi-channels peripherals)
} mapping_irq_t; 


////////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_coproc_s 
{
    char            name[32];        // coprocessor name
    unsigned int    psegid;          // global pseg index 
    unsigned int    ports;           // number of MWMR ports used by coprocessor
    unsigned int    port_offset;     // index of first MWMR port used by coprocessor
} mapping_coproc_t;


/////////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_cp_port_s 
{
    unsigned int    direction;       // TO_COPROC == 0 / FROM_COPROC == 1
    unsigned int    vspaceid;        // index of the vspace containing the MWMR channel
    unsigned int    mwmr_vobjid;     // local index of the vobj containing the MWMR channel
} mapping_cp_port_t;


////////////////////////////////////////////////////////
typedef struct __attribute__((packed))  mapping_periph_s 
{
    unsigned int    type;         
    unsigned int    subtype;         // periph specialization
    unsigned int    psegid;          // pseg index in cluster
    unsigned int    channels;        // number of channels
} mapping_periph_t;


#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:

// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

