///////////////////////////////////////////////////////////////////////////////////
// File      : bdv_driver.c
// Date      : 23/05/2013
// Author    : alain greiner
// Maintainer: cesar fuguet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The bdv_driver.c and bdv_driver.h files are part ot the GIET-VM kernel.
// This driver supports the SocLib vci_block_device component, that is
// a single channel, block oriented, external storage contrôler.
//
// It can exist only one block-device controler in the architecture.
//
// The _bdv_read() and _bdv_write() functions use the _bdv_access() function,
// that is always blocking, but can be called in 4 modes:
//
// - In BOOT_PA mode, the _bdv_access() function uses a polling policy on the
//   IOC_STATUS register to detect transfer completion, as hardware interrupts
//   are not activated. This mode is used by the boot code to load the map.bin
//   file into memory.
//
// - In BOOT_VA mode, the _bdv_access() function uses a polling policy on
//   IOC_STATUS register to detect transfer completion. This mode is used by
//   the boot code to load the various .elf files into memory.
//
// - In KERNEL mode, the _bdv_access() function uses a descheduling strategy:
//   The ISR executed when transfer completes should restart the calling task.
//   There is no checking of user access right to the memory buffer. This mode
//   must be used to access IOC, for an "open" system call.
//
// - In USER mode, the _bdv_access() function uses a descheduling strategy:
//   The ISR executed when transfer completes should restart the calling task, 
//   The user access right to the memory buffer must be checked.
//   This mode must be used to access IOC, for a "read/write" system call.
//
// As the BDV component can be used by several programs running in parallel,
// the _ioc_lock variable guaranties exclusive access to the device.  The
// _bdv_read() and _bdv_write() functions use atomic LL/SC to get the lock.
//
// Finally, the memory buffer must fulfill the following conditions:
// - The buffer must be word aligned, 
// - The buffer must be mapped in user space for an user access, 
// - The buffer must be writable in case of (to_mem) access,
// - The total number of physical pages occupied by the user buffer cannot
//   be larger than 512 pages if the IOMMU is activated,
// - All physical pages occupied by the user buffer must be contiguous
//   if the IOMMU is not activated.
// An error code is returned if these conditions are not verified.
///////////////////////////////////////////////////////////////////////////////////
// The seg_ioc_base virtual base addresses must be defined in giet_vsegs.ld file.
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <ioc_driver.h>
#include <bdv_driver.h>
#include <utils.h>
#include <tty_driver.h>
#include <ctx_handler.h>

///////////////////////////////////////////////////////////////////////////////
//      _bdv_access()
// This function transfer data between a memory buffer and the block device.
// The buffer lentgth is (count*block_size) bytes.
// Arguments are:
// - to_mem     : from external storage to memory when non 0.
// - kernel     : kernel buffer with identity mapping when non 0. 
// - lba        : first block index on the external storage.
// - buf_vaddr  : virtual base address of the memory buffer.
// - count      : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
static unsigned int _bdv_access( unsigned int to_mem,
                                 unsigned int mode,
                                 unsigned int lba,
                                 paddr_t buf_paddr,
                                 unsigned int count) 
{

#if GIET_DEBUG_IOC_DRIVER
_tty_get_lock( 0 );
_puts("\n[IOC DEBUG] Enter _bdv_access() at cycle ");
_putd( _get_proctime() );
_puts(" for processor ");
_putd( _get_procid() );
_puts("\n - mode    = ");
_putd( mode );
_puts("\n - paddr   = ");
_putx( buf_paddr );
_puts("\n - sectors = ");
_putd( count );
_puts("\n - lba     = ");
_putx( lba );
_puts("\n");
_tty_release_lock( 0 );
#endif

    volatile unsigned int * ioc_address = (unsigned int *) &seg_ioc_base ;
    unsigned int error = 0;

    // get the lock protecting IOC
    _get_lock(&_ioc_lock);

    // set the _ioc_status polling variable
    _ioc_status = BLOCK_DEVICE_BUSY;

    ioc_address[BLOCK_DEVICE_BUFFER]     = (unsigned int)buf_paddr;
    ioc_address[BLOCK_DEVICE_BUFFER_EXT] = (unsigned int)(buf_paddr>>32);
    ioc_address[BLOCK_DEVICE_COUNT]      = count;
    ioc_address[BLOCK_DEVICE_LBA]        = lba;

    // There is two policies for transfer completion 
	// detection, depending on the mode argument:

    if ( (mode == IOC_BOOT_PA_MODE) ||    // We poll directly the IOC_STATUS register
         (mode == IOC_BOOT_VA_MODE) )     // as IRQs are masked.
    {
        // Launch transfert
        if (to_mem == 0) ioc_address[BLOCK_DEVICE_OP] = BLOCK_DEVICE_WRITE;
        else             ioc_address[BLOCK_DEVICE_OP] = BLOCK_DEVICE_READ;

        unsigned int status;
        if ( _bdv_get_status(0, &status) ) return 1;

        while( (status != BLOCK_DEVICE_READ_SUCCESS)  &&
               (status != BLOCK_DEVICE_READ_ERROR)    &&
               (status != BLOCK_DEVICE_WRITE_SUCCESS) &&
               (status != BLOCK_DEVICE_WRITE_ERROR) ) 
        {
            if ( _bdv_get_status(0, &status) ) return 1;

#if GIET_DEBUG_IOC_DRIVER
_tty_get_lock( 0 );
_puts("\n[IOC DEBUG] _bdv_access() : ... waiting on IOC_STATUS register ...\n");
_tty_release_lock( 0 );
#endif

        }
        // analyse status
        error = ( (status == BLOCK_DEVICE_READ_ERROR) ||
                  (status == BLOCK_DEVICE_WRITE_ERROR) );

        // release lock
        _release_lock(&_ioc_lock);      
    }
    else                           // in USER or KERNEL mode, we deschedule the task.
                                   // When the task is rescheduled by the ISR, we reset 
                                   // the _ioc_status variable, and release the lock
    {
        // We need a critical section, because we must reset the RUN bit
		// before to launch the transfer, and we want to avoid to be descheduled
		// between these two operations. 

        // Enter critical section
        _it_disable(); 
        
        // set _ioc_gtid and reset runnable 
        unsigned int ltid = _get_proc_task_id();
        unsigned int pid = _get_procid();
        _ioc_gtid = (pid<<16) + ltid;
        _set_task_slot( pid, ltid, CTX_RUN_ID, 0 );  
        
        // Launch transfert
        if (to_mem == 0) ioc_address[BLOCK_DEVICE_OP] = BLOCK_DEVICE_WRITE;
        else             ioc_address[BLOCK_DEVICE_OP] = BLOCK_DEVICE_READ;

        // deschedule task
        _ctx_switch();                      

        // analyse status
        error = ( (_ioc_status == BLOCK_DEVICE_READ_ERROR) ||
                  (_ioc_status == BLOCK_DEVICE_WRITE_ERROR) );

        // reset _ioc_status and release lock
        _ioc_status = BLOCK_DEVICE_IDLE; 
        _release_lock(&_ioc_lock);      
    }

#if GIET_DEBUG_IOC_DRIVER
_tty_get_lock( 0 );
_puts("\n[IOC DEBUG] _bdv_access completed at cycle ");
_putd( _get_proctime() );
_puts(" for processor ");
_putd( _get_procid() );
_puts(" : error = ");
_putd( (unsigned int)error );
_puts("\n");
_tty_release_lock( 0 );
#endif

    return error;
} // end _bdv_access()

///////////////////////////////////////////////////////////////////////////////
//       _bdv_init()
// This function cheks block size, and activates the IOC interrupts.
// Return 0 for success, > 0 if error
///////////////////////////////////////////////////////////////////////////////
unsigned int _bdv_init( unsigned int channel )
{
    volatile unsigned int * ioc_address = (unsigned int *) &seg_ioc_base ;
    
    if ( ioc_address[BLOCK_DEVICE_BLOCK_SIZE] != 512 )
    {
        _tty_get_lock( 0 );
        _puts("\n[GIET ERROR] in _bdv_init() : block size must be 512 bytes\n");
        _tty_release_lock( 0 );
        return 1; 
    }

    if ( channel != 0 )
    {
        _tty_get_lock( 0 );
        _puts("\n[GIET ERROR] in _bdv_init() : illegal channel\n");
        _tty_release_lock( 0 );

        return 1;
    }

    ioc_address[BLOCK_DEVICE_IRQ_ENABLE] = 1;
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
//     _bdv_read()
// Transfer data from the block device to a memory buffer. 
// - mode     : BOOT / KERNEL / USER
// - lba      : first block index on the block device
// - buffer   : base address of the memory buffer (must be word aligned)
// - count    : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _bdv_read( unsigned int mode,  
                        unsigned int lba, 
                        paddr_t      buffer, 
                        unsigned int count) 
{
    return _bdv_access( 1,        // read access
                        mode,  
                        lba,
                        buffer,
                        count );
}

///////////////////////////////////////////////////////////////////////////////
//     _bdv_write()
// Transfer data from a memory buffer to the block device. 
// - mode     : BOOT / KERNEL / USER
// - lba      : first block index on the block device
// - buffer   : base address of the memory buffer (must be word aligned)
// - count    : number of blocks to be transfered.
// Returns 0 if success, > 0 if error.
///////////////////////////////////////////////////////////////////////////////
unsigned int _bdv_write( unsigned int mode,  
                         unsigned int lba, 
                         paddr_t  buffer, 
                         unsigned int count ) 
{
    return _bdv_access( 0,        // write access
                        mode,  
                        lba,
                        buffer,
                        count );
}

///////////////////////////////////////////////////////////////////////////////
//     _bdv_get_status()
// This function returns in the status variable, the transfert status, and
// acknowledge the IRQ if the IOC controler is not busy.
// Returns 0 if success, > 0 if error
///////////////////////////////////////////////////////////////////////////////
unsigned int _bdv_get_status( unsigned int  channel,
                              unsigned int* status )
{
    if ( channel != 0 )
    {
        _tty_get_lock( 0 );
        _puts("\n[GIET ERROR] in _bdv_get_status() : illegal channel\n");
        _tty_release_lock( 0 );

        return 1;
    }

    // get IOC base address
    volatile unsigned int * ioc_address = (unsigned int *) &seg_ioc_base;
    *status = ioc_address[BLOCK_DEVICE_STATUS];

    return 0;
}

///////////////////////////////////////////////////////////////////////////////
//     _bdv_get_block_size()
// This function returns the block_size with which the IOC has been configured.
///////////////////////////////////////////////////////////////////////////////
unsigned int _bdv_get_block_size() 
{
    // get IOC base address
    volatile unsigned int * ioc_address = (unsigned int *) &seg_ioc_base;
    
    return  ioc_address[BLOCK_DEVICE_BLOCK_SIZE];
}


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

