//////////////////////////////////////////////////////////////////////////////////
// File     : stdio.c         
// Date     : 01/04/2010
// Author   : alain greiner & Joel Porquet
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The stdio.c and stdio.h files are part of the GIET_VM nano-kernel.
// This library contains all user-level functions that contain a system call
// to access protected or shared ressources.
///////////////////////////////////////////////////////////////////////////////////

#include <stdarg.h>
#include <stdio.h>
#include <giet_config.h>

////////////////////////////////////////////////////////////////////////////////////
//////////////////////  MIPS32 related system calls  ///////////////////////////////
////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////
// giet_procid()
////////////////////////////////////////////////////////////////////////////////////
// This function returns the processor identifier.
////////////////////////////////////////////////////////////////////////////////////
int giet_procid() 
{
    return sys_call( SYSCALL_PROCID,
                     0, 0, 0, 0 );
}
////////////////////////////////////////////////////////////////////////////////////
// giet_proctime()
////////////////////////////////////////////////////////////////////////////////////
// This function returns the local processor time (clock cycles since boot)
////////////////////////////////////////////////////////////////////////////////////
int giet_proctime() 
{
    return sys_call( SYSCALL_PROCTIME, 
                     0, 0, 0, 0 );
}


////////////////////////////////////////////////////////////////////////////////////
/////////////////////  TTY device related system calls /////////////////////////////
////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////
// giet_tty_putc()
////////////////////////////////////////////////////////////////////////////////////
// This function displays a single ascii character on a terminal.
// The terminal index must be defined in the task context in the boot phase.
// It doesn't use the TTY_PUT_IRQ interrupt, and the associated kernel buffer.
// - Returns 1 if the character has been written, 0 otherwise.
////////////////////////////////////////////////////////////////////////////////////
int giet_tty_putc(char byte) 
{
    return sys_call(SYSCALL_TTY_WRITE, 
                    (unsigned int)(&byte),
                    1, 
                    0xFFFFFFFF,  
                    0);
}
////////////////////////////////////////////////////////////////////////////////////
// giet_tty_puts()
////////////////////////////////////////////////////////////////////////////////////
// This function displays a string on a terminal.
// The terminal index must be defined in the task context in the boot phase.
// The string must be terminated by a NUL character.
// It doesn't use the TTY_PUT_IRQ interrupt, and the associated kernel buffer.
// - Returns the number of written characters.
////////////////////////////////////////////////////////////////////////////////////
int giet_tty_puts(char * buf) 
{
    unsigned int length = 0;
    while (buf[length] != 0) { length++; }
    return sys_call(SYSCALL_TTY_WRITE, 
                   (unsigned int)buf, 
                   length, 
                   0xFFFFFFFF, 
                   0);
}
////////////////////////////////////////////////////////////////////////////////////
// giet_tty_putw()
////////////////////////////////////////////////////////////////////////////////////
// This function displays the value of a 32-bit word with decimal characters.
// The terminal index must be defined in the task context in the boot phase.
// It doesn't use the TTY_PUT_IRQ interrupt, and the associated kernel buffer.
// Returns the number of written characters (should be equal to ten).
////////////////////////////////////////////////////////////////////////////////////
int giet_tty_putw(unsigned int val) 
{
    char buf[10];
    unsigned int i;
    for (i = 0; i < 10; i++) 
    {
        buf[9 - i] = (val % 10) + 0x30;
        val = val / 10;
    }
    return sys_call(SYSCALL_TTY_WRITE, 
                   (unsigned int)buf,
                   10,
                   0xFFFFFFFF,
                   0);
}
////////////////////////////////////////////////////////////////////////////////////
// giet_tty_getc()
////////////////////////////////////////////////////////////////////////////////////
// This blocking function fetches a single ascii character from a terminal.
// The terminal index must be defined in the task context in the boot phase.
// It uses the IRQ_GET interrupt, and the associated kernel buffer.
// - Returns 0 when completed.
////////////////////////////////////////////////////////////////////////////////////
int giet_tty_getc(char * byte) 
{
    unsigned int ret = 0;
    while (ret == 0) 
    {
        ret = sys_call(SYSCALL_TTY_READ, 
                      (unsigned int)byte,  // buffer address
                      1,                   // number of characters
                      0xFFFFFFFF,          // channel index from task context
                      0);
    }
    return 0;
}
////////////////////////////////////////////////////////////////////////////////////
// giet_tty_gets()
////////////////////////////////////////////////////////////////////////////////////
// This blocking function fetches a string from a terminal to a fixed length buffer.
// The terminal index must be defined in the task context in the boot phase.
// It uses the TTY_GET_IRQ interrupt, anf the associated kernel buffer.
// - Returns 0 when completed.
// - Up to (bufsize - 1) characters (including the non printable characters)
//   will be copied into buffer, and the string is always completed by a NUL
//   character.
// - The <LF> character is interpreted, and the function close the string with a
//   NUL character if <LF> is read.
// - The <DEL> character is interpreted, and the corresponding character(s) are
//   removed from the target buffer.
////////////////////////////////////////////////////////////////////////////////////
int giet_tty_gets( char*        buf, 
                   unsigned int bufsize) 
{
    unsigned int ret;
    unsigned char byte;
    unsigned int index = 0;

    while (index < (bufsize - 1)) 
    {
        do 
        { 
            ret = sys_call(SYSCALL_TTY_READ, 
                           (unsigned int)(&byte),
                           1,
                           0xFFFFFFFF,
                           0);
        } 
        while (ret != 1);

        if (byte == 0x0A)  /* LF */
        {
            break; 
        }
        else if ((byte == 0x7F) && (index > 0))  /* DEL */
        {
            index--; 
        }
        else 
        {
            buf[index] = byte;
            index++;
        }
    }
    buf[index] = 0;
    return 0;
}
////////////////////////////////////////////////////////////////////////////////////
// giet_tty_getw()
////////////////////////////////////////////////////////////////////////////////////
// This blocking function fetches a string of decimal characters (most
// significant digit first) to build a 32-bit unsigned integer.
// The terminal index must be defined in the task context in the boot phase.
// It uses the TTY_GET_IRQ interrupt, anf the associated kernel buffer.
// - Returns necessarily 0 when completed.
//
// - The non-blocking system function _tty_read_irq is called several times,
//   and the decimal characters are written in a 32 characters buffer until a
//   <LF> character is read.
// - The <DEL> character is interpreted, and previous characters can be
//   cancelled. All others characters are ignored.
// - When the <LF> character is received, the string is converted to an
//   unsigned int value. If the number of decimal digit is too large for the 32
//   bits range, the zero value is returned.
////////////////////////////////////////////////////////////////////////////////////
int giet_tty_getw(unsigned int * val) 
{
    unsigned char buf[32];
    unsigned char byte;
    unsigned int save = 0;
    unsigned int dec = 0;
    unsigned int done = 0;
    unsigned int overflow = 0;
    unsigned int max = 0;
    unsigned int i;
    unsigned int ret;

    while (done == 0) 
    {
        do 
        { 
            ret = sys_call(SYSCALL_TTY_READ,
                           (unsigned int)(&byte),
                           1,
                           0xFFFFFFFF,
                           0); 
        } 
        while (ret != 1);

        if ((byte > 0x2F) && (byte < 0x3A))  /* decimal character */
        {
            buf[max] = byte;
            max++;
            giet_tty_putc(byte);
        }
        else if ((byte == 0x0A))   /* LF */
        {
            done = 1;
        }
        else if (byte == 0x7F)   /* DEL */
        {
            if (max > 0) 
            {
                max--;      /* cancel the character */
                giet_tty_putc(0x08);
                giet_tty_putc(0x20);
                giet_tty_putc(0x08);
            }
        }
        if (max == 32)  /* decimal string overflow */
        {
            for (i = 0; i < max; i++) 
            {
                /* cancel the string */
                giet_tty_putc(0x08);
                giet_tty_putc(0x20);
                giet_tty_putc(0x08);
            }
            giet_tty_putc(0x30);
            *val = 0;      /* return 0 value */
            return 0;
        }
    }

    /* string conversion */
    for (i = 0; i < max; i++) 
    {
        dec = dec * 10 + (buf[i] - 0x30);
        if (dec < save)  overflow = 1; 
        save = dec;
    }

    /* check overflow */
    if (overflow == 0) 
    {
        *val = dec; /* return decimal value */
    }
    else 
    {
        for (i = 0; i < max; i++) 
        {
            /* cancel the string */
            giet_tty_putc(0x08);
            giet_tty_putc(0x20);
            giet_tty_putc(0x08);
        }
        giet_tty_putc(0x30);
        *val = 0;         /* return 0 value */
    }
    return 0;
}
////////////////////////////////////////////////////////////////////////////////////
// giet_tty_printf()
////////////////////////////////////////////////////////////////////////////////////
// This function is a simplified version of the mutek_printf() function.
// The terminal index must be defined in the calling task context.
// It doesn't use the IRQ_PUT interrupt, and the associated kernel buffer.
// Only a limited number of formats are supported:
//   - %d : signed decimal
//   - %u : unsigned decimal
//   - %x : hexadecimal
//   - %c : char
//   - %s : string
// - Returns 0 if success, > 0 if error.
////////////////////////////////////////////////////////////////////////////////////
int giet_tty_printf(char * format, ...) 
{
    va_list ap;
    va_start(ap, format);
    unsigned int ret;

    if (NB_TTY_CHANNELS == 1)
    {
        ret = sys_call(SYSCALL_TTY_LOCK, 0, 0, 0, 0); // Get TTY lock
    }

printf_text:

    while (*format) 
    {
        unsigned int i;
        for (i = 0 ; format[i] && (format[i] != '%') ; i++);
        if (i) 
        {
            ret = sys_call(SYSCALL_TTY_WRITE, 
                           (unsigned int)format,
                           i, 
                           0xFFFFFFFF,
                           0);

            if (ret != i) goto return_error;

            format += i;
        }
        if (*format == '%') 
        {
            format++;
            goto printf_arguments;
        }
    }

    if (NB_TTY_CHANNELS == 1)
    {
        ret = sys_call(SYSCALL_TTY_LOCK, 1, 0, 0, 0); // Release TTY lock
    }

    va_end(ap);
    return 0;

printf_arguments:

    {
        int val = va_arg(ap, long);
        char buf[20];
        char * pbuf;
        unsigned int len = 0;
        static const char HexaTab[] = "0123456789ABCDEF";
        unsigned int i;

        switch (*format++) {
            case ('c'):             /* char conversion */
                len = 1;
                buf[0] = val;
                pbuf = buf;
                break;
            case ('d'):             /* decimal signed integer */
                if (val < 0) 
                {
                    val = -val;
                    ret = sys_call(SYSCALL_TTY_WRITE, 
                                   (unsigned int)"-",
                                   1,
                                   0xFFFFFFFF,
                                   0);
                    if (ret != 1) goto return_error;
                }
            case ('u'):             /* decimal unsigned integer */
                for(i = 0; i < 10; i++) 
                {
                    buf[9 - i] = HexaTab[val % 10];
                    if (!(val /= 10)) break;
                }
                len =  i + 1;
                pbuf = &buf[9 - i];
                break;
            case ('x'):             /* hexadecimal integer */
                ret = sys_call(SYSCALL_TTY_WRITE,
                               (unsigned int)"0x",
                               2,
                               0xFFFFFFFF,
                               0);
                if (ret != 2) goto return_error;       /* return error */
                for(i = 0; i < 8; i++) 
                {
                    buf[7 - i] = HexaTab[val % 16U];
                    if (!(val /= 16U))  break;
                }
                len =  i + 1;
                pbuf = &buf[7 - i];
                break;
            case ('s'):             /* string */
                {
                    char * str = (char *) val;
                    while (str[len]) 
                    {
                        len++;
                    }
                    pbuf = (char *) val;
                }
                break;
            default:
                goto printf_text;
        }

        ret = sys_call(SYSCALL_TTY_WRITE, 
                       (unsigned int)pbuf,
                       len,
                       0xFFFFFFFF,
                       0);
        if (ret != len)  goto return_error;
        
        goto printf_text;
    }

return_error:
    if (NB_TTY_CHANNELS == 1)
    {
        ret = sys_call(SYSCALL_TTY_LOCK, 1, 0, 0, 0); // Release TTY lock
    }

    return 1;
}



//////////////////////////////////////////////////////////////////////////////////
/////////////////////  TIMER related system calls //////////////////////////////// 
//////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////
// giet_timer_start()
//////////////////////////////////////////////////////////////////////////////////
// This function activates the private user timer allocated to the calling task
// in the boot phase.
// - Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
int giet_timer_start() 
{
    return sys_call( SYSCALL_TIMER_START, 
                     0, 0, 0, 0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_timer_stop()
//////////////////////////////////////////////////////////////////////////////////
// This function activates the user timer allocated to the calling task.
// - Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
int giet_timer_stop() 
{
    return sys_call( SYSCALL_TIMER_STOP, 
                     0, 0, 0, 0 );
}


//////////////////////////////////////////////////////////////////////////////////
///////////////  Frame buffer device related system calls  ///////////////////////
//////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////
// giet_fb_sync_write()
//////////////////////////////////////////////////////////////////////////////////
// This blocking function use a memory copy strategy to transfer data from a
// user buffer to the frame buffer device in kernel space.
//     offset : offset (in bytes) in the frame buffer
//     buffer : base address of the memory buffer
//     length : number of bytes to be transfered
// - Returns 0 if success, > 0 if error (e.g. memory buffer not in user space).
//////////////////////////////////////////////////////////////////////////////////
int giet_fb_sync_write( unsigned int offset, 
                        void *       buffer, 
                        unsigned int length ) 
{
    return sys_call( SYSCALL_FB_SYNC_WRITE, 
                     offset, 
                     (unsigned int)buffer, 
                     length, 
                     0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_fb_sync_read()
//////////////////////////////////////////////////////////////////////////////////
// This blocking function use a memory copy strategy to transfer data from the
// frame buffer device in kernel space to an user buffer.
//     offset : offset (in bytes) in the frame buffer
//     buffer : base address of the user buffer
//     length : number of bytes to be transfered
// - Returns 0 if success, > 0 if error (e.g. memory buffer not in user space).
//////////////////////////////////////////////////////////////////////////////////
int giet_fb_sync_read( unsigned int offset, 
                       void *       buffer, 
                       unsigned int length ) 
{
    return sys_call( SYSCALL_FB_SYNC_READ, 
                     offset, 
                     (unsigned int)buffer, 
                     length, 
                     0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_fb_cma_init()
//////////////////////////////////////////////////////////////////////////////////
// This function initializes the two chbuf SRC an DST used by the CMA controller 
// and activates the CMA channel allocated to the calling task.
// - buf0   : first user buffer virtual address
// - buf0   : second user buffer virtual address
// - length : buffer size (bytes)
// - Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
int giet_fb_cma_init( void *       buf0,
                      void *       buf1,
                      unsigned int length )
{
    return sys_call( SYSCALL_FB_CMA_INIT, 
                     (unsigned int)buf0, 
                     (unsigned int)buf1, 
                     length, 
                     0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_fb_cma_write()
//////////////////////////////////////////////////////////////////////////////////
// This function set the valid status for one of the SRC user buffer.
// and reset the valid status for the DST frame buffer.
// - bufffer_id : 0 => buf0 valid is set / not 0  => buf1 valid is set
// - Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
int giet_fb_cma_write( unsigned int buffer_id )
{
    return sys_call( SYSCALL_FB_CMA_WRITE, 
                     buffer_id, 
                     0, 0, 0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_fb_cma_stop()
//////////////////////////////////////////////////////////////////////////////////
// This function desactivates the CMA channel allocated to the calling task.
// - Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
int giet_fb_cma_stop( )
{
    return sys_call( SYSCALL_FB_CMA_STOP, 
                     0, 0, 0, 0 );
}


//////////////////////////////////////////////////////////////////////////////////
/////////////////////// NIC related system calls /////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////
// giet_nic_cma_init()
//////////////////////////////////////////////////////////////////////////////////
// This function initializes the memory chbuf used by the CMA controller, 
// activates the NIC channel allocated to the calling task, and the CMA channel.
// - tx     : RX channel if 0 / TX channel if non 0
// - buf0   : first user buffer virtual address
// - buf1   : second user buffer virtual address
// - length : buffer size (bytes)
// - Returns 0 if success, > 0 if error 
//////////////////////////////////////////////////////////////////////////////////
int giet_nic_cma_start()
{
    return sys_call( SYSCALL_NIC_CMA_START,
                     0, 0, 0, 0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_nic_cma_stop()
//////////////////////////////////////////////////////////////////////////////////
// This function desactivates the NIC channel and the two CMA channels
// allocated to the calling task.
// - Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
int giet_nic_cma_stop( )
{
    return sys_call( SYSCALL_NIC_CMA_STOP,
                     0, 0, 0, 0 );
}


//////////////////////////////////////////////////////////////////////////////////
///////////////////// Miscellaneous system calls /////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// giet_assert()
///////////////////////////////////////////////////////////////////////////////////
// This function uses the giet_tty_puts() and giet_exit() system calls.
///////////////////////////////////////////////////////////////////////////////////
void giet_assert( unsigned int condition,
                  char*        string )
{
    if ( condition == 0 )
    {
        giet_tty_puts( string );
        giet_exit();
    }
}
//////////////////////////////////////////////////////////////////////////////////
// giet_vobj_get_vbase()
//////////////////////////////////////////////////////////////////////////////////
// This function writes in argument (vobj_vaddr) the virtual base address 
// of a vobj (defined in the mapping_info data structure), identified by 
// the two arguments (vspace_name and vobj_name).
// The (vobj_type) argument is redundant, and used for coherence checking.
// - Returns the address if success,  0 if error ( not defined or wrong type )
//////////////////////////////////////////////////////////////////////////////////
int giet_vobj_get_vbase( char*         vspace_name, 
                         char*         vobj_name, 
                         unsigned int  vobj_type, 
                         unsigned int* vobj_vaddr ) 
{
    return sys_call( SYSCALL_VOBJ_GET_VBASE, 
                     (unsigned int) vspace_name,
                     (unsigned int) vobj_name,
                     (unsigned int) vobj_type,
                     (unsigned int) vobj_vaddr );
}
////////////////////////////////////////////////////////////////////////////////////
// giet_proc_number()
////////////////////////////////////////////////////////////////////////////////////
// This function returns in the buffer argument the number of processors 
// in the cluster specified by the cluster_id argument.
// - Returns 0 if success, > 0 if error ( cluster index too large )
////////////////////////////////////////////////////////////////////////////////////
int giet_proc_number( unsigned int  cluster_id, 
                      unsigned int* buffer ) 
{
    return sys_call(SYSCALL_PROC_NUMBER, cluster_id, (unsigned int) buffer, 0, 0);
}
//////////////////////////////////////////////////////////////////////////////////
// giet_exit()
//////////////////////////////////////////////////////////////////////////////////
// This function stops execution of the calling task with a TTY message, 
// the user task is descheduled and becomes not runable. 
// It does not consume processor cycles anymore.
//////////////////////////////////////////////////////////////////////////////////
void giet_exit() 
{
    sys_call( SYSCALL_EXIT,
              0, 0, 0, 0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_context_switch()
//////////////////////////////////////////////////////////////////////////////////
// The user task calling this function is descheduled and
// the processor is allocated to another task.
//////////////////////////////////////////////////////////////////////////////////
int giet_context_switch() 
{
    return sys_call( SYSCALL_CTX_SWITCH,
                     0, 0, 0, 0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_proc_task_id()
//////////////////////////////////////////////////////////////////////////////////
// This functions returns the local task id.
// If processor has n tasks the local task index is ranging from 0 to n-1
//////////////////////////////////////////////////////////////////////////////////
int giet_proc_task_id() 
{
    return sys_call( SYSCALL_LOCAL_TASK_ID, 
                     0, 0, 0, 0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_heap_info()
//////////////////////////////////////////////////////////////////////////////////
// This function returns the base address and size of the current task's heap
//////////////////////////////////////////////////////////////////////////////////
int giet_heap_info( unsigned int* vaddr, 
                             unsigned int* length ) 
{
    return sys_call( SYSCALL_HEAP_INFO, 
                     (unsigned int)vaddr, 
                     (unsigned int)length, 
                     0, 0 );
}
//////////////////////////////////////////////////////////////////////////////////
// giet_global_task_id()
//////////////////////////////////////////////////////////////////////////////////
// This functions returns the global task id, which is unique in all the giet.
//////////////////////////////////////////////////////////////////////////////////
int giet_global_task_id() 
{
    return sys_call( SYSCALL_GLOBAL_TASK_ID, 
                     0, 0, 0, 0 );
}

//////////////////////////////////////////////////////////////////////////////////
// giet_thread_id()
//////////////////////////////////////////////////////////////////////////////////
// This functions returns the thread index of the current task.
//////////////////////////////////////////////////////////////////////////////////
int giet_thread_id() 
{
    return sys_call( SYSCALL_THREAD_ID, 
                     0, 0, 0, 0 );
}

///////////////////////////////////////////////////////////////////////////////////
///////////////////// FAT related system calls ////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// giet_fat_open()
///////////////////////////////////////////////////////////////////////////////////
// Open a file identified by a pathname, and contained in the system FAT.
// The read/write flags are not supported yet: no effect.
///////////////////////////////////////////////////////////////////////////////////
int giet_fat_open( const char*   pathname, 
                   unsigned int  flags )
{
    return sys_call( SYSCALL_FAT_OPEN, 
                     (unsigned int)pathname, 
                     flags,
                     0, 0 );
}
///////////////////////////////////////////////////////////////////////////////////
// giet_fat_read()
///////////////////////////////////////////////////////////////////////////////////
// Read "count" sectors from a file identified by "fd", skipping "offset"
// sectors in file, and writing into the user "buffer". 
// The user buffer base address shoulb be 64 bytes aligned.
///////////////////////////////////////////////////////////////////////////////////
// This system call specification should evolve to the UNIX specification:
// - count must be a number of bytes, with no alignment constraint on user buffer.
// - offset argument should be removed and replaced by an implicit "lseek" pointer
//   stored in the file descriptor.
// This suppose to implement a sectors cache
///////////////////////////////////////////////////////////////////////////////////
int giet_fat_read( unsigned int fd, 
                   void*        buffer, 
                   unsigned int count,
                   unsigned int offset )
{
    return sys_call( SYSCALL_FAT_READ,
                     fd, 
                     (unsigned int)buffer,
                     count, 
                     offset );
}
///////////////////////////////////////////////////////////////////////////////////
// giet_fat_write()
///////////////////////////////////////////////////////////////////////////////////
// Write "count" sectors from a file identified by "fd", skipping "offset"
// sectors in file, and reading from the user "buffer". 
// The user buffer base address shoulb be 64 bytes aligned.
///////////////////////////////////////////////////////////////////////////////////
// This system call specification should evolve to the UNIX specification:
// - count must be a number of bytes, with no alignment constraint on buffer
// - offset argument should be removed and replaced by an implicit "lseek" pointer
//   stored in the file descriptor.
// This suppose to implement a sectors cache
///////////////////////////////////////////////////////////////////////////////////
int giet_fat_write( unsigned int fd,
                    void*        buffer, 
                    unsigned int count,
                    unsigned int offset )
{
    return sys_call( SYSCALL_FAT_WRITE, 
                     fd, 
                     (unsigned int)buffer,
                     count, 
                     offset );
}
///////////////////////////////////////////////////////////////////////////////////
// giet_fat_lseek()
///////////////////////////////////////////////////////////////////////////////////
// Change the lseek file pointer value for a file identified by "fd".
///////////////////////////////////////////////////////////////////////////////////
int giet_fat_lseek( unsigned int fd, 
                    unsigned int offset, 
                    unsigned int whence)
{
    return sys_call( SYSCALL_FAT_LSEEK, 
                     fd, 
                     offset, 
                     whence, 
                     0 );
}

///////////////////////////////////////////////////////////////////////////////////
// giet_fat_fstat()
///////////////////////////////////////////////////////////////////////////////////
// Return stats of a file identified by "fd". 
// (Only the file_size in sectors for this moment)
///////////////////////////////////////////////////////////////////////////////////
int giet_fat_fstat( unsigned int fd )
{
    return sys_call( SYSCALL_FAT_FSTAT,
                     fd,
                     0, 0, 0 );
}

///////////////////////////////////////////////////////////////////////////////////
// giet_fat_close()
///////////////////////////////////////////////////////////////////////////////////
// Close a file identified by "fd".
///////////////////////////////////////////////////////////////////////////////////
int giet_fat_close( unsigned int fd )
{
    return sys_call( SYSCALL_FAT_CLOSE,
                     fd,
                     0, 0, 0 );
}


///////////////////////////////////////////////////////////////////////////////////
////////////////// Pseudo system calls (no syscall instruction) ///////////////////
///////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////
// giet_rand()
// This function returns a pseudo-random value derived from the processor cycle
// count. This value is comprised between 0 & 65535.
///////////////////////////////////////////////////////////////////////////////////
int giet_rand() 
{
    unsigned int x = sys_call(SYSCALL_PROCTIME, 0, 0, 0, 0);
    if ((x & 0xF) > 7) {
        return (x*x & 0xFFFF);
    }
    else {
        return (x*x*x & 0xFFFF);
    }
}

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

