////////////////////////////////////////////////////////////////////////////
// File     : xml_driver.c
// Date     : 04/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
////////////////////////////////////////////////////////////////////////////
// This program translate a binary file containing a MAPPING_INFO 
// data structure to an xml file.
////////////////////////////////////////////////////////////////////////////

#include  <stdlib.h>
#include  <fcntl.h>
#include  <unistd.h>
#include  <stdio.h>
#include  <string.h>
#include  <stdint.h>
#include  <mapping_info.h>

//////////////////////////////////////////////////////
void buildXml(mapping_header_t * header, FILE * fpout) {
    const char * vobj_type[] = 
    { 
        "ELF",        // binary code generated by GCC
        "BLOB",       // binary code generated by GCC
        "PTAB",       // page table 
        "PERI",       // hardware component
        "MWMR",       // MWMR channel
        "LOCK",       // Spin-Lock
        "BUFFER",     // Any "no intialiasation needed" objects (stacks...)
        "BARRIER",    // Barrier
        "CONST",      // Constant
        "MEMSPACE",   // Memspace
        "SCHED",      // Scheduler
    };

    const char * pseg_type[] = 
    {
        "RAM",
        "ROM",
        "PERI",
    };

    const char * irq_type[] =
    {
        "HARD",
        "SOFT",
    };

    const char * isr_type[] =
    {
        "ISR_DEFAULT",
        "ISR_SWITCH",
        "ISR_TTY",
        "ISR_DMA",
        "ISR_IOC",
        "ISR_TIMER",
    };

    const char * periph_type[] =
    {
        "CMA",
        "DMA",
        "FBF",
        "ICU",
        "IOB",
        "IOC",
        "MMC",
        "MWR",
        "NIC",
        "ROM",
        "SIM",
        "TIM",
        "TTY",
        "XCU",
    };

    const char * periph_subtype[] =
    {
        "BDV",
        "HBA",
        "SPI",
    };

    const char * port_direction[] =
    {
        "TO_COPROC",
        "FROM_COPROC",
    };

    const char * mode_str[] = 
    { 
        "____",
        "___U", 
        "__W_", 
        "__WU", 
        "_X__", 
        "_X_U", 
        "_XW_", 
        "_XWU", 
        "C___", 
        "C__U", 
        "C_W_", 
        "C_WU", 
        "CX__", 
        "CX_U", 
        "CXW_", 
        "CXWU", 
    };

    unsigned int vspace_id;
    unsigned int cluster_id;
    unsigned int pseg_id;
    unsigned int vseg_id;
    unsigned int vobj_id;
    unsigned int task_id;
    unsigned int proc_id;
    unsigned int irq_id;
    unsigned int coproc_id;
    unsigned int port_id;
    unsigned int periph_id;

    mapping_cluster_t * cluster;
    mapping_pseg_t * pseg;
    mapping_vspace_t * vspace;
    mapping_vseg_t * vseg;
    mapping_vobj_t * vobj;
    mapping_task_t * task;
    mapping_proc_t * proc;
    mapping_irq_t * irq;    
    mapping_coproc_t * coproc;
    mapping_cp_port_t * cp_port;
    mapping_periph_t * periph;

    // computes the base adresss for clusters array, 
    cluster = (mapping_cluster_t *)((char *) header +
            MAPPING_HEADER_SIZE);

    // computes the base adresss for psegs array, 
    pseg = (mapping_pseg_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size);

    // computes the base adresss for vspaces array, 
    vspace = (mapping_vspace_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs);

    // computes the base adresss for vsegs array, 
    vseg = (mapping_vseg_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces);

    // computes the base adresss for vobjs array, 
    vobj = (mapping_vobj_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VSEG_SIZE * header->vsegs);

    // computes the base address for tasks array 
    task = (mapping_task_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs);

    // computes the base address for procs array 
    proc = (mapping_proc_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks);

    // computes the base address for irqs array 
    irq = (mapping_irq_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs);

    // computes the base address for coprocs array 
    coproc = (mapping_coproc_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs +
            MAPPING_IRQ_SIZE * header->irqs);

    // computes the base address for cp_ports array 
    cp_port = (mapping_cp_port_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs +
            MAPPING_IRQ_SIZE * header->irqs +
            MAPPING_COPROC_SIZE * header->coprocs);

    // computes the base address for periphs array 
    periph = (mapping_periph_t *) ((char *) header +
            MAPPING_HEADER_SIZE +
            MAPPING_CLUSTER_SIZE * header->x_size * header->y_size +
            MAPPING_PSEG_SIZE * header->psegs +
            MAPPING_VSPACE_SIZE * header->vspaces +
            MAPPING_VOBJ_SIZE * header->vobjs +
            MAPPING_VSEG_SIZE * header->vsegs +
            MAPPING_TASK_SIZE * header->tasks +
            MAPPING_PROC_SIZE * header->procs +
            MAPPING_IRQ_SIZE * header->irqs +
            MAPPING_COPROC_SIZE * header->coprocs +
            MAPPING_CP_PORT_SIZE * header->cp_ports);

    ///////////////////////// header /////////////////////////////////////////////

    fprintf(fpout, "<?xml version = \"1.0\"?>\n\n");

    fprintf(fpout, "<mapping_info signature = \"0x%x\" \n", header->signature);
    fprintf(fpout, "              name      = \"%s\"   \n", header->name);
    fprintf(fpout, "              x_size    = \"%d\"   \n", header->x_size);
    fprintf(fpout, "              y_size    = \"%d\"   \n", header->y_size);
    fprintf(fpout, "              x_width   = \"%d\"   \n", header->x_width);
    fprintf(fpout, "              y_width   = \"%d\"   \n", header->y_width);
    fprintf(fpout, "              vspaces   = \"%d\"   \n", header->vspaces);
    fprintf(fpout, "              increment = \"%d\" >\n\n", header->vspaces);

    ///////////////////// clusters ///////////////////////////////////////////////

    fprintf( fpout, "    <clusterset>\n");
    for (cluster_id = 0; cluster_id < (header->x_size * header->y_size); cluster_id++) 
    {
        fprintf(fpout, "        <cluster x = \"%d\" y = \"%d\" >\n", 
                cluster[cluster_id].x, cluster[cluster_id].y );

        ///////////////////// psegs   ////////////////////////////////////////////////

        for (pseg_id = cluster[cluster_id].pseg_offset;
             pseg_id < cluster[cluster_id].pseg_offset + cluster[cluster_id].psegs;
             pseg_id++) 
        {
            fprintf(fpout, "            <pseg name = \"%s\" ", pseg[pseg_id].name);
            fprintf(fpout, "type = \"%s\" ", pseg_type[pseg[pseg_id].type]);
            fprintf(fpout, "base = \"0x%llx\" ", pseg[pseg_id].base);
            fprintf(fpout, "length = \"0x%llx\" />\n",   pseg[pseg_id].length);
        }

        ///////////////////// processors /////////////////////////////////////////////

        unsigned int proc_index = 0;
        for (proc_id = cluster[cluster_id].proc_offset;
             proc_id < cluster[cluster_id].proc_offset + cluster[cluster_id].procs;
             proc_id++) 
        {
            fprintf(fpout, "            <proc index = \"%d\" >\n", proc_index);
            for (irq_id = proc[proc_id].irq_offset; 
                 irq_id < proc[proc_id].irq_offset + proc[proc_id].irqs;
                 irq_id++) 
            {
                fprintf(fpout, "                <irq type = \"%s\" ", irq_type[irq[irq_id].type]);
                fprintf(fpout, " icuid = \"0x%x\" ", irq[irq_id].icuid);
                fprintf(fpout, " isr = \"%s\" ", isr_type[irq[irq_id].isr]);
                fprintf(fpout, " channel = \"0x%x\" />\n", irq[irq_id].channel);
            }
            fprintf(fpout, "            </proc>\n" );
        }


        ///////////////////// coprocessors ///////////////////////////////////////////

        for (coproc_id = cluster[cluster_id].coproc_offset;
             coproc_id < cluster[cluster_id].coproc_offset + cluster[cluster_id].coprocs;
             coproc_id++) 
        {
            fprintf(fpout, "            <coproc name = \"%s\" ", coproc[coproc_id].name);
            fprintf(fpout, " psegname = \"%s\" >\n", pseg[coproc[coproc_id].psegid].name);
            for (port_id = coproc[coproc_id].port_offset;
                 port_id < coproc[coproc_id].port_offset + coproc[coproc_id].ports;
                 port_id++) 
            {
                unsigned int vobj_id = cp_port[port_id].mwmr_vobjid + vspace[cp_port[port_id].vspaceid].vobj_offset; 
                fprintf(fpout, "             <port direction = \"%s\" ",  port_direction[cp_port[port_id].direction]);
                fprintf(fpout, " vspacename = \"%s\" ", vspace[cp_port[port_id].vspaceid].name);
                fprintf(fpout, " vobjname = \"%s\" />\n",  vobj[vobj_id].name);
            }
            fprintf(fpout, "            </coproc>\n" );
        }

        ///////////////////// periphs  ///////////////////////////////////////////////

        for (periph_id = cluster[cluster_id].periph_offset;
             periph_id < cluster[cluster_id].periph_offset + cluster[cluster_id].periphs;
             periph_id++) 
        {
            fprintf(fpout, "            <periph type = \"%s\" ", periph_type[periph[periph_id].type]);

            if (periph[periph_id].subtype < PERIPH_SUBTYPE_MAX_VALUE) 
                fprintf(fpout, " subtype = \"%s\" ", periph_subtype[periph[periph_id].subtype]);

            fprintf(fpout, " psegname = \"%s\" ", pseg[periph[periph_id].psegid].name);
            fprintf(fpout, " channels = \"%d\" />\n",  periph[periph_id].channels);
        }
        fprintf(fpout, "        </cluster>\n" );
    }
    fprintf(fpout, "    </clusterset>\n\n" );

    /////////////////// globals /////////////////////////////////////////////////

    fprintf(fpout, "    <globalset>\n" );
    for (vseg_id = 0; vseg_id < header->globals; vseg_id++) 
    {
        unsigned int pseg_id = vseg[vseg_id].psegid; 

        fprintf(fpout, "        <vseg name = \"%s\" ", vseg[vseg_id].name);
        fprintf(fpout, "vbase = \"0x%x\" ", vseg[vseg_id].vbase);
        fprintf(fpout, "mode  = \"%s\" ", mode_str[vseg[vseg_id].mode]);
        fprintf(fpout, "clusterid = \"%d\" ", pseg[pseg_id].clusterid);
        fprintf(fpout, "psegname = \"%s\" ", pseg[pseg_id].name);
        fprintf(fpout, "ident = \"%d\" >\n", vseg[vseg_id].ident);
        for (vobj_id = vseg[vseg_id].vobj_offset;
             vobj_id < (vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs); 
             vobj_id++) 
        {
            fprintf(fpout, "            <vobj name = \"%s\" ", vobj[vobj_id].name);
            fprintf(fpout, "type = \"%s\" ", vobj_type[vobj[vobj_id].type]);
            fprintf(fpout, "length = \"0x%x\" ", vobj[vobj_id].length);
            fprintf(fpout, "align = \"%d\" ", vobj[vobj_id].align);
            fprintf(fpout, "init = \"%d\" ", vobj[vobj_id].init);
            fprintf(fpout, "binpath = \"%s\" />\n", vobj[vobj_id].binpath);
        }
        fprintf(fpout, "        </vseg>\n");
    }
    fprintf(fpout, "    </globalset>\n" );

    //////////////////// vspaces ////////////////////////////////////////////////

    fprintf( fpout, "\n    <vspaceset>\n\n" );
    for (vspace_id = 0; vspace_id < header->vspaces; vspace_id++) 
    {
        unsigned int func_id = vspace[vspace_id].vobj_offset + vspace[vspace_id].start_offset; 
        fprintf(fpout, "        <vspace name = \"%s\" ", vspace[vspace_id].name); 
        fprintf(fpout, " startname = \"%s\" >\n", vobj[func_id].name); 

        //////////////////// vsegs //////////////////////////////////////////////

        for (vseg_id = vspace[vspace_id].vseg_offset;
             vseg_id < (vspace[vspace_id].vseg_offset + vspace[vspace_id].vsegs);
             vseg_id++) 
        {
            unsigned int pseg_id    = vseg[vseg_id].psegid; 
            unsigned int cluster_id = pseg[pseg_id].clusterid;
            unsigned int x          = cluster_id >> header->y_width;
            unsigned int y          = cluster_id & ((1<<header->y_width)-1);

            fprintf(fpout, "            <vseg name = \"%s\" ", vseg[vseg_id].name);
            fprintf(fpout, "vbase = \"0x%x\" ", vseg[vseg_id].vbase);
            fprintf(fpout, "mode = \"%s\" ", mode_str[vseg[vseg_id].mode]);
            fprintf(fpout, "x = \"%d\" ", x);
            fprintf(fpout, "y = \"%d\" ", y);
            fprintf(fpout, "psegname = \"%s\" ", pseg[pseg_id].name);
            fprintf(fpout, "ident = \"%d\" >\n", vseg[vseg_id].ident);

            for (vobj_id = vseg[vseg_id].vobj_offset;
                 vobj_id < (vseg[vseg_id].vobj_offset + vseg[vseg_id].vobjs);
                 vobj_id++) 
            {
                fprintf(fpout, "             <vobj name = \"%s\" ", vobj[vobj_id].name);
                fprintf(fpout, "type = \"%s\" ", vobj_type[vobj[vobj_id].type]);
                fprintf(fpout, "length = \"0x%x\" ", vobj[vobj_id].length);
                fprintf(fpout, "align = \"%d\" ", vobj[vobj_id].align);
                fprintf(fpout, "init = \"%d\" ", vobj[vobj_id].init);
                fprintf(fpout, "binpath = \"%s\" />\n", vobj[vobj_id].binpath);
            }
            fprintf(fpout, "            </vseg>\n\n");
        }

        //////////////////// tasks //////////////////////////////////////////////

        for (task_id = vspace[vspace_id].task_offset;
             task_id < (vspace[vspace_id].task_offset + vspace[vspace_id].tasks);
             task_id++) 
        {
            unsigned int stack_vobj_id = task[task_id].stack_vobjid + vspace[vspace_id].vobj_offset; 
            unsigned int heap_vobj_id = task[task_id].heap_vobjid + vspace[vspace_id].vobj_offset; 
            unsigned int cluster_id   = task[task_id].clusterid;
            unsigned int x            = cluster_id >> header->y_width;
            unsigned int y            = cluster_id & ((1<<header->y_width)-1);

            fprintf(fpout, "            <task name = \"%s\" ", task[task_id].name);
            fprintf(fpout, "x = \"%d\" ", x);
            fprintf(fpout, "y = \"%d\" ", y);
            fprintf(fpout, "proclocid = \"%d\" ", task[task_id].proclocid);
            fprintf(fpout, "stackname = \"%s\" ", vobj[stack_vobj_id].name);
            if (heap_vobj_id != -1) 
            {
                fprintf(fpout, "heapname = \"%s\" ", vobj[heap_vobj_id].name);
            }
            fprintf(fpout, "startid = \"%d\" ", task[task_id].startid);
            fprintf(fpout, "usetty = \"%d\" ", task[task_id].use_tty);
            fprintf(fpout, "usenic = \"%d\" ", task[task_id].use_nic);
            fprintf(fpout, "usecma = \"%d\" ", task[task_id].use_cma);
            fprintf(fpout, "usetim = \"%d\" ", task[task_id].use_tim);
            fprintf(fpout, "usehba = \"%d\" />\n", task[task_id].use_hba);
        }
        fprintf(fpout, "        </vspace>\n\n");
    }
    fprintf(fpout, "    </vspaceset>\n");
    fprintf(fpout, "</mapping_info>\n");
} // end buildXml()


/////////////////////////////////////
int main(int argc, char * argv[]) {
    if (argc < 2) {
        printf("Usage: bin2xml <input_file_path> <output_file_path>\n");
        return 1;
    }

    unsigned int bin[0x10000]; // 64 K int = 256 Kbytes 

    int fdin = open(argv[1], O_RDONLY);
    if (fdin < 0) {
        perror("open");
        exit(1);
    }

    FILE * fpout = fopen( argv[2], "w");
    if (fpout == NULL) {
        perror("open");
        exit(1);
    }

    unsigned int length = read(fdin, bin, 0x40000);

    if (length <= 0) {
        perror("read");
        exit(1);
    }

    if (bin[0] == IN_MAPPING_SIGNATURE) {
        buildXml((mapping_header_t *) bin, fpout);
    } 
    else {
        printf("[ERROR] Wrong file format\n");
        exit(1);
    }
    return 0;
} // end main()



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

