///////////////////////////////////////////////////////////////////////////////////
// File     : irq_handler.c
// Date     : 01/04/2012
// Author   : alain greiner 
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <irq_handler.h>
#include <sys_handler.h>
#include <ctx_handler.h>
#include <tim_driver.h>
#include <icu_driver.h>
#include <xcu_driver.h>
#include <tty_driver.h>
#include <nic_driver.h>
#include <cma_driver.h>
#include <bdv_driver.h>
#include <dma_driver.h>
#include <mapping_info.h>
#include <utils.h>

#if !defined( USE_XICU )
# error: You must define USE_XICU in the hard_config.h file
#endif

#if NB_TIM_CHANNELS
extern volatile unsigned char _user_timer_event[X_SIZE*Y_SIZE*NB_TIM_CHANNELS] ;
#endif

///////////////////////////////////////////////////////////////////////////////////
// This function uses the ICU or XICU component (Interrupt Controler Unit)
// to get the interrupt vector entry. There is one ICU or XICU component per
// cluster, and this component can support up to NB_PROCS_MAX output IRQs.
// It returns the highest priority active interrupt index (smaller
// indexes have the highest priority).
// Any value larger than 31 means "no active interrupt", and no ISR is executed.
//
// There is three interrupt vectors per processor (stored in the processor's
// scheduler) for the three HWI, PTI, and WTI interrupts types.
// Each interrupt vector entry contains three bits fields:
// - isr_id     bits[15:0]  : defines the type of ISR to be executed.
// - channel_id bits[30:16] : defines the channel for multi-channels peripherals.
// - valid      bit 31      : valid interrupt vector entry
// If the peripheral is replicated in clusters, the channel_id is 
// a global index : channel_id = cluster_id * NB_CHANNELS_MAX + loc_id   
///////////////////////////////////////////////////////////////////////////////////
void _irq_demux() 
{
    unsigned int gpid           = _get_procid();
    unsigned int cluster_xy     = gpid / NB_PROCS_MAX;
    unsigned int x              = cluster_xy >> Y_WIDTH;
    unsigned int y              = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int lpid           = gpid % NB_PROCS_MAX;
    unsigned int irq_id;
    unsigned int irq_type;
    char*        irq_type_str[] = { "HWI", "WTI", "PTI" }; 

    // get the highest priority active IRQ index 
    unsigned int icu_out_index = lpid * IRQ_PER_PROCESSOR;

#if USE_XICU
    _xcu_get_index( cluster_xy, icu_out_index, &irq_id, &irq_type );
#else
    irq_type = IRQ_TYPE_HWI;
    _icu_get_index( cluster_xy, icu_out_index, &irq_id );
#endif

    if (irq_id < 32) 
    {
        static_scheduler_t* psched = (static_scheduler_t*)_get_sched();
        unsigned int        entry;
        unsigned int        isr_type;
        unsigned int        channel;

        if      (irq_type == IRQ_TYPE_HWI) entry = psched->hwi_vector[irq_id];
        else if (irq_type == IRQ_TYPE_PTI) entry = psched->pti_vector[irq_id];
        else if (irq_type == IRQ_TYPE_WTI) entry = psched->wti_vector[irq_id];
        else
        {
            _printf("\n[GIET ERROR] illegal irq_type in irq_demux()\n");
            _exit();
        }

        isr_type   = (entry    ) & 0x0000FFFF;
        channel    = (entry>>16) & 0x00007FFF;

#if GIET_DEBUG_IRQS // we don't take the TTY lock to avoid deadlocks
_puts("\n[IRQS DEBUG] Processor[");
_putd(x);
_puts(",");
_putd(y);
_puts(",");
_putd(lpid);
_puts("] enters _irq_demux() at cycle ");
_putd(_get_proctime() );
_puts("\n  ");
_puts(irq_type_str[irq_type] );
_puts(" : irq_id = ");
_putd(irq_id);
_puts(" / isr_type = ");
_putd(isr_type);
_puts(" / channel = ");
_putd(channel);
_puts("\n");
#endif

        // ISR call
        if      ( isr_type == ISR_TICK   ) _isr_tick   ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_WAKUP  ) _isr_wakup  ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_BDV    ) _bdv_isr    ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_CMA    ) _cma_isr    ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_TTY_RX ) _tty_rx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_TTY_TX ) _tty_tx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_NIC_RX ) _nic_rx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_NIC_TX ) _nic_tx_isr ( irq_type, irq_id, channel );
        else if ( isr_type == ISR_TIMER  ) _timer_isr  ( irq_type, irq_id, channel );
        else
        {
            // we don't take the TTY lock to avoid deadlock
            _puts("\n[GIET ERROR] in _irq_demux() illegal ISR type on processor[");
            _putd(x);
            _puts(",");
            _putd(y);
            _puts(",");
            _putd(lpid);
            _puts("] at cycle ");
            _putd(_get_proctime() );
            _puts("\n  ");
            _puts(irq_type_str[irq_type] );
            _puts(" : irq_id = ");
            _putd(irq_id);
            _puts(" / isr_type = ");
            _putd(isr_type);
            _puts(" / channel = ");
            _putd(channel);
            _puts("\n");
            _exit();
        }
    }
    else   // no interrupt active
    {
        _isr_default();
    } 
}

///////////////////////////////////////////////////////////////////////////////////
// The default ISR is called when there is no active IRQ when the interrupt
// handler is called. It simply displays a warning message on TTY[0].
///////////////////////////////////////////////////////////////////////////////////
void _isr_default()
{
    unsigned int gpid       = _get_procid();
    unsigned int cluster_xy = gpid / NB_PROCS_MAX;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int lpid       = gpid % NB_PROCS_MAX;

    // We don't take TTY lock to avoid deadlock
    _puts("\n[GIET WARNING] IRQ handler called but no active IRQ on processor[");
    _putd( x );
    _puts(",");
    _putd( y );
    _puts(",");
    _putd( lpid );
    _puts("] at cycle ");
    _putd( _get_proctime() );
    _puts("\n  ");
}


///////////////////////////////////////////////////////////////////////////////////
// This ISR can only be executed after a WTI (IPI) to force a context switch
// on a remote processor. The context switch is only executed if the current task
// is the IDLE_TASK, or if the value written in the mailbox is non zero.
///////////////////////////////////////////////////////////////////////////////////
void _isr_wakup( unsigned int irq_type,   // HWI / WTI / PTI
                 unsigned int irq_id,     // index returned by ICU
                 unsigned int channel )   // unused
{
    unsigned int procid     = _get_procid();
    unsigned int cluster_xy = procid / NB_PROCS_MAX;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
    unsigned int lpid       = procid % NB_PROCS_MAX;
    unsigned int task       = _get_current_task_id();
    unsigned int value;

    if ( irq_type != IRQ_TYPE_WTI )
    {
        // we don't take the TTY lock to avoid deadlocks
        _puts("[GIET ERROR] _isr_wakup() not called by a WTI on processor[");
        _putd( x );
        _puts(",");
        _putd( y );
        _puts(",");
        _putd( lpid );
        _puts("] at cycle ");
        _putd( _get_proctime() );
        _puts("\n");
        _exit();
    }

    // get mailbox value and acknowledge WTI
    _xcu_get_wti_value( cluster_xy, irq_id, &value );

#if GIET_DEBUG_IRQS // we don't take the TTY lock to avoid deadlocks
_puts("\n[IRQS DEBUG] Processor[");
_putd( x );
_puts(",");
_putd( y );
_puts(",");
_putd( lpid );
_puts("] enters _isr_wakup() at cycle ");
_putd( _get_proctime() );
_puts("\n  WTI / mailbox data = ");
_putx( value );
_puts(" / current task index = ");
_putd( task );
_puts("\n  ");
#endif

    // context swich if required
    if ( (task == IDLE_TASK_INDEX) || (value != 0) ) _ctx_switch();
}

/////////////////////////////////////////////////////////////////////////////////////
// This ISR is in charge of context switch, and handles the IRQs generated by
// the "system" timers contained in the MULTI_TIMER or in the XICU component.
// The ISR acknowledges the IRQ, and calls the _ctx_switch() function.
/////////////////////////////////////////////////////////////////////////////////////
void _isr_tick( unsigned int irq_type,   // HWI / WTI / PTI
                unsigned int irq_id,     // index returned by ICU
                unsigned int channel )   // channel index if HWI
{
    unsigned int procid     = _get_procid();
    unsigned int cluster_xy = procid / NB_PROCS_MAX;

    // acknowledge HWI or PTI
    if ( irq_type == IRQ_TYPE_HWI ) _timer_reset_irq( cluster_xy, channel );
    else                            _xcu_timer_reset_irq( cluster_xy, irq_id );

#if GIET_DEBUG_IRQS  // we don't take the lock to avoid deadlock
unsigned int x          = cluster_xy >> Y_WIDTH;
unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);
unsigned int lpid       = procid % NB_PROCS_MAX;
_puts("\n[IRQS DEBUG] Processor[");
_putd( x );
_puts(",");
_putd( y );
_puts(",");
_putd( lpid );
_puts("] enters _isr_tick() at cycle ");
_putd( _get_proctime() );
_puts("\n  ");
#endif

    // context switch
    _ctx_switch();
}


// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

