///////////////////////////////////////////////////////////////////////////////////
// File     : mmc_driver.c
// Date     : 23/05/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The mmc_driver.c and mmc_driver.h files are part ot the GIET-VM nano-kernel.
// This driver supports the vci_mem_cache component used in the TSAR architecture.
//
// This component is replicated in all clusters, and can be accessed through 
// a configuration interface as a set of uncached, memory mapped registers.
///////////////////////////////////////////////////////////////////////////////////
// The (virtual) base address of the associated segment is:
//
//       seg_mmc_base + cluster_id * vseg_cluster_increment
//
// The seg_mmc_base and vseg_cluster_increment values must be defined 
// in the giet_vsegs.ld file.
////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <mmc_driver.h>
#include <tty_driver.h>
#include <utils.h>

#if !defined(X_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(Y_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(X_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(Y_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

///////////////////////////////////////////////////////////////////////////////////
// This function invalidates all cache lines covering a memory buffer defined
// by the physical base address, and the length.
// The buffer address MSB are used to compute the cluster index.
///////////////////////////////////////////////////////////////////////////////////
void _mmc_inval( paddr_t      buf_paddr,
                 unsigned int buf_length )
{
    // compute cluster coordinates
    unsigned int cluster_xy = (unsigned int)(buf_paddr>>(40-X_WIDTH-Y_WIDTH));
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    // parameters checking 
    if ( (x >= X_SIZE) || (y >= Y_SIZE) )
    {
        _printf("\n[GIET ERROR] in _memc_inval() : illegal cluster_xy for paddr %l\n",
                 buf_paddr );
        _exit();
    }

    unsigned int* mmc_address = (unsigned int*)((unsigned int)&seg_mmc_base + 
                                (cluster_xy * (unsigned int)&vseg_cluster_increment));

    // get the hard lock protecting exclusive access to MEMC
    while ( mmc_address[MEMC_LOCK] ) { asm volatile("nop"); }

    // write inval arguments
    mmc_address[MEMC_ADDR_LO]    = (unsigned int)buf_paddr;
    mmc_address[MEMC_ADDR_HI]    = (unsigned int)(buf_paddr>>32);
    mmc_address[MEMC_BUF_LENGTH] = buf_length;
    mmc_address[MEMC_CMD_TYPE]   = MEMC_CMD_INVAL;

    // release the lock 
    mmc_address[MEMC_LOCK] = 0;
}
///////////////////////////////////////////////////////////////////////////////////
// This function copies to external RAM all cache lines covering a memory buffer 
// defined by the physical base address, and the length, if they are dirty.
// The buffer address MSB are used to compute the cluster index.
///////////////////////////////////////////////////////////////////////////////////
void _mmc_sync( paddr_t      buf_paddr,
                unsigned int buf_length )
{
    // compute cluster coordinates
    unsigned int cluster_xy = (unsigned int)(buf_paddr>>(40-X_WIDTH-Y_WIDTH));
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    // parameters checking 
    if ( (x >= X_SIZE) || (y >= Y_SIZE) )
    {
        _printf( "\n[GIET ERROR] in _memc_sync() : illegal cluster_xy for paddr %l\n",
                 buf_paddr );
        _exit();
    }

    unsigned int * mmc_address = (unsigned int *) ((unsigned int)&seg_mmc_base + 
                                 (cluster_xy * (unsigned int)&vseg_cluster_increment));

    // get the hard lock protecting exclusive access to MEMC
    while ( mmc_address[MEMC_LOCK] ) { asm volatile("nop"); }

    // write inval arguments
    mmc_address[MEMC_ADDR_LO]    = (unsigned int)buf_paddr;
    mmc_address[MEMC_ADDR_HI]    = (unsigned int)(buf_paddr>>32);
    mmc_address[MEMC_BUF_LENGTH] = buf_length;
    mmc_address[MEMC_CMD_TYPE]   = MEMC_CMD_SYNC;

    // release the lock protecting MEMC
    mmc_address[MEMC_LOCK] = 0;
}

//////////////////////////////////////////////////////////////////////////////////
// This ISR access the vci_mem_cache component to get the faulty physical
// address and the associated SRCID. It must also acknowledge the IRQ.
//
// TODO implement...
//////////////////////////////////////////////////////////////////////////////////
void _mmc_isr( unsigned int irq_type,  // should be HWI 
               unsigned int irq_id,    // index returned by ICU
               unsigned int channel )  // unused
{
    unsigned int procid     = _get_procid();
    unsigned int cluster_xy = procid / NB_PROCS_MAX;
    unsigned int lpid       = procid % NB_PROCS_MAX;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    _printf("[GIET ERROR] MMC IRQ received by processor[%d,%d,%d]"
            " but _mmc_isr() not implemented...\n", x, y, lpid );
}



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

