#!/usr/bin/env python

from mapping import *

#################################################################################width #
#   file   : tsar_generic_iob.py
#   date   : april 2014
#   author : Alain Greiner
#######################################################################################
#  This file contains a mapping generator for the tsar_generic_iob platform. 
#  This includes both the hardware architecture (clusters, processors, peripherals,
#  physical space segmentation) and the mapping of all kernel objects (global vsegs).
#  The parameters are:
#  - x_size    : number of clusters in a row
#  - y_size    : number of clusters in a column
#  - nb_procs  : number of processors per cluster
#  - nb_ttys   : number of TTY channels
#  - nb_nics   : number of NIC channels
#  - fbf_size  : frame_buffer width = frame_buffer heigth
#  - x_io      : cluster_io x coordinate
#  - y_io      : cluster_io y coordinate
####################################################################################

#####################################
def tsar_generic_iob( x_size     = 2,
                      y_size     = 2,
                      nb_procs   = 1,
                      nb_ttys    = 1,
                      nb_nics    = 2, 
                      fbf_width  = 128,
                      x_io       = 0,
                      y_io       = 0 ):
                     
    ### parameters checking
    assert ( nb_procs <= 4 )
    assert ( (x_size == 1) or (x_size == 2) or (x_size == 4) 
             or (y_size == 8) or (x_size == 16) )
    assert ( (y_size == 1) or (y_size == 2) or (y_size == 4) 
             or (y_size == 8) or (y_size == 16) )
    assert ( nb_ttys <= 2 )
    
    ### define architecture constants

    platform_name = 'tsar_iob_%d_%d_%d' % (x_size, y_size, nb_procs)
    x_width       = 4
    y_width       = 4
    paddr_width   = 40
    irq_per_proc  = 1
    use_ramdisk   = False

    ### define physical segments

    ram_base = 0x0000000000
    ram_size = 0x4000000                         # 64 Mbytes

    xcu_base = 0x00B0000000 
    xcu_size = 0x1000                            # 4 Kbytes 

    dma_base = 0x00B1000000
    dma_size = 0x1000 * nb_procs                 # 4 Kbytes * nb_procs

    mmc_base = 0x00B2000000 
    mmc_size = 0x1000                            # 4 Kbytes

    offset_io = ((x_io << y_width) + y_io) << (paddr_width - x_width - y_width)

    bdv_base  = 0x00B3000000 + offset_io
    bdv_size  = 0x1000                           # 4kbytes

    tty_base  = 0x00B4000000 + offset_io
    tty_size  = 0x4000                           # 16 Kbytes

    nic_base  = 0x00B5000000 + offset_io
    nic_size  = 0x80000                          # 512 kbytes

    cma_base  = 0x00B6000000 + offset_io
    cma_size  = 0x1000 * 2 * nb_nics             # 4 kbytes * 2 * nb_nics

    fbf_base  = 0x00B7000000 + offset_io
    fbf_size  = fbf_width * fbf_width            # fbf_width * fbf_width bytes

    pic_base  = 0x00B8000000 + offset_io
    pic_size  = 0x1000                           # 4 Kbytes

    iob_base  = 0x00BE000000 + offset_io
    iob_size  = 0x1000                           # 4kbytes

    rom_base  = 0x00BFC00000 + offset_io
    rom_size  = 0x4000                           # 16 Kbytes

    ### define  bootloader vsegs base addresses

    boot_mapping_vbase   = 0x00000000            # ident
    boot_mapping_vsize   = 0x00010000            # 64 Kbytes

    boot_code_vbase      = 0x00010000            # ident
    boot_code_vsize      = 0x00020000            # 128 Kbytes 
  
    boot_data_vbase      = 0x00030000            # ident
    boot_data_vsize      = 0x00010000            # 64 Kbytes

    boot_buffer_vbase    = 0x00040000            # ident
    boot_buffer_vsize    = 0x00060000            # 384 Kbytes

    boot_stack_vbase     = 0x000A0000            # ident
    boot_stack_vsize     = 0x00050000            # 320 Kbytes

    ### define kernel vsegs base addresses

    kernel_code_vbase    = 0x80000000            
    kernel_code_vsize    = 0x00020000            # 128 Kbytes

    kernel_data_vbase    = 0x80020000
    kernel_data_vsize    = 0x00060000            # 384 Kbytes

    kernel_uncdata_vbase = 0x80080000
    kernel_uncdata_vsize = 0x00040000            # 256 Kbytes

    kernel_init_vbase    = 0x800C0000
    kernel_init_vsize    = 0x00010000            # 64 Kbytes

    kernel_sched_vbase   = 0xF0000000            # distributed in all clusters
    kernel_sched_vsize   = 0x1000 * nb_procs     # 4 kbytes per processor

    ### create mapping

    mapping = Mapping(  name         = platform_name,
                        x_size       = x_size,
                        y_size       = y_size,
                        nb_procs     = nb_procs,
                        x_width      = x_width,  
                        y_width      = y_width,
                        paddr_width  = paddr_width,
                        coherence    = True,
                        irq_per_proc = irq_per_proc,
                        use_ramdisk  = use_ramdisk,
                        x_io         = x_io,
                        y_io         = y_io )

    ###  add external peripherals

    iob = Iob( 'PSEG_IOB', base = iob_base, size = iob_size )
    mapping.addPeripheral( iob )

    bdv = Bdv( 'PSEG_BDV', base = bdv_base, size = bdv_size )
    mapping.addPeripheral( bdv )

    tty = Tty( 'PSEG_TTY', base = tty_base, size = tty_size, channels = nb_ttys )
    mapping.addPeripheral( tty )

    nic = Nic( 'PSEG_NIC', base = nic_base, size = nic_size, channels = nb_nics ) 
    mapping.addPeripheral( nic )

    cma = Cma( 'PSEG_CMA', base = cma_base, size = cma_size, channels = 2 * nb_nics )
    mapping.addPeripheral( cma )

    fbf = Fbf( 'PSEG_FBF', base = fbf_base, size = fbf_size )
    mapping.addPeripheral( fbf )

    rom = Rom( 'PSEG_ROM', base = rom_base, size = rom_size )
    mapping.addPeripheral( rom )

    pic = Pic( 'PSEG_PIC', base = pic_base, size = pic_size, channels = 32 )
    pic.add( PicIrq( srcid = 0  , dstx = 0, dsty = 0, dstid = 4  ) )
    pic.add( PicIrq( srcid = 1  , dstx = 0, dsty = 0, dstid = 5  ) )
    pic.add( PicIrq( srcid = 2  , dstx = 0, dsty = 0, dstid = 6  ) )
    pic.add( PicIrq( srcid = 3  , dstx = 0, dsty = 0, dstid = 7  ) )
    pic.add( PicIrq( srcid = 4  , dstx = 0, dsty = 0, dstid = 8  ) )
    pic.add( PicIrq( srcid = 5  , dstx = 0, dsty = 0, dstid = 9  ) )
    pic.add( PicIrq( srcid = 6  , dstx = 0, dsty = 0, dstid = 10 ) )
    pic.add( PicIrq( srcid = 7  , dstx = 0, dsty = 0, dstid = 11 ) )
    pic.add( PicIrq( srcid = 8  , dstx = 0, dsty = 0, dstid = 12 ) )
    pic.add( PicIrq( srcid = 16 , dstx = 0, dsty = 0, dstid = 13 ) )
    pic.add( PicIrq( srcid = 17 , dstx = 0, dsty = 0, dstid = 14 ) )
    pic.add( PicIrq( srcid = 18 , dstx = 0, dsty = 1, dstid = 13 ) )
    pic.add( PicIrq( srcid = 19 , dstx = 0, dsty = 1, dstid = 14 ) )
    pic.add( PicIrq( srcid = 20 , dstx = 1, dsty = 0, dstid = 13 ) )
    pic.add( PicIrq( srcid = 21 , dstx = 1, dsty = 0, dstid = 14 ) )
    pic.add( PicIrq( srcid = 22 , dstx = 1, dsty = 1, dstid = 13 ) )
    pic.add( PicIrq( srcid = 23 , dstx = 1, dsty = 1, dstid = 14 ) )
    mapping.addPeripheral( pic )

    ### add components replicated in all clusters   

    for x in xrange( x_size ):
        for y in xrange( y_size ):
            cluster_xy = (x << y_width) + y;
            offset     = cluster_xy << (paddr_width - x_width - y_width)

            ram = Pseg( 'PSEG_RAM', base = ram_base + offset, size = ram_size, segtype = 'RAM' )
            mapping.addPseg( ram )

            mmc = Mmc( 'PSEG_MMC', base  = mmc_base + offset, size = mmc_size )
            mapping.addPeripheral( mmc )

            dma = Dma( 'PSEG_DMA', base = dma_base + offset, size = dma_size, channels  = nb_procs ) 
            mapping.addPeripheral( dma )

            xcu = Xcu( 'PSEG_XCU', base = xcu_base + offset, size = xcu_size, channels = nb_procs )
            xcu.add ( XcuIrq ( srctype = 'HWI', srcid = 0, isrtype = 'ISR_MMC', dstid = 0 ) )
            for p in xrange( nb_procs ):
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = p, isrtype = 'ISR_WAKUP', dstid = p ) )
                xcu.add ( XcuIrq ( srctype = 'PTI', srcid = p, isrtype = 'ISR_TICK' , dstid = p ) )
            if (x == 0) and (y == 0):
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 4  , isrtype = 'ISR_NIC_RX', channel = 0) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 5  , isrtype = 'ISR_NIC_RX', channel = 1) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 6  , isrtype = 'ISR_NIC_TX', channel = 0) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 7  , isrtype = 'ISR_NIC_TX', channel = 1) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 8  , isrtype = 'ISR_CMA'   , channel = 0) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 9  , isrtype = 'ISR_CMA'   , channel = 1) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 10 , isrtype = 'ISR_CMA'   , channel = 2) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 11 , isrtype = 'ISR_CMA'   , channel = 3) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 12 , isrtype = 'ISR_BDV'   , channel = 0) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 13 , isrtype = 'ISR_TTY_RX', channel = 0) )
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = 14 , isrtype = 'ISR_TTY_RX', channel = 1) )
            mapping.addPeripheral( xcu )

    ### add processors

    for x in xrange (x_size):
        for y in xrange (y_size):
            for p in xrange (nb_procs):
                proc = Processor ( x, y, p )
                mapping.addProc ( proc )

    ### add global vsegs for boot_loader segments

    vseg = Vseg( 'seg_boot_mapping'   , boot_mapping_vbase   , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_mapping'   , boot_mapping_vsize   , 'BLOB' , binpath = 'map.bin' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_boot_code'      , boot_code_vbase      , 'CXW_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_code'      , boot_code_vsize      , 'BUFFER' ) )
    mapping.addGlobal( vseg )    
    
    vseg = Vseg( 'seg_boot_data'      , boot_data_vbase      , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_data'      , boot_data_vsize      , 'BUFFER' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_boot_buffer'    , boot_buffer_vbase    , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_buffer'    , boot_buffer_vsize    , 'BUFFER' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_boot_stack'     , boot_stack_vbase     , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_stack'     , boot_stack_vsize     , 'BUFFER' ) )
    mapping.addGlobal( vseg )    

    ### add global vsegs for kernel  segments 

    vseg = Vseg( 'seg_kernel_code'    , kernel_code_vbase    , 'CXW_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_code'    , kernel_code_vsize    , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_kernel_data'    , kernel_data_vbase    , 'C_W_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_data'    , kernel_data_vsize    , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_kernel_uncdata' , kernel_uncdata_vbase , '__W_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_uncdata' , kernel_uncdata_vsize , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_kernel_init'    , kernel_init_vbase    , 'CXW_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_init'    , kernel_init_vsize    , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    ### add global vsegs for external peripherals

    vseg = Vseg( 'seg_iob'            , iob_base , '__W_' , 0, 0, 'PSEG_IOB' )
    vseg.add ( Vobj( 'iob'            , iob_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_bdv'            , bdv_base , '__W_' , 0, 0, 'PSEG_BDV' )
    vseg.add ( Vobj( 'bdv'            , bdv_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_tty'            , tty_base , '__W_' , 0, 0, 'PSEG_TTY' )
    vseg.add ( Vobj( 'tty'            , tty_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_nic'            , nic_base , '__W_' , 0, 0, 'PSEG_NIC' )
    vseg.add ( Vobj( 'nic'            , nic_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_cma'            , cma_base , '__W_' , 0, 0, 'PSEG_CMA' )
    vseg.add ( Vobj( 'cma'            , cma_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_fbf'            , fbf_base , '__W_' , 0, 0, 'PSEG_FBF' )
    vseg.add ( Vobj( 'fbf'            , fbf_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_pic'            , pic_base , '__W_' , 0, 0, 'PSEG_PIC' )
    vseg.add ( Vobj( 'pic'            , pic_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_rom'            , rom_base , 'CXW_' , 0, 0, 'PSEG_ROM' )
    vseg.add ( Vobj( 'rom'            , rom_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    ### Global vsegs for replicated XCU, DMA, MMC and Scheduler ###

    for x in xrange (x_size):
        for y in xrange (y_size):
            offset = ((x << y_width) + y) << 16

            vseg = Vseg( 'seg_xcu_%d_%d' %(x,y) , xcu_base + offset, '__W_' , x, y, 'PSEG_XCU' )
            vseg.add ( Vobj( 'xcu_%d_%d' %(x,y) , xcu_size, 'PERI' ) )
            mapping.addGlobal( vseg )

            vseg = Vseg( 'seg_dma_%d_%d' %(x,y) , dma_base + offset, '__W_' , x, y, 'PSEG_DMA' )
            vseg.add ( Vobj( 'dma_%d_%d' %(x,y) , dma_size, 'PERI' ) )
            mapping.addGlobal( vseg )

            vseg = Vseg( 'seg_mmc_%d_%d' %(x,y) , mmc_base + offset, '__W_' , x, y, 'PSEG_MMC' )
            vseg.add ( Vobj( 'mmc_%d_%d' %(x,y) , mmc_size, 'PERI' ) )
            mapping.addGlobal( vseg )

            vseg = Vseg( 'seg_sch_%d_%d' %(x,y) , kernel_sched_vbase + offset, '__W_' , x, y, 'PSEG_RAM' )
            vseg.add ( Vobj( 'sch_%d_%d' %(x,y) , kernel_sched_vsize, 'SCHED' ) )
            mapping.addGlobal( vseg )

    ### return mapping ###

    return mapping

################################# transpose test #######################################################

if __name__ == '__main__':
    print tsar_generic_iob()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

