#!/usr/bin/env python

from mapping import *

#######################################################################################
#   file   : tsar_generic_leti.py
#   date   : april 2014
#   author : Alain Greiner
#######################################################################################
#  This file contains a mapping generator for the tsar_generic_leti platform. 
#  This includes both the hardware architecture (clusters, processors, peripherals,
#  physical space segmentation) and the mapping of all kernel objects (global vsegs).
#  This platform does not include any external peripherals, but only a block-device
#  controler and a single channel TTY in cluster[0,0].
#  The parameters are:
#  - x_size    : number of clusters in a row
#  - y_size    : number of clusters in a column 
#  - nb_procs  : number of processors per cluster
#######################################################################################

#####################################
def tsar_generic_leti( x_size     = 2,
                       y_size     = 2,
                       nb_procs   = 4 ):
                     
    ### parameters checking 

    assert ( nb_procs <= 4 ) 
    assert ( (x_size==1) or (x_size==2) or (x_size==4) or (x_size==8) or (x_size==16) )
    assert ( (y_size==1) or (y_size==2) or (y_size==4) or (y_size==8) )
    
    ### define architecture constants

    platform_name = 'tsar_leti_%d_%d_%d' % (x_size, y_size, nb_procs)
    x_width       = 4
    y_width       = 4
    paddr_width   = 40
    irq_per_proc  = 4
    use_ram_disk  = True
    x_io          = 0
    y_io          = 0

    ### physical segments

    ram_base = 0x0000000000
    ram_size = 0x4000000                         # 64 Mbytes

    xcu_base = 0x00F0000000 
    xcu_size = 0x1000                            # 4 Kbytes 

    mmc_base = 0x00E0000000
    mmc_size = 0x1000                            # 4 Kbytes

    bdv_base = 0x00F2000000 
    bdv_size = 0x1000                            # 4 Kbytes

    tty_base = 0x00F4000000 
    tty_size = 0x1000                            # 4 Kbytes

    ### Global vsegs / identity mapping

    reset_vbase          = 0x00000000
    reset_vsize          = 0x00010000            # 64 Kbytes

    boot_code_vbase      = 0x00010000
    boot_code_vsize      = 0x00020000            # 128 Kbytes 
  
    boot_data_vbase      = 0x00030000
    boot_data_vsize      = 0x00010000            # 64 Kbytes

    boot_buffer_vbase    = 0x00040000
    boot_buffer_vsize    = 0x00060000            # 384 Kbytes

    boot_stack_vbase     = 0x000A0000
    boot_stack_vsize     = 0x00050000            # 320 Kbytes

    boot_mapping_vbase   = 0x000F0000
    boot_mapping_vsize   = 0x00010000            # 64 Kbytes

    ram_disk_vbase       = 0x02000000         
    ram_disk_vsize       = 0x02000000            # 1 Mbytes

    ### Global vsegs / no identity mapping

    kernel_code_vbase    = 0x80000000            
    kernel_code_vsize    = 0x00020000            # 128 Kbytes

    kernel_data_vbase    = 0x80020000
    kernel_data_vsize    = 0x00060000            # 384 Kbytes

    kernel_uncdata_vbase = 0x80080000
    kernel_uncdata_vsize = 0x00040000            # 256 Kbytes

    kernel_init_vbase    = 0x800C0000
    kernel_init_vsize    = 0x00010000            # 64 Kbytes

    kernel_sched_vbase   = 0x800D0000            # distributed in all clusters
    kernel_sched_vsize   = 0x1000 * nb_procs     # 4 kbytes per processor

    ### create mapping

    mapping = Mapping(  name         = platform_name,
                        x_size       = x_size,
                        y_size       = y_size,
                        nb_procs     = nb_procs,
                        x_width      = x_width,  
                        y_width      = y_width,
                        paddr_width  = paddr_width,
                        coherence    = True,
                        irq_per_proc = irq_per_proc,
                        use_ram_disk = use_ram_disk,
                        x_io         = x_io,
                        y_io         = y_io )

    ### add components in clusters 

    for x in xrange( x_size ):
        for y in xrange( y_size ):
            cluster_xy = (x << y_width) + y;
            offset     = cluster_xy << (paddr_width - x_width - y_width)

            if( (x==0) and (y==0) ):
                bdv = Bdv( 'PSEG_BDV', base = bdv_base , size = bdv_size )
                mapping.addPeripheral( bdv )

                tty = Tty( 'PSEG_TTY', base = tty_base , size = tty_size , channels = 1 )
                mapping.addPeripheral( tty )

            ram = Pseg( 'PSEG_RAM', base = ram_base + offset, size = ram_size, segtype = 'RAM' )
            mapping.addPseg( ram )

            mmc = Mmc( 'PSEG_MMC', base = mmc_base + offset, size = mmc_size )
            mapping.addPeripheral( mmc )

            xcu = Xcu( 'PSEG_XCU', base = xcu_base + offset, size = xcu_size, channels = nb_procs * irq_per_proc )
            xcu.add ( XcuIrq     ( srctype = 'HWI', srcid = 8  , isrtype = 'ISR_MMC'   , dstid = 0 ) )
            for p in xrange( nb_procs ):
                xcu.add ( XcuIrq ( srctype = 'WTI', srcid = p  , isrtype = 'ISR_WAKUP' , dstid = p ) )
                xcu.add ( XcuIrq ( srctype = 'PTI', srcid = p  , isrtype = 'ISR_TICK'  , dstid = p ) )
            if ( (x==0) and (y==0) ):
                xcu.add ( XcuIrq ( srctype = 'HWI', srcid = 9  , isrtype = 'ISR_BDV'   , dstid = 0 ) )
                xcu.add ( XcuIrq ( srctype = 'HWI', srcid = 10 , isrtype = 'ISR_TTY_RX', dstid = 0 ) )
            mapping.addPeripheral( xcu )

    ### processors

    for x in xrange ( x_size ):
        for y in xrange ( y_size ):
            for p in xrange (nb_procs):
                proc = Processor ( x, y, p )
                mapping.addProc ( proc )

    ### glogal vsegs for Preloader and RamDisk / identity mapping

    vseg = Vseg( 'seg_reset_code'     , reset_vbase          , 'CX__' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'reset_code'     , reset_vsize          , 'BUFFER' ) )
    mapping.addGlobal( vseg )

    vseg = Vseg( 'seg_ram_disk'       , ram_disk_vbase       , 'CX__' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'reset_code'     , ram_disk_vsize       , 'BUFFER' ) )
    mapping.addGlobal( vseg )

    ### global vsegs for boot_loader segments / identity mapping

    vseg = Vseg( 'seg_boot_mapping'   , boot_mapping_vbase   , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_mapping'   , boot_mapping_vsize   , 'BLOB' , binpath = 'map.bin' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_boot_code'      , boot_code_vbase      , 'CXW_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_code'      , boot_code_vsize      , 'BUFFER' ) )
    mapping.addGlobal( vseg )    
    
    vseg = Vseg( 'seg_boot_data'      , boot_data_vbase      , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_data'      , boot_data_vsize      , 'BUFFER' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_boot_buffer'    , boot_buffer_vbase    , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_buffer'    , boot_buffer_vsize    , 'BUFFER' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_boot_stack'     , boot_stack_vbase     , 'C_W_' , 0, 0, 'PSEG_RAM', ident = True )
    vseg.add ( Vobj( 'boot_stack'     , boot_stack_vsize     , 'BUFFER' ) )
    mapping.addGlobal( vseg )    

    ### add global vsegs for kernel  segments 

    vseg = Vseg( 'seg_kernel_code'    , kernel_code_vbase    , 'CXW_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_code'    , kernel_code_vsize    , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_kernel_data'    , kernel_data_vbase    , 'C_W_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_data'    , kernel_data_vsize    , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_kernel_uncdata' , kernel_uncdata_vbase , '__W_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_uncdata' , kernel_uncdata_vsize , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_kernel_init'    , kernel_init_vbase    , 'CXW_' , 0, 0, 'PSEG_RAM' )
    vseg.add ( Vobj( 'kernel_init'    , kernel_init_vsize    , 'ELF'  , binpath = 'build/kernel/kernel.elf' ) )
    mapping.addGlobal( vseg )    

    ###  global vsegs for for bdv and tty peripherals

    vseg = Vseg( 'seg_bdv'            , bdv_base , '__W_' , 0, 0, 'PSEG_BDV' )
    vseg.add ( Vobj( 'bdv'            , bdv_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    vseg = Vseg( 'seg_tty'            , tty_base , '__W_' , 0, 0, 'PSEG_TTY' )
    vseg.add ( Vobj( 'tty'            , tty_size , 'PERI' ) )
    mapping.addGlobal( vseg )    

    ### Global vsegs for replicated XCU, MMC and Scheduler

    for x in xrange ( x_size ):
        for y in xrange ( y_size ):
            offset = ((x << y_width) + y) << 16

            vseg = Vseg( 'seg_xcu_%d_%d' %(x,y) , xcu_base + offset, '__W_' , x, y, 'PSEG_XCU' )
            vseg.add ( Vobj( 'xcu_%d_%d' %(x,y) , xcu_size, 'PERI' ) )
            mapping.addGlobal( vseg )

            vseg = Vseg( 'seg_mmc_%d_%d' %(x,y) , mmc_base + offset, '__W_' , x, y, 'PSEG_MMC' )
            vseg.add ( Vobj( 'mmc_%d_%d' %(x,y) , mmc_size, 'PERI' ) )
            mapping.addGlobal( vseg )

            vseg = Vseg( 'seg_sch_%d_%d' %(x,y) , kernel_sched_vbase + offset, '__W_' , x, y, 'PSEG_RAM' )
            vseg.add ( Vobj( 'sch_%d_%d' %(x,y) , kernel_sched_vsize, 'SCHED' ) )
            mapping.addGlobal( vseg )

    ### return mapping 

    return mapping

################################# transpose test #######################################################

if __name__ == '__main__':
    print tsar_generic_leti()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

