#!/usr/bin/env python

#######################################################################################
#   file   : giet_mapping.py
#   date   : april 2014
#   author : Alain Greiner
#######################################################################################
#  This file contains the classes required to define a generic mapping in python
#  for the GIET_VM.
# - A 'Mapping' contains + various structural constants,
#                        + a 'Cluster' set,     (hardware architecture)
#                        + a 'Vseg' set,        (kernel virtual segments mapping)
#                        + a 'Vspace' set       (several user applications).
# - A 'Cluster' contains + coordinates (x,y), 
#                        + a 'Pseg' set,        (all physical segments in cluster)
#                        + a 'Proc' set,        (processors in cluster)
#                        + a 'Peripheral' set   (peripherals in cluster)
# - A 'Vspace' contains  + a 'Vseg' set,        (user virtual segments mapping)
#                        + a 'Task' set         (user tasks mapping)
####################################################################################

#########################
class Mapping( object ):
#########################
    def __init__( self,
                  name,                   # mapping name 
                  x_size,                 # number of clusters in a row
                  y_size,                 # number of clusters in a column
                  nb_procs,               # max number of processors per cluster
                  x_width      = 4,       # number of bits encoding x coordinate
                  y_width      = 4,       # number of bits encoding y coordinate
                  paddr_width  = 32,      # number of bits for physical address
                  coherence    = False,   # hardware cache coherence 
                  irq_per_proc = 1,       # number or IRQ lines from XCU to processor 
                  use_ram_disk = False,   # architecture contains ram_disk when true
                  x_io         = 0,       # cluster_io x coordinate
                  y_io         = 0 ):     # cluster_io y coordinate

        self.name         = name
        self.paddr_width  = paddr_width           
        self.coherence    = coherence
        self.x_size       = x_size
        self.y_size       = y_size
        self.x_width      = x_width
        self.y_width      = y_width
        self.irq_per_proc = irq_per_proc
        self.nb_procs     = nb_procs      
        self.use_ram_disk = use_ram_disk
        self.x_io         = x_io
        self.y_io         = y_io

        self.total_procs  = 0
        self.total_globs  = 0

        # clusters array
        self.clusters     = []
        for x in xrange (1<<x_width):
            for y in xrange (1<<y_width):
                self.clusters.append( Cluster(x,y) )

        # globals array
        self.globs        = []

        # vspace array
        self.vspaces      = []

        return

    ##########################
    def addPseg( self, pseg ):
        cluster_xy = pseg.base >> (self.paddr_width - self.x_width - self.y_width)
        x = cluster_xy >> (self.y_width);
        y = cluster_xy & ((1 << self.y_width) - 1) 
        if (x >= self.x_size) or (y >= self.y_size):
            print '(x,y) coordinates too large for pdeg %s' % str(pseg)
        self.clusters[cluster_xy].psegs.append( pseg )
        return

    ######################################
    def addPeripheral( self, peripheral ):
        cluster_xy = peripheral.pseg.base >> (self.paddr_width - self.x_width - self.y_width)
        x = cluster_xy >> (self.y_width);
        y = cluster_xy & ((1 << self.y_width) - 1) 
        if (x >= self.x_size) or (y >= self.y_size):
            print '(x,y) coordinates too large for pdeg %s' % str(pseg)
        self.clusters[cluster_xy].peripherals.append( peripheral )
        return

    ###########################
    def addProc ( self, proc ):
        cluster_xy = (proc.x << self.y_width) + proc.y
        self.clusters[cluster_xy].procs.append( proc )
        self.total_procs += 1
        if (proc.x >= self.x_size) or (proc.y >= self.y_size):
            print '(x,y) coordinates too large for node %s' % str(proc)
        return

    ############################
    def addGlobal( self, vseg ):
        self.globs.append( vseg ) 
        return

    ##############################
    def addVspace( self, vspace ):
        self.vspaces.append( vspace )
        return
    
    ####################
    def __str__( self ):    # xml generation for mapping
        s = '<?xml version="1.0"?>\n\n'
        s += '<mapping_info signature    = "0xDACE2014"\n'
        s += '              name         = "%s"\n'   % (self.name)
        s += '              x_size       = "%d"\n'   % (self.x_size)
        s += '              y_size       = "%d"\n'   % (self.y_size)
        s += '              x_width      = "%d"\n'   % (self.x_width)
        s += '              y_width      = "%d"\n'   % (self.y_width)
        s += '              irq_per_proc = "%d"\n'   % (self.irq_per_proc)
        s += '              use_ram_disk = "%d"\n'   % (self.use_ram_disk)
        s += '              x_io         = "%d"\n'   % (self.x_io)
        s += '              y_io         = "%d" >\n' % (self.y_io)
        s += '    <clusterset>\n'
        for x in xrange ( self.x_size ):
            for y in xrange ( self.y_size ):
                cluster_xy = (x << self.y_width) + y 
                s += str( self.clusters[cluster_xy] )
        s += '    </clusterset>\n'
        s += '    <globalset>\n'
        for vseg in self.globs:
            s += str ( vseg )
        s += '    </globalset>\n'
        s += '    <vspaceset>\n'
        for vspace in self.vspaces:
            s += str( vspace )
        s += '    </vspaceset>\n'
        s += '</mapping_info>\n'
        return s
       
#######################
class Vspace( object ):
#######################
    def __init__( self, name, startname ):
        self.name      = name           # application name
        self.startname = startname      # name of vobj containing the start_vector
        self.vsegs     = []
        self.tasks     = []
        return
    ##########################
    def addTask( self, task ):
        self.tasks.append( task )
        return

    ##########################
    def addVseg( self, vseg ):
        self.vsegs.append( vseg )
        return

    ####################
    def __str__( self ):   # xml for one vspace
        s =  '        <vspace name="%s" startname="%s" >\n' % ( self.name, self.startname )
        for vseg in self.vsegs:
            s += str( vseg )
        for task in self.tasks:
            s += str( task )
        s += '        </vspace>\n'
        return s

#####################
class Task( object ):
#####################
    def __init__( self, name, trdid, x, y, p, stackname, heapname, startid, \
                  usetty = False, usenic = False, usehba = False ):
        self.name      = name       # thread name
        self.trdid     = trdid      # thread index (unique in vspace)
        self.x         = x          # processor x coordinate 
        self.y         = y          # processor y coordinate 
        self.p         = p          # processor local index  
        self.stackname = stackname  # name of vobj containing the stack
        self.heapname  = heapname   # name of vobj containing the heap
        self.startid   = startid    # index in start_vector
        self.usetty    = usetty     # request a private TTY channel
        self.usenic    = usenic     # request a private NIC channel
        self.usehba    = usehba     # request a private HBA channel
        return

    ####################
    def __str__( self ):    # xml for one task
        s = '        <task name="%s" trdid="%d" x="%d" y="%d" p="%d" stackname="%s" heapname="%s" startid="%d"' \
            % (self.name, self.trdid, self.x, self.y, self.p, self.stackname, self.heapname, self.startid)
        if self.usetty != 0:
            s += ' usetty="1"'  
        if self.usenic != 0:
            s += ' usenic="1"'  
        if self.usehba != 0:
            s += ' usehba="1"'  
        s += ' />\n'           
        return s

######################
class Vseg( object ):
######################
    def __init__( self, name, vbase, mode, x, y, psegname, ident = False ):
        assert mode in ['CXWU','CXW_','CX_U','CX__',
                        'C_WU','C_W_','C__U','C___',
                        '_XWU','_XW_','_X_U','_X__',
                        '__WU','__W_','___U','____']
        self.name     = name
        self.vbase    = vbase
        self.mode     = mode
        self.x        = x
        self.y        = y
        self.psegname = psegname
        self. ident   = ident
        self.vobjs   = []
        return

    #####################
    def add ( self, vo ):
        self.vobjs.append( vo )
        return

    ####################
    def __str__( self ):  # xml for one vseg
        s = '        <vseg name="%s" vbase="0x%x" mode="%s" x="%d" y="%d" psegname="%s"' \
            % ( self.name, self.vbase, self.mode, self.x, self.y, self.psegname )
        if self.ident != 0:
            s += ' ident="1"'
        s += ' >\n'
        for vobj in self.vobjs:
            s += str( vobj )
        s += '        </vseg>\n'
        return s

######################
class Vobj( object ):
######################
    def __init__( self, name, length, vtype, binpath = None, align = 0, init = 0 ):
        assert vtype in ['ELF','BLOB','PTAB','PERI','MWMR','LOCK', \
                         'BUFFER','BARRIER','CONST','MEMSPACE','SCHED']
        self.name     = name
        self.vtype    = vtype
        self.length   = length
        self.binpath  = binpath
        self.align    = align
        self.init     = init  
        return

    ####################
    def __str__( self ):  # xml for a vobj
        s = '            <vobj name="%s" type="%s" length="0x%x"' \
                           % ( self.name, self.vtype, self.length )
        if (self.binpath != None):
            s += ' binpath="%s"' % (self.binpath)
        if (self.align != 0):
            s += ' align="%d"' % (self.align)
        if (self.init != 0):
            s += ' init="%d"' % (self.init)
        s += ' />\n'
        return s

#########################
class Cluster ( object ):
#########################
    def __init__( self, x, y ):
        self.x           = x
        self.y           = y
        self.psegs       = []
        self.peripherals = []
        self.procs       = []
        return

    ####################
    def __str__( self ):  # xml for a cluster
        s = '        <cluster x="%d" y="%d" >\n' % (self.x, self.y)
        for pseg in self.psegs:
            s += str( pseg )             # psegs type RAM
        for peri in self.peripherals:
            s += str(peri.pseg)          # psegs type PERI
        for proc in self.procs:
            s += str( proc )             # processors
        for peri in self.peripherals:
            s += str( peri )             # peripherals
        s += '        </cluster>\n'
        return s

###########################
class Processor ( object ):
###########################
    def __init__( self, x, y, lpid ):
        self.x    = x
        self.y    = y
        self.lpid = lpid
        return

    ####################
    def __str__( self ):   # xml for a processor      
        return '            <proc index="%d" />\n' % (self.lpid)

######################
class Pseg ( object ):
######################
    def __init__( self, name, base, size, segtype ):
        self.name     = name
        self.base     = base
        self.size     = size
        self.segtype  = segtype
        assert segtype in ['RAM','PERI']
        return
   
    ####################
    def __str__( self ):   # xml for a pseg
        return '            <pseg name="%s" type="%s" base="0x%x" length="0x%x" />\n' \
                % (self.name, self.segtype, self.base, self.size)

############################
class Peripheral ( object ):
############################
    def __init__( self, name, base, size, ptype, subtype, channels = 1):
        self.channels   = channels
        self.ptype      = ptype
        self.subtype    = subtype
        self.pseg       = Pseg( name, base, size, 'PERI' )
        return

    ####################
    def __str__( self ):    # xml for a peripheral
        s = '            <periph type="%s" subtype="%s" psegname="%s" channels="%d" >\n' \
               % ( self.ptype, self.subtype, self.pseg.name, self.channels )
        if ( self.ptype == 'PIC' ):
            for i in self.irqs:
                s += str( i )         #  picIrq
        if ( (self.ptype == 'XCU') or (self.ptype == 'ICU') ):
            for i in self.irqs:
                s += str( i )         #  xcuIrq
        s += '            </periph>\n'
        return s

########################
class Tty( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'TTY', 'NONE', channels )
        return

########################
class Dma( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'DMA', 'NONE', channels )
        return

########################
class Bdv( Peripheral ):
########################
    def __init__( self, name, base, size ):
        Peripheral.__init__( self, name, base, size, 'IOC', 'BDV' )
        return

########################
class Spi( Peripheral ):
########################
    def __init__( self, name, base, size ):
        Peripheral.__init__( self, name, base, size, 'IOC', 'SPI' )
        return

########################
class Hba( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'IOC', 'HBA', channels )
        return

########################
class Nic( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'NIC', 'NONE', channels )
        return

########################
class Cma( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'CMA', 'NONE', channels )
        return

########################
class Fbf( Peripheral ):
########################
    def __init__( self, name, base, size ):
        Peripheral.__init__( self, name, base, size, 'FBF', 'NONE' )
        return

########################
class Tim( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'TIM', 'NONE', channels )
        return

########################
class Iob( Peripheral ):
########################
    def __init__( self, name, base, size ):
        Peripheral.__init__( self, name, base, size, 'IOB', 'NONE' )
        return

########################
class Rom( Peripheral ):
########################
    def __init__( self, name, base, size ):
        Peripheral.__init__( self, name, base, size, 'ROM', 'NONE' )
        return

########################
class Mmc( Peripheral ):
########################
    def __init__( self, name, base, size ):
        Peripheral.__init__( self, name, base, size, 'MMC', 'NONE' )
        return

########################
class Xcu( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'XCU', 'NONE', channels )
        self.irqs  = []
        return

    def add ( self, irq ):
        self.irqs.append( irq )
        return

########################
class Icu( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'ICU', 'NONE', channels )
        self.irqs  = []
        return

    ######################
    def add ( self, irq ):
        self.irqs.append( irq )
        return

########################
class Pic( Peripheral ):
########################
    def __init__( self, name, base, size, channels ):
        Peripheral.__init__( self, name, base, size, 'PIC', 'NONE', channels )
        self.irqs  = []
        return

    ######################
    def add ( self, irq ):
        self.irqs.append( irq )
        return

########################
class XcuIrq ( object ):
########################
    def __init__( self, srctype, srcid, isrtype, channel = 0, dstid = 0 ):
        assert srctype in ['HWI','WTI','PTI']
        assert isrtype in ['ISR_DEFAULT','ISR_TICK','ISR_TTY_RX','ISR_TTY_TX',
                           'ISR_BDV', 'ISR_TIMER', 'ISR_WAKUP', 'ISR_NIC_RX',
                           'ISR_NIC_TX','ISR_CMA','ISR_MMC']
        assert srcid < 32
        self.srctype = srctype
        self.srcid   = srcid
        self.isrtype = isrtype
        self.channel = channel
        self.dstid   = dstid
        return

    ####################
    def __str__( self ):   # xml for an xcuIrq
        s = '                <irq srctype="%s" srcid="%d" isr="%s" channel="%d" dstid="%d" />\n' \
                % (self.srctype, self.srcid, self.isrtype, self.channel, self.dstid)
        return s

########################
class PicIrq ( object ):
########################
    def __init__( self, srcid, dstx, dsty, dstid ):
        assert srcid < 32
        self.srcid   = srcid
        self.dstx    = dstx
        self.dsty    = dsty
        self.dstid   = dstid
        return

    ####################
    def __str__( self ):   # xml for a picIrq
        s = '                <irq srcid="%d" dstx="%d" dsty="%d" dstid="%d" />\n' \
                % (self.srcid, self.dstx, self.dsty, self.dstid)
        return s


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

