/////////////////////////////////////////////////////////////////////////////////////////
// File     : ctx_handler.h
// Date     : 01/04/2012
// Authors  : alain greiner & joel porquet
// Copyright (c) UPMC-LIP6
/////////////////////////////////////////////////////////////////////////////////////////
// The ctx_handler.h and ctx_handler.c files are part of the GIET-VM nano-kernel.
// This code is used to support context switch when several tasks are executing
// in time multiplexing on a single processor.
// The tasks are statically allocated to a processor in the boot phase, and
// there is one private scheduler per processor. Each sheduler occupies 4K bytes, 
// and contains up to 14 task contexts (task_id is from 0 to 13).
// The task context [13] is reserved for the "idle" task that does nothing, and
// is launched by the scheduler when there is no other runable task.
/////////////////////////////////////////////////////////////////////////////////////////
// A task context is an array of 64 words = 256 bytes. 
// It contains copies of processor registers (when the task is preempted):
// - GPR[i], generally stored in slot (i). $0, $26 & $27 are not saved.
// - HI & LO registers
// - CP0 registers: EPC, SR, CR, BVAR 
// - CP2 registers : PTPR 
// It contains some general informations associated to the task:
// - TTY    : TTY channel global index
// - NIC    : NIC channel global index
// - CMA    : CMA channel global index
// - HBA    : HBA channel global index
// - DMA    : DMA channel local index
// - TIM    : TIM channel local index
// - PTAB   : page table virtual base address 
// - LTID   : Task local index (in scheduler)
// - VSID   : Virtual space index
// - RUN    : Task state (0 => sleeping / 1 => runnable ) 
// - TRDID  : Thread ID index (in vspace)
//
// ctx[0] <- ***   |ctx[8] <- $8    |ctx[16]<- $16   |ctx[24]<- $24
// ctx[1] <- $1    |ctx[9] <- $9    |ctx[17]<- $17   |ctx[25]<- $25
// ctx[2] <- $2    |ctx[10]<- $10   |ctx[18]<- $18   |ctx[26]<- LO
// ctx[3] <- $3    |ctx[11]<- $11   |ctx[19]<- $19   |ctx[27]<- HI
// ctx[4] <- $4    |ctx[12]<- $12   |ctx[20]<- $20   |ctx[28]<- $28
// ctx[5] <- $5    |ctx[13]<- $13   |ctx[21]<- $21   |ctx[29]<- SP
// ctx[6] <- $6    |ctx[14]<- $14   |ctx[22]<- $22   |ctx[30]<- $30
// ctx[7] <- $7    |ctx[15]<- $15   |ctx[23]<- $23   |ctx[31]<- RA
//
// ctx[32]<- EPC   |ctx[40]<- TTY   |ctx[48]<- TRDID |
// ctx[33]<- CR    |ctx[41]<- DMA
// ctx[34]<- SR    |ctx[42]<- NIC
// ctx[35]<- BVAR  |ctx[43]<- TIM
// ctx[36]<- PTAB  |ctx[44]<- HBA
// ctx[37]<- LTID  |ctx[45]<- CMA 
// ctx[38]<- VSID  |ctx[46]<- GTID
// ctx[39]<- PTPR  |ctx[47]<- RUN
/////////////////////////////////////////////////////////////////////////////////////////

#ifndef _CTX_HANDLER_H
#define _CTX_HANDLER_H

#include <giet_config.h>

/////////////////////////////////////////////////////////////////////////////////
//    Definition of the scheduler structure
/////////////////////////////////////////////////////////////////////////////////

typedef struct static_scheduler_s 
{
    unsigned int context[14][64];      // at most 14 task (including idle_task)
    unsigned int tasks;                // actual number of tasks
    unsigned int current;              // current task index
    unsigned int hwi_vector[32];       // hardware interrupt vector
    unsigned int pti_vector[32];       // timer    interrupt vector
    unsigned int wti_vector[32];       // software interrupt vector
    unsigned int reserved[30];         // padding to 4 Kbytes
} static_scheduler_t;


/////////////////////////////////////////////////////////////////////////////////
//  "idle" task index definition
/////////////////////////////////////////////////////////////////////////////////

#define IDLE_TASK_INDEX        13

/////////////////////////////////////////////////////////////////////////////////
//    Definition of the task context slots indexes
/////////////////////////////////////////////////////////////////////////////////

#define CTX_SP_ID        29  // Stack Pointer
#define CTX_RA_ID        31  // Return Address

#define CTX_EPC_ID       32  // Exception Program Counter (CP0)
#define CTX_CR_ID        33  // Cause Register (CP0)
#define CTX_SR_ID        34  // Status Register (CP0)
#define CTX_BVAR_ID      35	 // Bad Virtual Address Register (CP0)
#define CTX_PTAB_ID      36  // Page Table Virtual address
#define CTX_LTID_ID      37  // Local  Task Index (in scheduler)
#define CTX_VSID_ID      38  // Vspace Index     
#define CTX_PTPR_ID      39  // Page Table Pointer Register (PADDR>>13)

#define CTX_TTY_ID       40  // global TTY channel   
#define CTX_DMA_ID       41  // local DMA channel 
#define CTX_NIC_ID       42  // global NIC channel 
#define CTX_TIM_ID       43  // local TIMER channel
#define CTX_HBA_ID       44  // global HBA channel
#define CTX_CMA_ID       45  // global CMA channel 
#define CTX_GTID_ID      46  // Global Task Index
#define CTX_RUN_ID       47  // Boolean: task runable

#define CTX_TRDID_ID     48  // Thread Index in vspace

//////////////////////////////////////////////////////////////////////////////////
//     Prototype of the context switch function
//////////////////////////////////////////////////////////////////////////////////

extern void _ctx_switch();
extern void _ctx_eret();
extern void _idle_task();

extern static_scheduler_t _scheduler[];

#endif
