///////////////////////////////////////////////////////////////////////////////////
// File     : xcu_driver.h
// Date     : 01/11/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The xcu_driver.c and xcu_driver.h files are part ot the GIET-VM nano-kernel.
// This driver supports the SoCLib vci_xicu, that is a vectorised interrupt
// controler supporting IPI (Inter Processor Interrupts) and integrated timers.
//
// It can exist several interrupt controller unit in the architecture 
// (one per cluster), and each one can contain several channels.
// The number of XICU channels is equal to NB_PROCS_MAX, because there is 
// one private XICU channel per processor in a cluster.
////////////////////////////////////////////////////////////////////////////////
// The virtual base address of the segment associated to the component is:
//
//      seg_xcu_base + cluster_xy * vseg_cluster_increment
//
// The seg_xcu_base and vseg_cluster_increment values must be defined 
// in giet_vsegs.ld file.
////////////////////////////////////////////////////////////////////////////////

#ifndef _GIET_XCU_DRIVER_H_
#define _GIET_XCU_DRIVER_H_

///////////////////////////////////////////////////////////////////////////////////
// XICU registers offsets
///////////////////////////////////////////////////////////////////////////////////

enum Xcu_registers 
{
    XCU_WTI_REG = 0,
    XCU_PTI_PER = 1,
    XCU_PTI_VAL = 2,
    XCU_PTI_ACK = 3,

    XCU_MSK_PTI = 4,
    XCU_MSK_PTI_ENABLE = 5,
    XCU_MSK_PTI_DISABLE = 6,
    XCU_PTI_ACTIVE = 6,

    XCU_MSK_HWI = 8,
    XCU_MSK_HWI_ENABLE = 9,
    XCU_MSK_HWI_DISABLE = 10,
    XCU_HWI_ACTIVE = 10,

    XCU_MSK_WTI = 12,
    XCU_MSK_WTI_ENABLE = 13,
    XCU_MSK_WTI_DISABLE = 14,
    XCU_WTI_ACTIVE = 14,

    XCU_PRIO = 15,
};

#define XCU_REG(func, index) (((func)<<5)|(index))
 
///////////////////////////////////////////////////////////////////////////////////
// XICU access functions 
///////////////////////////////////////////////////////////////////////////////////

extern void _xcu_set_mask( unsigned int cluster_xy,
                           unsigned int channel,  
                           unsigned int mask, 
                           unsigned int irq_type );

extern void _xcu_get_index( unsigned int   cluster_xy, 
                            unsigned int   channel,   
                            unsigned int * index,
                            unsigned int * irq_type );

extern void _xcu_send_wti( unsigned int cluster_xy,
                           unsigned int wti_index,
                           unsigned int wdata );

extern void _xcu_get_wti_value( unsigned int   cluster_xy,
                                unsigned int   wti_index,
                                unsigned int * value );

extern void _xcu_get_wti_address( unsigned int   wti_index,
                                  unsigned int * address );

extern void _xcu_timer_start( unsigned int cluster_xy, 
                              unsigned int pti_index,
                              unsigned int period ); 

extern void _xcu_timer_stop( unsigned int cluster_xy, 
                             unsigned int pti_index ); 

extern unsigned int _xcu_timer_reset_irq( unsigned int cluster_xy, 
                                          unsigned int pti_index );

extern void _xcu_timer_reset_cpt( unsigned int cluster_xy, 
                                  unsigned int pti_index ); 

///////////////////////////////////////////////////////////////////////////////////

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

