///////////////////////////////////////////////////////////////////////////////////
// File     : utils.h
// Date     : 18/10/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The utils.c and utils.h files are part of the GIET-VM nano-kernel.
// They define more or less the GIET-VM HAL (Hardware Abstraction Layer),
// and contains various utility functions, that can be used by both the 
// boot code and the kernel code.
///////////////////////////////////////////////////////////////////////////////////

#ifndef GIET_UTILS_H
#define GIET_UTILS_H

#include <mapping_info.h>
#include <giet_config.h>

//////////////////////////////////////////////////////////////////////////////////
// NULL pointer definition
//////////////////////////////////////////////////////////////////////////////////

#define NULL (void *)0

///////////////////////////////////////////////////////////////////////////////////
// For retrieving base addresses defined in seg.ld file.
///////////////////////////////////////////////////////////////////////////////////

typedef struct _ld_symbol_s _ld_symbol_t;

// boot segments vbase addresses & sizes
extern _ld_symbol_t boot_code_vbase;
extern _ld_symbol_t boot_code_size;

extern _ld_symbol_t boot_data_vbase;
extern _ld_symbol_t boot_data_size;

extern _ld_symbol_t boot_stack_vbase;
extern _ld_symbol_t boot_stack_size;

extern _ld_symbol_t boot_mapping_vbase;
extern _ld_symbol_t boot_mapping_size;

extern _ld_symbol_t boot_buffer_vbase;
extern _ld_symbol_t boot_buffer_size;

// kernel segments vbase addresses & sizes
extern _ld_symbol_t kernel_code_vbase;
extern _ld_symbol_t kernel_code_size;

extern _ld_symbol_t kernel_data_vbase;
extern _ld_symbol_t kernel_data_size;

extern _ld_symbol_t kernel_uncdata_vbase;
extern _ld_symbol_t kernel_uncdata_size;

extern _ld_symbol_t kernel_init_vbase;
extern _ld_symbol_t kernel_init_size;

///////////////////////////////////////////////////////////////////////////////////
//     Memory handling functions
///////////////////////////////////////////////////////////////////////////////////

extern void* _memcpy( void* _dst, 
                      const void* _src, 
                      unsigned int size );

extern void* _memset( void* dst, 
                      int s, 
                      unsigned int count );

///////////////////////////////////////////////////////////////////////////////////
//     Break function
///////////////////////////////////////////////////////////////////////////////////

extern void _break( char* str);

///////////////////////////////////////////////////////////////////////////////////
//     Suicide function
///////////////////////////////////////////////////////////////////////////////////

extern void _exit(void);

///////////////////////////////////////////////////////////////////////////////////
//     Eret function
///////////////////////////////////////////////////////////////////////////////////

extern void _eret(void);

///////////////////////////////////////////////////////////////////////////////////
//     CP0 and CP2 registers access functions
///////////////////////////////////////////////////////////////////////////////////

extern unsigned int _get_sched(void);
extern unsigned int _get_mmu_ptpr(void);
extern unsigned int _get_mmu_mode(void);
extern unsigned int _get_epc(void);
extern unsigned int _get_bvar(void);
extern unsigned int _get_cr(void);
extern unsigned int _get_sr(void);
extern unsigned int _get_procid(void);
extern unsigned int _get_proctime(void);
extern unsigned int _get_current_task_id(void);

extern void         _it_disable( unsigned int* save_sr_ptr );
extern void         _it_enable(void);
extern void         _it_restore( unsigned int* save_sr_ptr );

extern void         _set_mmu_ptpr(unsigned int value);
extern void         _set_mmu_mode(unsigned int value);
extern void         _set_sched(unsigned int value);
extern void         _set_sr(unsigned int value);

///////////////////////////////////////////////////////////////////////////////////
//     Physical addressing related functions
///////////////////////////////////////////////////////////////////////////////////

extern unsigned int _physical_read(  unsigned long long paddr );

extern void         _physical_write( unsigned long long paddr,
                                     unsigned int       value );

extern unsigned int _io_extended_read(  unsigned int* vaddr );

extern void         _io_extended_write( unsigned int* vaddr,
                                        unsigned int  value );

///////////////////////////////////////////////////////////////////////////////////
//     Locks access functions
///////////////////////////////////////////////////////////////////////////////////

extern void         _get_lock(unsigned int * lock);
extern void         _release_lock(unsigned int * lock);

///////////////////////////////////////////////////////////////////////////////////
//     TTY0 access functions
///////////////////////////////////////////////////////////////////////////////////

extern void         _puts( char*        string );

extern void         _putx( unsigned int val );

extern void         _putd( unsigned int val ); 

extern void         _putl( unsigned long long val );

extern void         _printf(char* format, ...); 

extern void         _getc( char*        byte );       

///////////////////////////////////////////////////////////////////////////////////
//     string compare functions
///////////////////////////////////////////////////////////////////////////////////

extern unsigned int _strncmp(const char*  s1, 
                             const char*  s2, 
                             unsigned int n);

extern char*        _strcpy( char*        dest,
                             char*        source );

///////////////////////////////////////////////////////////////////////////////////
//     cache invalidate function
///////////////////////////////////////////////////////////////////////////////////

extern void  _dcache_buf_invalidate( void * buffer, 
                                     unsigned int size );

///////////////////////////////////////////////////////////////////////////////////
//     Schedulers access functions
///////////////////////////////////////////////////////////////////////////////////

extern unsigned int _get_task_slot( unsigned int gpid,
                                    unsigned int ltid,
                                    unsigned int slot );

extern void         _set_task_slot( unsigned int gpid,
                                    unsigned int ltid,
                                    unsigned int slot,
                                    unsigned int value );

extern unsigned int _get_context_slot( unsigned int slot );

extern void         _set_context_slot( unsigned int slot,
                                       unsigned int value );

extern unsigned int _heap_info( unsigned int* vaddr,
                                unsigned int* size );

///////////////////////////////////////////////////////////////////////////////////
//     Mapping access functions
///////////////////////////////////////////////////////////////////////////////////

extern mapping_cluster_t *  _get_cluster_base(mapping_header_t* header);
extern mapping_pseg_t *     _get_pseg_base(mapping_header_t* header);
extern mapping_vspace_t *   _get_vspace_base(mapping_header_t* header);
extern mapping_vseg_t *     _get_vseg_base(mapping_header_t* header);
extern mapping_vobj_t *     _get_vobj_base(mapping_header_t* header);
extern mapping_task_t *     _get_task_base(mapping_header_t* header);
extern mapping_proc_t *     _get_proc_base(mapping_header_t* header);
extern mapping_irq_t *      _get_irq_base(mapping_header_t* header);
extern mapping_coproc_t *   _get_coproc_base(mapping_header_t* header);
extern mapping_cp_port_t *  _get_cp_port_base(mapping_header_t* header);
extern mapping_periph_t *   _get_periph_base(mapping_header_t* header);

#endif

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

