///////////////////////////////////////////////////////////////////////////////////
// File     : nic_driver.c
// Date     : 23/05/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The nic_driver.c and nic_driver.h files are part ot the GIET-VM nano-kernel.
// This driver supports the vci_multi_nic component.
// 
// It can exist only one network controller in the architecture, but this
// component supports several channels.
//
// It can be accessed directly by software with memcpy(),
// or it can be accessed through the vci_chbuf_dma component:
//  
// The '_nic_sync_write' and '_nic_sync_read' functions use a memcpy strategy to
// implement the transfer between a data buffer (user space) and the NIC
// buffer (kernel space). They are blocking until completion of the transfer.
//
// The _nic_cma_*_init() and _nic_cma_stop() functions use the VciChbufDma component 
// to transfer a flow of packets from the NIC RX hard chbuf (two containers) 
// to an user RX chbuf (two containers), and to transfer another flow of packets
// from an user TX chbuf (two containers) to the NIC TX chbuf (two containers).
// One NIC channel and two CMA channels must be allocated to the task 
// in the mapping_info data structure.
//
// The SEG_NIC_BASE address must be defined in the hard_config.h file.
//////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <nic_driver.h>
#include <utils.h>

#if !defined(GIET_USE_IOMMU) 
# error: You must define GIET_USE_IOMMU in the giet_config.h file
#endif

#if !defined(SEG_NIC_BASE)
# error: You must define SEG_NIC_BASE in the hard_config.h file 
#endif

#if !defined(NB_NIC_CHANNELS)
# error: You must define NB_NIC_CHANNELS in the hard_config.h file 
#endif

#if ( NB_NIC_CHANNELS > 8 )
# error: NB_NIC_CHANNELS cannot be larger than 8
#endif

#if !defined(NB_CMA_CHANNELS)
# error: You must define NB_CMA_CHANNELS in the hard_config.h file 
#endif

#if ( NB_CMA_CHANNELS > 8 )
# error: NB_CMA_CHANNELS cannot be larger than 8
#endif

#if !defined( USE_IOB )
# error: You must define USE_IOB in the hard_config.h file
#endif

#define in_unckdata __attribute__((section (".unckdata")))

///////////////////////////////////////////////////////////////////////////////
// This low_level function returns the value contained in register (index).
///////////////////////////////////////////////////////////////////////////////
unsigned int _nic_get_register( unsigned int channel,
                                unsigned int index )
{
    unsigned int* vaddr = (unsigned int*)SEG_NIC_BASE + 
                           NIC_CHANNEL_SPAN * channel + index;
    return _io_extended_read( vaddr );
}

///////////////////////////////////////////////////////////////////////////////
// This low-level function set a new value in register (index).
///////////////////////////////////////////////////////////////////////////////
void _nic_set_register( unsigned int channel,
                        unsigned int index,
                        unsigned int value ) 
{
    unsigned int* vaddr = (unsigned int*)SEG_NIC_BASE + 
                           NIC_CHANNEL_SPAN * channel + index;
    _io_extended_write( vaddr, value );
}

//////////////////////////////////////////////////////////////////////////////////
// Transfer data from an memory buffer to the NIC device using a memcpy.
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_sync_write( const void*    buffer,
                              unsigned int   length ) 
{
    _printf("[GIET ERROR] _nic_sync_write function not implemented / cycle %d\n",
            _get_proctime() );
    _exit();

    return 0;
}
//////////////////////////////////////////////////////////////////////////////////
// Transfer data from the NIC device to a memory buffer using a memcpy. 
// - buffer : base address of the memory buffer.
// - length : number of bytes to be transfered.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_sync_read( const void*    buffer, 
                             unsigned int   length ) 
{
    _printf("[GIET ERROR] _nic_sync_read function not implemented / cycle %d\n",
            _get_proctime() );
    _exit();

    return 0;
}
//////////////////////////////////////////////////////////////////////////////////
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_cma_start( )
{
    _printf("[GIET ERROR] _nic_cma_start() not implemented / cycle %d\n",
            _get_proctime() );
    _exit();

    return 0;
}
//////////////////////////////////////////////////////////////////////////////////
// Returns 0 if success, > 0 if error.
//////////////////////////////////////////////////////////////////////////////////
unsigned int _nic_cma_stop()
{
    _printf("[GIET ERROR] _nic_cma_stop() not implemented / cycle %d\n",
            _get_proctime() );
    _exit();

    return 0;
}

//////////////////////////////////////////////////////////////////////////////////
// This ISR handles IRQx from a NIC RX channeL
//////////////////////////////////////////////////////////////////////////////////
void _nic_rx_isr( unsigned int irq_type,
                  unsigned int irq_id,
                  unsigned int channel )
{
    _printf("[GIET ERROR] _nic_rx_isr() not implemented / cycle %d\n",
            _get_proctime() );
    _exit();
}

//////////////////////////////////////////////////////////////////////////////////
// This ISR handles IRQx from a NIC RX channeL
//////////////////////////////////////////////////////////////////////////////////
void _nic_tx_isr( unsigned int irq_type,
                  unsigned int irq_id,
                  unsigned int channel )
{
    _printf("[GIET ERROR] _nic_tx_isr() not implemented / cycle %d\n",
            _get_proctime() );
    _exit();
}

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

