#!/usr/bin/env python

from mapping import *

####################################################################################
#   file   : sort.py  (for the sort application)
#   date   : may 2014
#   author : Alain Greiner
####################################################################################
#  This file describes the mapping of the multi-threaded "sort"
#  application on a multi_clusters, multi-processors architecture.
#  This include both the mapping of virtual segments on the clusters, 
#  and the mapping of tasks on processors.
#  This mapping uses 5 platform parameters, (obtained from the "mapping" argument)
#  - x_size    : number of clusters in a row
#  - y_size    : number of clusters in a column
#  - x_width   : number of bits coding x coordinate
#  - y_width   : number of bits coding y coordinate
#  - procs_max : number of processors per cluster
####################################################################################

####################
def sort( mapping ):

    x_size    = mapping.x_size
    y_size    = mapping.y_size
    procs_max = mapping.procs_max
    x_width   = mapping.x_width
    y_width   = mapping.y_width

    ntasks    = x_size * y_size * procs_max

    # define vsegs base & size
    code_base  = 0x10000000
    code_size  = 0x00010000     # 64 Kbytes
    
    data_base  = 0x20000000
    data_size  = 0x00010000     # 64 Kbytes

    ptab_base  = 0x30000000
    ptab_size  = 0x00040000     # 256 Kbytes

    stack_base = 0x40000000 
    stack_size = 0x00010000     # 64 Kbytes

    heap_base  = 0x50000000 
    heap_size  = 0x00010000     # 64 Kbytes

    args_base  = 0x60000000 
    args_size  = 0x00000004     # 4 bytes

    # create Vspace
    vspace = mapping.addVspace( name = 'sort', startname = 'sort_data' )
    
    # non replicated vsegs in cluster[0,0]
    mapping.addVseg( vspace, 'sort_code', code_base , code_size, 'CXWU', vtype = 'ELF', 
                     x = 0, y = 0, pseg = 'RAM', binpath = 'build/sort/sort.elf' )

    mapping.addVseg( vspace, 'sort_data', data_base , data_size, 'C_WU', vtype = 'ELF',
                     x = 0, y = 0, pseg = 'RAM', binpath = 'build/sort/sort.elf' )

    mapping.addVseg( vspace, 'sort_ptab', ptab_base , ptab_size, 'C_WU', vtype = 'PTAB',
                     x = 0, y = 0, pseg = 'RAM', align = 13 )

    mapping.addVseg( vspace, 'sort_args', args_base , args_size, 'C_WU', vtype = 'CONST',
                     x = 0, y = 0, pseg = 'RAM', init = ntasks )

    # distributed vsegs: one stack per processor/task, one heap per cluster
    for x_rep in xrange (x_size):
        for y_rep in xrange (y_size):
            cluster_offset = ((x_rep << y_width) + y_rep) << 20  # 1 Mbytes per cluster
            mapping.addVseg( vspace, 'sort_heap_%d_%d' % (x_rep, y_rep), 
                             heap_base + cluster_offset, heap_size, 'C_WU', 
                             vtype = 'BUFFER', x = x_rep, y = y_rep, pseg = 'RAM' )
            
            for p in xrange( procs_max ):
                proc_offset = cluster_offset + (p << 18)         # 256 Kbytes per proc
                mapping.addVseg( vspace, 'sort_stack_%d_%d_%d' % (x_rep, y_rep, p),
                                 stack_base + proc_offset, stack_size, 'C_WU',
                                 vtype = 'BUFFER', x = x_rep, y = y_rep, pseg = 'RAM' )
            
    # distributed tasks / one task per processor
    for x in xrange (x_size):
        for y in xrange (y_size):
            for p in xrange( procs_max ):

                trdid = (((x * y_size) + y) * procs_max) + p
                mapping.addTask( vspace, 'sort_%d_%d_%d' % (x,y,p), trdid, x, y, p,
                                 'sort_stack_%d_%d_%d' % (x,y,p),
                                 'sort_heap_%d_%d' % (x,y), 0 )

    # extend mapping name
    mapping.name += '_sort'

    return vspace  # useful for test

################################ test ######################################################

if __name__ == '__main__':
    
    vspace = sort( Mapping( 'test', 2, 2, 4 ) )
    print vspace.xml()


# Local Variables:
# tab-width: 4;
# c-basic-offset: 4;
# c-file-offsets:((innamespace . 0)(inline-open . 0));
# indent-tabs-mode: nil;
# End:
#
# vim: filetype=python:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

