///////////////////////////////////////////////////////////////////////////////////
// File     : mmc_driver.c
// Date     : 23/05/2013
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// The mmc_driver.c and mmc_driver.h files are part ot the GIET-VM nano-kernel.
// This driver supports the vci_mem_cache component used in the TSAR architecture.
//
// This component is replicated in all clusters, and can be accessed through 
// a configuration interface as a set of uncached, memory mapped registers.
//
// The (virtual) base address of the associated segment is:
//       SEG_MMC_BASE + cluster_id * PERI_CLUSTER_INCREMENT
//
// SEG_MMC_BASE and PERI_CLUSTER_INCREMENT must be defined in hard_config.h.
////////////////////////////////////////////////////////////////////////////////

#include <giet_config.h>
#include <mmc_driver.h>
#include <tty_driver.h>
#include <utils.h>
#include <io.h>

#if !defined(X_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(Y_SIZE) 
# error: You must define X_SIZE in the hard_config.h file
#endif

#if !defined(X_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(Y_WIDTH) 
# error: You must define X_WIDTH in the hard_config.h file
#endif

#if !defined(SEG_MMC_BASE) 
# error: You must define SEG_MMC_BASE in the hard_config.h file
#endif

#if !defined(PERI_CLUSTER_INCREMENT) 
# error: You must define PERI_CLUSTER_INCREMENT in the hard_config.h file
#endif

///////////////////////////////////////////////////////////////////////////////
// This low level function returns the value contained in register "index"
// in the MMC component contained in cluster "cluster_xy"
///////////////////////////////////////////////////////////////////////////////
static
unsigned int _mmc_get_register( unsigned int cluster_xy, // cluster index
                                unsigned int func,       // function index
                                unsigned int index )     // register index
{
    unsigned int vaddr =
        SEG_MMC_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) +
        (MMC_REG(func, index) << 2);

    return ioread32( (void*)vaddr );
}

///////////////////////////////////////////////////////////////////////////////
// This low level function sets a new value in register "index"
// in the MMC component contained in cluster "cluster_xy"
///////////////////////////////////////////////////////////////////////////////
static
void _mmc_set_register( unsigned int cluster_xy,       // cluster index
                        unsigned int func,             // func index
                        unsigned int index,            // register index
                        unsigned int value )           // value to be written
{
    unsigned int vaddr =
        SEG_MMC_BASE + 
        (cluster_xy * PERI_CLUSTER_INCREMENT) +
        (MMC_REG(func, index) << 2);
        
    iowrite32( (void*)vaddr, value );
}

///////////////////////////////////////////////////////////////////////////////////
// This function invalidates all cache lines covering a memory buffer defined
// by the physical base address, and the length.
// The buffer address MSB are used to compute the cluster index.
///////////////////////////////////////////////////////////////////////////////////
void _mmc_inval( paddr_t      buf_paddr,
                 unsigned int buf_length )
{
    // compute cluster coordinates
    unsigned int cluster_xy = (unsigned int)(buf_paddr>>(40-X_WIDTH-Y_WIDTH));
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    // parameters checking 
    if ( (x >= X_SIZE) || (y >= Y_SIZE) )
    {
        _printf("\n[GIET ERROR] in _memc_inval() : illegal cluster_xy for paddr %l\n",
                 buf_paddr );
        _exit();
    }

    // get the hard lock protecting exclusive access to MEMC
    while ( _mmc_get_register(cluster_xy, 0, MEMC_LOCK) );

    // write inval arguments
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_LO   , (unsigned int)buf_paddr);
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_HI   , (unsigned int)(buf_paddr>>32));
    _mmc_set_register(cluster_xy, 0, MEMC_BUF_LENGTH, buf_length);
    _mmc_set_register(cluster_xy, 0, MEMC_CMD_TYPE  , MEMC_CMD_INVAL);

    // release the lock 
    _mmc_set_register(cluster_xy, 0, MEMC_LOCK, 0);
}
///////////////////////////////////////////////////////////////////////////////////
// This function copies to external RAM all cache lines covering a memory buffer 
// defined by the physical base address, and the length, if they are dirty.
// The buffer address MSB are used to compute the cluster index.
///////////////////////////////////////////////////////////////////////////////////
void _mmc_sync( paddr_t      buf_paddr,
                unsigned int buf_length )
{
    // compute cluster coordinates
    unsigned int cluster_xy = (unsigned int)(buf_paddr>>(40-X_WIDTH-Y_WIDTH));
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    // parameters checking 
    if ( (x >= X_SIZE) || (y >= Y_SIZE) )
    {
        _printf( "\n[GIET ERROR] in _memc_sync() : illegal cluster_xy for paddr %l\n",
                 buf_paddr );
        _exit();
    }

    // get the hard lock protecting exclusive access to MEMC
    while ( _mmc_get_register(cluster_xy, 0, MEMC_LOCK) );

    // write inval arguments
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_LO   , (unsigned int)buf_paddr);
    _mmc_set_register(cluster_xy, 0, MEMC_ADDR_HI   , (unsigned int)(buf_paddr>>32));
    _mmc_set_register(cluster_xy, 0, MEMC_BUF_LENGTH, buf_length);
    _mmc_set_register(cluster_xy, 0, MEMC_CMD_TYPE  , MEMC_CMD_SYNC);

    // release the lock 
    _mmc_set_register(cluster_xy, 0, MEMC_LOCK, 0);
}

//////////////////////////////////////////////////////////////////////////////////
// This ISR access the vci_mem_cache component to get the faulty physical
// address and the associated SRCID. It must also acknowledge the IRQ.
//
// TODO implement...
//////////////////////////////////////////////////////////////////////////////////
void _mmc_isr( unsigned int irq_type,  // should be HWI 
               unsigned int irq_id,    // index returned by ICU
               unsigned int channel )  // unused
{
    unsigned int procid     = _get_procid();
    unsigned int cluster_xy = procid / NB_PROCS_MAX;
    unsigned int lpid       = procid % NB_PROCS_MAX;
    unsigned int x          = cluster_xy >> Y_WIDTH;
    unsigned int y          = cluster_xy & ((1<<Y_WIDTH)-1);

    _printf("[GIET ERROR] MMC IRQ received by processor[%d,%d,%d]"
            " but _mmc_isr() not implemented...\n", x, y, lpid );
}



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

