//////////////////////////////////////////////////////////////////////////////////
// File     : barrier.c      
// Date     : 01/04/2012
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////
// These barrier.c and barrier.h files are part of the GIET nano-kernel.
// This user-level library provides a synchronisation service between several
// tasks sharing the same address space in a parallel multi-tasks application.
// Neither the barrier_init(), nor the barrier_wait() function require a syscall.
// The barrier itself must have been allocated in a shared data segment.
///////////////////////////////////////////////////////////////////////////////////

#include "barrier.h"

///////////////////////////////////////////////////////////////////////////////////
// This function initializes the barrier: this should be done by a single task.
///////////////////////////////////////////////////////////////////////////////////
void barrier_init( giet_barrier_t* barrier, 
                   unsigned int    value ) 
{
    barrier->init  = (volatile unsigned int)value;
    barrier->count = (volatile unsigned int)value;
    asm volatile ("sync" ::: "memory");
}


///////////////////////////////////////////////////////////////////////////////////
// This blocking function uses LL/SC to decrement the barrier's counter.
// Then, it uses a busy_waiting mechanism if it is not the last.
///////////////////////////////////////////////////////////////////////////////////
void barrier_wait( giet_barrier_t* barrier ) 
{
    volatile unsigned int * pcount = (unsigned int *) &barrier->count;
    volatile unsigned int maxcount = barrier->init;
    volatile unsigned int count;

    // parallel decrement barrier counter using atomic instructions LL/SC
    // - input : pointer on the barrier counter
    // - output : counter value
    asm volatile (
            "_barrier_decrement:                \n"
            "ll     %0,   0(%1)                 \n"
            "addi   $3,   %0,     -1            \n"
            "sc     $3,   0(%1)                 \n"
            "beqz   $3,   _barrier_decrement    \n"
            : "+r"(count)
            : "r"(pcount)
            : "$3", "memory");

    // the last task re-initializes the barrier counter to the max value,
    // waking up all other waiting tasks

    if (count == 1) 
    {
        // last task
        *pcount = maxcount;
    }
    else {
        // other tasks busy-wait
        while (*pcount != maxcount);
    }

    asm volatile ("sync" ::: "memory");
}

// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

