/*
 * file   : boot.S
 * date   : 01/17/2014
 * author : Cesar Fuguet & Alain Greiner
 *
 * This file contains the boot_entry() function that is the entry
 * point for the GIET_VM bootloader.
 * It supports a generic multi-clusters / multi-processors architecture,
 * containing at most 1024 processors.
 * - The number of clusters is defined by the (X_SIZE,Y_SIZE) parameters 
 *   in the hard_config.h file (up to 256 clusters).
 * - The number of processors per cluster is defined by the NB_PROCS_MAX 
 *   parameter in the hard_config.h file (up to 4 processors per cluster). 
 *
 * This assembly code is executed by all processors.
 * It initializes the stack pointer depending on the proc_id,
 * and using the SEG_BOOT_STACK_BASE and SEG_BOOT_STACK_SIZE 
 * parameters defined in the hard_config.h file, and jumps
 * to the boot_init() fuction defined in the boot.c file.
 *
 * - Processor 0 uses a larger stack:         64 Kbytes.
 * - Other processors use a smaller stack:    256 bytes.
 *     => the SEG_BOOT_STACK_SIZE cannot be smaller 320 Kytes.
 *         (64K + 1024 * 256 = 320 Kbytes = 0x50000)
 */

    #include "mips32_registers.h"
    #include "hard_config.h"

    .section .text,"ax",@progbits

    .globl  boot_entry 
    .ent    boot_entry

    .align  2
    .set noreorder

boot_entry:


    /* All processors compute proc_id = (cluster_id * NBPROCS) + lpid             */
    /* where cluster_id = (x * Y_SIZE) + y                                        */
    /* The (x,y,lpid) values are obtained from the processor CP0 register,        */
    /* where  proc_xyl == (((x<<4) + y) * NB_PROCS_MAX) + lpid                    */ 
    /* (proc_id is a "continuous" index, while proc_xyl is a "fixed format" index */

    mfc0   k0,      CP0_EBASE
    andi   k0,      k0,     0x3FF   /* k0 <= proc_xyl                             */
    la     t7,      NB_PROCS_MAX    /* t7 <= NBPROCS                              */
    divu   k0,      t7
    mfhi   t1                       /* t1 <= lpid                                 */
    mflo   t2                       /* t2 <= cluster_xy                           */
    srl    t3,      t2,     4       /* t3 <= x coordinate                         */
    andi   t4,      t2,     0xF     /* t4 <= y coordinate                         */
    la     t6,      Y_SIZE          /* t6 <= Y_SIZE                               */
    multu  t3,      t6
    mflo   t5
    addu   t5,      t5,     t4      /* t5 <= cluster_id                           */
    multu  t5,      t7
    mflo   t0
    addu   t0,      t0,     t1      /* t0 <= proc_id                              */
    

    /* All processors initializes stack pointer, depending on proc_id             */

   la      k0,      SEG_BOOT_STACK_BASE
   li      k1,      0x10000         /* k1 <= P0 stack size == 64 Kbytes           */
   addu    sp,      k0,     k1      /* P0 stack from base to (base + 64K)         */

   li      k1,      0x100           /* k1 <= Pi stack size == 256 bytes           */
   multu   k1,      t0              
   mflo    k0                       /* k0 <= 256 * proc_id                        */
   addu    sp,      sp,     k1
   addu    sp,      sp,     k0      /* Pi stacks from base + 64K + proc_id*256    */


    /* All processors jump to the boot_init function                              */

    la	    k0,     boot_init
    jr      k0
    nop

    .end boot_entry

    .set reorder

/*
 * vim: tabstop=4 : shiftwidth=4 : expandtab
 */
