//////////////////////////////////////////////////////////////////////////////////
// File     : remote_malloc.c         
// Date     : 01/08/2014
// Author   : alain greiner
// Copyright (c) UPMC-LIP6
///////////////////////////////////////////////////////////////////////////////////

#include "giet_config.h"
#include "hard_config.h"
#include "remote_malloc.h"
#include "stdio.h"
#include "spin_lock.h"

///////////////////////////////////////////////////////////////////////////////////
// Global variables describing allocators: one heap per cluster.
// For each cluster, cur[x][y] is the virtual adrress of first free byte 
// in heap, max[x][y] is the virtual address of first forbidden byte on top
// of heap, and lock[x][y] protect exclusive access to heap[x][y].
///////////////////////////////////////////////////////////////////////////////////

unsigned int remote_malloc_cur[X_SIZE][Y_SIZE] = {[0 ... Y_SIZE-1][0 ... X_SIZE-1] = 0};

unsigned int remote_malloc_max[X_SIZE][Y_SIZE] = {[0 ... Y_SIZE-1][0 ... X_SIZE-1] = 0};

giet_lock_t  remote_malloc_lock[X_SIZE][Y_SIZE] __attribute__((aligned(512)));

///////////////////////////////////////////////////////////////////////////////////
// This function returne the virtual base address of the allocated block.
// - length is the block length (number of bytes)
// - align is the alignment constrain (vbase multiple of Ox1 << align).
// - If (x < X_SIZE) and (y < Y_SIZE), it uses the heap defined in cluster(x,y).
// - It uses the heap in cluster running the calling task if x oy y too large.
// The calling task exit with an error message if the request cannot be satisfied.
///////////////////////////////////////////////////////////////////////////////////
void* remote_malloc( unsigned int length,
                     unsigned int align,
                     unsigned int x,
                     unsigned int y )
{
    // checking alignment constraint
    if ( align > 31  ) 
    {
        giet_exit("in remote_malloc(), align constraint > 31\n");
    }

    // checking requested length
    if ( length == 0 ) 
    {
        giet_exit("in remote_malloc(), requested length = 0\n");
    }

#if GIET_DEBUG_MALLOC
unsigned int procid  = giet_procid();
unsigned int cluster = procid / NB_PROCS_MAX;
unsigned int proc_x  = cluster >> Y_WIDTH;
unsigned int proc_y  = cluster & ((1<<Y_WIDTH)-1);
unsigned int lpid    = procid % NB_PROCS_MAX;
giet_shr_printf("\n[DEBUG MALLOC] Processor[%d,%d,%d] enters remote_malloc()"
                " : length = %x / align = %x for heap(%d,%d)\n",
                proc_x, proc_y, lpid, length, (1<<align), x, y );
#endif


    unsigned int heap_x;        // heap x coordinate
    unsigned int heap_y;        // heap y coordibnate
    unsigned int heap_base;     // heap base address
    unsigned int heap_length;   // heap length
    unsigned int vbase;         // virtual address to be returned

    unsigned int quantum = 1 << align;

    // compute cluster coordinates
    if ( (x < X_SIZE) && (y < Y_SIZE) )
    {
        heap_x = x;
        heap_y = y;
    }
    else
    {
        unsigned int pid = giet_procid();
        heap_x = (pid / NB_PROCS_MAX) >> Y_WIDTH;
        heap_y = (pid / NB_PROCS_MAX) & ((1<<Y_WIDTH)-1);
    }

    // get heap vbase and length if allocator[x][y] not initialised yet
    if ( remote_malloc_max[heap_x][heap_y] == 0 )
    {
        giet_heap_info( &heap_base,
                        &heap_length,
                        heap_x, 
                        heap_y );

        if ( heap_length == 0 ) 
        {
            giet_exit("in remote_malloc(): heap length = 0\n");
        }

        remote_malloc_cur[heap_x][heap_y] = heap_base;
        remote_malloc_max[heap_x][heap_y] = heap_base + heap_length;

        // initialise lock
        lock_release( &remote_malloc_lock[heap_x][heap_y] );

#if GIET_DEBUG_MALLOC
giet_shr_printf("\n[DEBUG MALLOC] Processor[%d,%d,%d] initializes heap(%d,%d)\n"
                " - heap_base        = %x\n"
                " - heap_size        = %x\n",
                proc_x, proc_y, lpid, heap_x, heap_y,
                heap_base, heap_length );
#endif

    }

    // take the lock
    lock_acquire( &remote_malloc_lock[heap_x][heap_y] );

#if GIET_DEBUG_MALLOC
giet_shr_printf("\n[DEBUG MALLOC] Processor[%d,%d,%d] takes lock for heap(%d,%d)\n",
                proc_x, proc_y, lpid, heap_x, heap_y );
#endif

    // compute vbase
    vbase = remote_malloc_cur[heap_x][heap_y];
    if ( vbase % quantum )  vbase = ((vbase / quantum) + 1 ) * quantum; 

#if GIET_DEBUG_MALLOC
giet_shr_printf("\n[DEBUG MALLOC] Processor[%d,%d,%d] allocate vaddr = %x in heap(%d,%d)\n",
                proc_x, proc_y, lpid, vbase, heap_x, heap_y );
#endif

    // check overflow
    if ( (vbase + length) > remote_malloc_max[heap_x][heap_y] )
    {
        giet_exit("in remote_malloc(), heap overflow\n");
    }

    // update heap pointer
    remote_malloc_cur[heap_x][heap_y] = vbase + length;

    // release the lock
    lock_release( &remote_malloc_lock[heap_x][heap_y] );

#if GIET_DEBUG_MALLOC
giet_shr_printf("\n[DEBUG MALLOC] Processor[%d,%d,%d] releases lock for heap(%d,%d)\n",
                proc_x, proc_y, lpid, heap_x, heap_y );
#endif

    return (void*)vbase;
}  // end remote_malloc()



// Local Variables:
// tab-width: 4
// c-basic-offset: 4
// c-file-offsets:((innamespace . 0)(inline-open . 0))
// indent-tabs-mode: nil
// End:
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4



